﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_CommandProcessingTimeEstimatorVersion2.h"

namespace nn { namespace audio { namespace server {

namespace  {

struct EstimationParameters
{
    // Delay
    float delay1chDisabled;
    float delay2chDisabled;
    float delay4chDisabled;
    float delay6chDisabled;
    float delay1chEnabled;
    float delay2chEnabled;
    float delay4chEnabled;
    float delay6chEnabled;

    // Reverb
    float reverb1chDisabled;
    float reverb2chDisabled;
    float reverb4chDisabled;
    float reverb6chDisabled;
    float reverb1chEnabled;
    float reverb2chEnabled;
    float reverb4chEnabled;
    float reverb6chEnabled;

    // I3dl2Reverb
    float i3dl2Reverb1chDisabled;
    float i3dl2Reverb2chDisabled;
    float i3dl2Reverb4chDisabled;
    float i3dl2Reverb6chDisabled;
    float i3dl2Reverb1chEnabled;
    float i3dl2Reverb2chEnabled;
    float i3dl2Reverb4chEnabled;
    float i3dl2Reverb6chEnabled;

    float auxEnabled;
    float auxDisabled;
    float depopForMixBuffers;
    float performance;
    float volume;
    float biquadFilter;
    float copyMixBuffer;
    float depopPrepare;
    float volumeRamp;

    // mixRampGroupedA * m_SampleCount * Count
    float mixRampGroupedA;

    float mixRamp;
    float mix;
    float downMix6chTo2ch;
    float upsample;
    float deviceSink2ch;
    float deviceSink6ch;

    // A * (pitch * voiceSampleRate / RendererSampleRate) + B
    float pcmInt16DataSourceA;
    float pcmInt16DataSourceB;

    // A * (pitch * voiceSampleRate / RendererSampleRate) + B
    float adpcmDataSourceA;
    float adpcmDataSourceB;

    // A * MixBufferCount + B
    float clearMixBufferA;
    float clearMixBufferB;

    // A * ChannelCount + B
    float circularBufferSinkA;
    float circularBufferSinkB;
};


const float SafetyFactor = 1.05f;

const EstimationParameters EstimationParametersForSampleRate32000 = {
    // Delay
    SafetyFactor * 550.98f,
    SafetyFactor * 631.49f,
    SafetyFactor * 670.46f,
    SafetyFactor * 723.84f,
    SafetyFactor * 39652.91f,
    SafetyFactor * 93201.16f,
    SafetyFactor * 183348.12f,
    SafetyFactor * 287386.67f,

    // Reverb
    SafetyFactor * 468.58f,
    SafetyFactor * 528.06f,
    SafetyFactor * 567.49f,
    SafetyFactor * 625.35f,
    SafetyFactor * 92564.03f,
    SafetyFactor * 98360.53f,
    SafetyFactor * 104360.99f,
    SafetyFactor * 109586.13f,

    // I3dl2Reverb
    SafetyFactor * 684.48f,
    SafetyFactor * 715.52f,
    SafetyFactor * 759.49f,
    SafetyFactor * 826.12f,
    SafetyFactor * 132225.24f,
    SafetyFactor * 128979.21f,
    SafetyFactor * 189697.f,
    SafetyFactor * 235567.54f,

    // Aux
    SafetyFactor * 465.87f,
    SafetyFactor * 6836.13f,

    // DepopForMixBuffers
    SafetyFactor * 726.63f,

    // Performance
    SafetyFactor * 466.05f,

    // Volume
    SafetyFactor * 1219.33f,

    // BiquadFilter
    SafetyFactor * 4584.f,

    // CopyMixBuffer
    SafetyFactor * 796.5f,

    // DepopPrepare
    SafetyFactor * 292.02f,

    // VolumeRamp
    SafetyFactor * 1337.09f,

    // mixRampGroupedA
    SafetyFactor * 6.9f,

    // mixRamp
    SafetyFactor * 1770.48f,

    // mix
    SafetyFactor * 1278.31f,

    // DownMix6chTo2ch
    SafetyFactor * 9532.32f,

    // Upsample
    SafetyFactor * 278097.14f,

    // DeviceSink
    SafetyFactor * 8820.52f,
    SafetyFactor * 8677.95f,

    SafetyFactor * 713.59f, // PcmInt16A
    SafetyFactor * 5846.61f, // PcmInt16B
    SafetyFactor * 2024.37f, // AdpcmA
    SafetyFactor * 8609.02f, // AdpcmB
    SafetyFactor * 248.f, // ClearMixBufferA
    SafetyFactor * 133.f, // ClearMixBufferB
    SafetyFactor * 812.98f, // CircularBufferSinkA
    SafetyFactor * 1223.35f, // CircularBufferSinkB
};

const EstimationParameters EstimationParametersForSampleRate48000 = {
    // Delay
    SafetyFactor * 496.46f,
    SafetyFactor * 557.52f,
    SafetyFactor * 599.89f,
    SafetyFactor * 679.59f,
    SafetyFactor * 8352.71f,
    SafetyFactor * 24515.41f,
    SafetyFactor * 45286.83f,
    SafetyFactor * 77742.12f,

    // Reverb
    SafetyFactor * 472.18f,
    SafetyFactor * 502.06f,
    SafetyFactor * 570.24f,
    SafetyFactor * 634.31f,
    SafetyFactor * 129965.38f,
    SafetyFactor * 138808.62f,
    SafetyFactor * 147425.65f,
    SafetyFactor * 154255.63f,

    // I3dl2Reverb
    SafetyFactor * 508.8f,
    SafetyFactor * 543.69f,
    SafetyFactor * 629.46f,
    SafetyFactor * 661.52f,
    SafetyFactor * 190431.18f,
    SafetyFactor * 185904.3f,
    SafetyFactor * 276738.94f,
    SafetyFactor * 346185.27f,

    // Aux
    SafetyFactor * 462.44f,
    SafetyFactor * 9047.45f,

    // DepopForMixBuffers
    SafetyFactor * 692.34f,

    // Performance
    SafetyFactor * 467.79f,

    // Volume
    SafetyFactor * 1655.04f,

    // BiquadFilter
    SafetyFactor * 6586.06f,

    // CopyMixBuffer
    SafetyFactor * 953.21f,

    // DepopPrepare
    SafetyFactor * 279.26f,

    // VolumeRamp
    SafetyFactor * 1794.58f,

    // mixRampGroupedA
    SafetyFactor * 6.9f,

    // mixRamp
    SafetyFactor * 2177.26f,

    // mix
    SafetyFactor * 1745.88f,

    // DownMix6chTo2ch
    SafetyFactor * 13883.11f,

    // Upsample
    SafetyFactor * 0.f,

    // DeviceSink
    SafetyFactor * 8891.48f,
    SafetyFactor * 9111.17f,

    SafetyFactor * 1138.53f, // PcmInt16A
    SafetyFactor * 7425.76f, // PcmInt16B
    SafetyFactor * 3394.37f, // AdpcmA
    SafetyFactor * 5929.02f, // AdpcmB
    SafetyFactor * 637.f, // ClearMixBufferA
    SafetyFactor * 184.f, // ClearMixBufferB
    SafetyFactor * 1643.83f, // CircularBufferSinkA
    SafetyFactor * 1304.46f, // CircularBufferSinkB
};

const EstimationParameters& GetParameters(int sampleCount)
{
    switch(sampleCount)
    {
    case 160:
        return EstimationParametersForSampleRate32000;
    case 240:
        return EstimationParametersForSampleRate48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// TODO: AudioRenderSystem から渡す
const float AudioFrameCountPerSecond = 200.f;

} // anonymous namespace

void CommandProcessingTimeEstimatorVersion2::Initialize(int sampleCount, int mixBufferCount) NN_NOEXCEPT
{
    m_SampleCount = sampleCount;
    m_MixBufferCount = mixBufferCount;
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const PcmInt16DataSourceCommand& command) NN_NOEXCEPT
{
    const auto& parameters = GetParameters(m_SampleCount);
    const auto sampleCount = command.sampleRate / AudioFrameCountPerSecond;
    const auto pitch = static_cast<float>(command.pitch) / (1 <<  15);

    return static_cast<uint32_t>(parameters.pcmInt16DataSourceA * (sampleCount / m_SampleCount * pitch) + parameters.pcmInt16DataSourceB);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const AdpcmDataSourceCommand& command) NN_NOEXCEPT
{
    const auto& parameters = GetParameters(m_SampleCount);
    const auto sampleCount = command.sampleRate / AudioFrameCountPerSecond;
    const auto pitch = static_cast<float>(command.pitch) / (1 << 15);
    return static_cast<uint32_t>(parameters.adpcmDataSourceA * (sampleCount / m_SampleCount * pitch) + parameters.adpcmDataSourceB);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const OpusDataSourceCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return 0u;
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const VolumeCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).volume);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const VolumeRampCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).volumeRamp);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const BiquadFilterCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).biquadFilter);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const MixCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).mix);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const MixRampCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).mixRamp);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const MixRampGroupedCommand& command) NN_NOEXCEPT
{
    auto count = 0;
    for (auto i = 0; i < command.mixBufferCount; ++i)
    {
        if (command.volume0[i] != 0 || command.volume1[i] != 0)
        {
            count += 1;
        }
    }

    return static_cast<uint32_t>(GetParameters(m_SampleCount).mixRampGroupedA * m_SampleCount * count);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const DepopPrepareCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).depopPrepare);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const DepopForMixBuffersCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).depopForMixBuffers);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const DelayCommand& command) NN_NOEXCEPT
{
    const auto channelCount = command.delayConfig._numChannels;
    const auto& parameters = GetParameters(m_SampleCount);

    if(command.enabled)
    {
        switch(channelCount)
        {
        case 1:
            return static_cast<uint32_t>(parameters.delay1chEnabled);
        case 2:
            return static_cast<uint32_t>(parameters.delay2chEnabled);
        case 4:
            return static_cast<uint32_t>(parameters.delay4chEnabled);
        case 6:
            return static_cast<uint32_t>(parameters.delay6chEnabled);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        switch(channelCount)
        {
        case 1:
            return static_cast<uint32_t>(parameters.delay1chDisabled);
        case 2:
            return static_cast<uint32_t>(parameters.delay2chDisabled);
        case 4:
            return static_cast<uint32_t>(parameters.delay4chDisabled);
        case 6:
            return static_cast<uint32_t>(parameters.delay6chDisabled);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const UpsampleCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).upsample);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const AuxCommand& command) NN_NOEXCEPT
{
    if(command.enabled)
    {
        return static_cast<uint32_t>(GetParameters(m_SampleCount).auxEnabled);
    }
    else
    {
        return static_cast<uint32_t>(GetParameters(m_SampleCount).auxDisabled);
    }
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const DeviceSinkCommand& command) NN_NOEXCEPT
{
    switch(command.inputCount)
    {
    case 2:
        return static_cast<uint32_t>(GetParameters(m_SampleCount).deviceSink2ch);
    case 6:
        return static_cast<uint32_t>(GetParameters(m_SampleCount).deviceSink6ch);
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const CircularBufferSinkCommand& command) NN_NOEXCEPT
{
    const auto& parameters = GetParameters(m_SampleCount);
    return static_cast<uint32_t>(parameters.circularBufferSinkA * command.inputCount + parameters.circularBufferSinkB);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const ReverbCommand& command) NN_NOEXCEPT
{
    const auto channelCount = command.reverbConfig._numChannels;
    const auto& parameters = GetParameters(m_SampleCount);
    if(command.enabled)
    {
        switch(channelCount)
        {
        case 1:
            return static_cast<uint32_t>(parameters.reverb1chEnabled);
        case 2:
            return static_cast<uint32_t>(parameters.reverb2chEnabled);
        case 4:
            return static_cast<uint32_t>(parameters.reverb4chEnabled);
        case 6:
            return static_cast<uint32_t>(parameters.reverb6chEnabled);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        switch(channelCount)
        {
        case 1:
            return static_cast<uint32_t>(parameters.reverb1chDisabled);
        case 2:
            return static_cast<uint32_t>(parameters.reverb2chDisabled);
        case 4:
            return static_cast<uint32_t>(parameters.reverb4chDisabled);
        case 6:
            return static_cast<uint32_t>(parameters.reverb6chDisabled);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const I3dl2ReverbCommand& command) NN_NOEXCEPT
{
    const auto channelCount = command.i3dl2ReverbConfig._numChannels;
    const auto& parameters = GetParameters(m_SampleCount);

    if(command.enabled)
    {
        switch(channelCount)
        {
        case 1:
            return static_cast<uint32_t>(parameters.i3dl2Reverb1chEnabled);
        case 2:
            return static_cast<uint32_t>(parameters.i3dl2Reverb2chEnabled);
        case 4:
            return static_cast<uint32_t>(parameters.i3dl2Reverb4chEnabled);
        case 6:
            return static_cast<uint32_t>(parameters.i3dl2Reverb6chEnabled);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        switch(channelCount)
        {
        case 1:
            return static_cast<uint32_t>(parameters.i3dl2Reverb1chDisabled);
        case 2:
            return static_cast<uint32_t>(parameters.i3dl2Reverb2chDisabled);
        case 4:
            return static_cast<uint32_t>(parameters.i3dl2Reverb4chDisabled);
        case 6:
            return static_cast<uint32_t>(parameters.i3dl2Reverb6chDisabled);
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const PerformanceCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).performance);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const ClearMixBufferCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    const auto& parameters = GetParameters(m_SampleCount);
    return static_cast<uint32_t>(parameters.clearMixBufferA * m_MixBufferCount + parameters.clearMixBufferB);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const DownMix6chTo2chCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).downMix6chTo2ch);
}

uint32_t CommandProcessingTimeEstimatorVersion2::Estimate(const CopyMixBufferCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    return static_cast<uint32_t>(GetParameters(m_SampleCount).copyMixBuffer);
}

}}}  // namespace nn::audio::server
