﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_CommandProcessingTimeEstimator.h"

namespace nn { namespace audio { namespace server {

namespace  {
const float SafetyFactor = 1.2f;

const float PcmInt16DataSourceCommandCoefficient = 0.25f;
const float AdpcmDataSourceCommandCoefficient = 0.46f;
const float VolumeCommandCoefficient = 8.8f;
const float VolumeRampCommandCoefficient = 9.8f;
const float BiquadFilterCommandCoefficient = 58.f;
const float MixCommandCoefficient = 10.0f;
const float MixRampCommandCoefficient = 14.4f;
const float DepopPrepareCommandCoefficient = 1080.0f;
const float DepopForMixBuffersCommandCoefficient = 8.9f;
const float DelayCommandTimeCommandCoefficient = 202.5f;
const int UpsampleCommandTime = 298263;
const int AuxCommandEnableTime= 13297; // 48 k
const int AuxCommandDisableTime = 3138; // 48 k
const int DeviceSinkCommandTime = 8369; // 48 k
const float CircularBufferSinkCommandTime = 46;
const int ReverbCommandCoefficient = 750;
const int I3dl2ReverbCommandCoefficient = 530;
const int PerformanceCommandTime = 1212;
const float ClearMixBufferCommandCoefficient = 0.83f;
const int DownMix6chTo2chCommandTime = 13424;
} // anonymous namespace

void CommandProcessingTimeEstimatorVersion1::Initialize(int sampleCount, int mixBufferCount) NN_NOEXCEPT
{
    m_SampleCount = sampleCount;
    m_MixBufferCount = mixBufferCount;
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const PcmInt16DataSourceCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(PcmInt16DataSourceCommandCoefficient * command.pitch * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const AdpcmDataSourceCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(AdpcmDataSourceCommandCoefficient * command.pitch * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const OpusDataSourceCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return 0u;
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const VolumeCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(VolumeCommandCoefficient * m_SampleCount * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const VolumeRampCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(VolumeRampCommandCoefficient * m_SampleCount * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const BiquadFilterCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(BiquadFilterCommandCoefficient * m_SampleCount * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const MixCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(MixCommandCoefficient * m_SampleCount * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const MixRampCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(MixRampCommandCoefficient * m_SampleCount * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const MixRampGroupedCommand& command) NN_NOEXCEPT
{
    auto count = 0;
    for (auto i = 0; i < command.mixBufferCount; ++i)
    {
        if (command.volume0[i] != 0 || command.volume1[i] != 0)
        {
            count += 1;
        }
    }
    return static_cast<uint32_t>(MixRampCommandCoefficient * m_SampleCount * SafetyFactor * count);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const DepopPrepareCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(DepopPrepareCommandCoefficient);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const DepopForMixBuffersCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(DepopForMixBuffersCommandCoefficient * m_SampleCount * command.mixBufferCount);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const DelayCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    const int channelCount = command.delayConfig._numChannels;
    return static_cast<uint32_t>(m_SampleCount * channelCount * DelayCommandTimeCommandCoefficient);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const UpsampleCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(UpsampleCommandTime * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const AuxCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    if(command.enabled)
    {
        return static_cast<uint32_t>(AuxCommandEnableTime * SafetyFactor);
    }
    else
    {
        return static_cast<uint32_t>(AuxCommandDisableTime * SafetyFactor);
    }
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const DeviceSinkCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(DeviceSinkCommandTime * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const CircularBufferSinkCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(CircularBufferSinkCommandTime * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const ReverbCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    NN_SDK_ASSERT(command.reverbConfig._numChannels == 1 || command.reverbConfig._numChannels == 2 || command.reverbConfig._numChannels == 4 || command.reverbConfig._numChannels == 6);

    if(command.enabled)
    {
        return static_cast<uint32_t>(ReverbCommandCoefficient * m_SampleCount * command.reverbConfig._numChannels * SafetyFactor);
    }
    else
    {
        return 0;
    }
}
uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const I3dl2ReverbCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);

    if(command.enabled)
    {
        return static_cast<uint32_t>(I3dl2ReverbCommandCoefficient * m_SampleCount * command.i3dl2ReverbConfig._numChannels * SafetyFactor);
    }
    else
    {
        return 0;
    }
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const PerformanceCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(PerformanceCommandTime * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const ClearMixBufferCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(ClearMixBufferCommandCoefficient * m_SampleCount * m_MixBufferCount * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const DownMix6chTo2chCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    return static_cast<uint32_t>(DownMix6chTo2chCommandTime * SafetyFactor);
}

uint32_t CommandProcessingTimeEstimatorVersion1::Estimate(const CopyMixBufferCommand& command) NN_NOEXCEPT
{
    NN_UNUSED(command);
    NN_SDK_ASSERT(false, "This command is not supported on Version1");
    return 0u;
}

}}}  // namespace nn::audio::server
