﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_SampleFormat.private.h>
#include "../dsp/audio_Dsp.h"
#include "audio_PerformanceMetricsManager.h"
#include "audio_CommandBuffer.h"
#include "audio_ServiceEffectInfo.h"
#include "audio_ServiceVoiceInfo.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_ServiceMixInfo.h"
#include "audio_ServiceSinkInfo.h"
#include "audio_ServiceSplitterInfo.h"
#include "audio_UpsamplerManager.h"
#include "../common/audio_PerformanceMetricsCommon.h"
#include "../common/audio_NodeIdManager.h"

#include "audio_CommandGenerator.h"

namespace nn { namespace audio { namespace server {

namespace {

const NodeId InvalidNodeId = common::NodeIdManager::GetInvalidNodeId();

} // anonymous namespace

EntryAspect::EntryAspect(CommandGenerator* publisher, PerformanceEntryType type, NodeId id) NN_NOEXCEPT
    : BaseAspect(publisher)
    , m_Published(false)
    , m_NodeId(id)
{
    if (m_Publisher->m_PerformanceManager != nullptr &&
        m_Publisher->m_PerformanceManager->IsInitialized() &&
        m_Publisher->m_PerformanceManager->GetNextEntry(&m_Addresses, type, id))
    {
        m_Published = true;
        m_Publisher->GeneratePerformanceCommand(&m_Addresses, PerformanceCommandType_Begin, m_NodeId);
    }
}

EntryAspect::~EntryAspect() NN_NOEXCEPT
{
    if (m_Published)
    {
        m_Publisher->GeneratePerformanceCommand(&m_Addresses, PerformanceCommandType_End, m_NodeId);
    }
}

DetailAspect::DetailAspect(CommandGenerator* publisher, PerformanceEntryType parentType, NodeId parentId, PerformanceDetailType target) NN_NOEXCEPT
    : BaseAspect(publisher)
    , m_ParentId(parentId)
    , m_Published(false)
{
    if (m_Publisher->m_PerformanceManager != nullptr &&
        m_Publisher->m_PerformanceManager->IsInitialized() &&
        m_Publisher->m_PerformanceManager->IsDetailTarget(m_ParentId) &&
        m_Publisher->m_PerformanceManager->GetNextEntry(&m_Addresses, target, parentType, m_ParentId))
    {
        m_Published = true;
        m_Publisher->GeneratePerformanceCommand(&m_Addresses, PerformanceCommandType_Begin, m_ParentId);
    }
}

DetailAspect::~DetailAspect() NN_NOEXCEPT
{
    if (m_Published)
    {
        m_Publisher->GeneratePerformanceCommand(&m_Addresses, PerformanceCommandType_End, m_ParentId);
    }
}

NullAspect::NullAspect(CommandBuffer* publisher) NN_NOEXCEPT
    : BaseAspect(publisher)
{

}

NullAspect::~NullAspect() NN_NOEXCEPT
{

}

template <>
EntryAspect PerfAspect<EntryAspect, CommandGenerator>::operator-> () NN_NOEXCEPT
{
    return EntryAspect(m_Publisher, m_Type, m_NodeId);
};

template <>
DetailAspect PerfAspect<DetailAspect, CommandGenerator>::operator-> () NN_NOEXCEPT
{
    return DetailAspect(m_Publisher, m_Type, m_NodeId, m_DetailType);
};

typedef PerfAspect<EntryAspect, CommandGenerator>  PerfEntry;
typedef PerfAspect<DetailAspect, CommandGenerator> PerfDetail;

CommandGenerator::CommandGenerator(
    ICommandBuffer* pCommandBuffer,
    const CommandListHeader* pListHeader,
    RendererSystemContext* sysContext,
    VoiceContext* voiceData,
    MixContext* mixData,
    EffectContext* effectData,
    SinkContext* sinkData,
    SplitterContext* splitterData,
    PerformanceManager* pPerformanceManager) NN_NOEXCEPT
    : m_pCommandBuffer(pCommandBuffer)
    , m_pCommandListHeader(pListHeader)
    , m_SysContext(sysContext)
    , m_VoiceContext(voiceData)
    , m_MixContext(mixData)
    , m_EffectContext(effectData)
    , m_SinkContext(sinkData)
    , m_SendContext(splitterData)
    , m_PerformanceManager(pPerformanceManager)
{
    m_pCommandBuffer->GenerateClearMixBufferCommand(InvalidNodeId);
}

void CommandGenerator::GenerateDataSourceCommand(server::VoiceInfo& voice, VoiceState& state, int channel) NN_NOEXCEPT
{
    const auto nodeId = voice._inParameter._nodeId;
    const auto sampleFormat = voice._inParameter._sampleFormat;
    const auto shouldDepop = voice._inParameter._shouldDepop;
    const auto destinationMixId = voice.GetDestinationMixId();
    const auto splitterInfoId = voice.GetSplitterInfoId();
    const auto isDestinationMix = destinationMixId != Invalid_MixId;
    const auto isDestinationSpliter = splitterInfoId != common::InvalidSplitterInfoId;

    // Depop
    if (isDestinationMix)
    {
        auto& mixInfo = m_MixContext->GetInfo(destinationMixId);
        m_pCommandBuffer->GenerateDepopPrepareCommand(&state, m_SysContext->depopBuffer, mixInfo.bufferCount, mixInfo.bufferOffset, nodeId, voice._inParameter._shouldDepop);
    }
    else if (isDestinationSpliter)
    {
        int searchIndex = 0;
        while (const auto destData = GetDestinationData(splitterInfoId, searchIndex++))
        {
            if (destData->IsConfigured() == false)
            {
                continue;
            }
            auto& mixInfo = m_MixContext->GetInfo(destData->GetMixId());
            m_pCommandBuffer->GenerateDepopPrepareCommand(&state, m_SysContext->depopBuffer, mixInfo.bufferCount, mixInfo.bufferOffset, nodeId, voice._inParameter._shouldDepop);
        }
    }

    // DataSource
    if(!shouldDepop)
    {
        switch (sampleFormat)
        {
            case SampleFormat_PcmInt16:
                m_pCommandBuffer->GeneratePcmInt16DataSourceCommand(&voice, &state, m_SysContext->mixBufferCount, channel, nodeId);
                break;
            case SampleFormat_Adpcm:
                NN_SDK_ASSERT_EQUAL(channel, 0);
                m_pCommandBuffer->GenerateAdpcmDataSourceCommand(&voice, &state, m_SysContext->mixBufferCount, nodeId);
                break;
            case SampleFormat_Opus:
                NN_SDK_ASSERT_EQUAL(channel, 0);
                m_pCommandBuffer->GenerateOpusDataSourceCommand(&voice, &state, m_SysContext->mixBufferCount, nodeId);
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
}

void CommandGenerator::GenerateVoiceMixCommand(const float* pMixVolume, const float* pMixVolumePrev, VoiceState* state, int mixBufferOffset, int mixBufferCount, int voiceBufferIndex, NodeId nodeId) NN_NOEXCEPT
{
    if (mixBufferCount < (sizeof(MixRampGroupedCommand) / sizeof(MixRampCommand)) + 1)
    {
        for (int i = 0; i < mixBufferCount; ++i)
        {
            const auto inputBufferIndex = voiceBufferIndex;
            const auto outputBufferIndex = i + mixBufferOffset;
            const auto volume0 = pMixVolumePrev[i];
            const auto volume1 = pMixVolume[i];

            const auto lastSampleAddress = m_SysContext->pServicePool->Translate(state->lastSamples + i, sizeof(int32_t));
            if(volume0 != 0.0f || volume1 != 0.0f)
            {
                m_pCommandBuffer->GenerateMixRampCommand(inputBufferIndex, outputBufferIndex, volume0, volume1, lastSampleAddress, nodeId);
            }
        }
    }
    else
    {
        const auto lastSamplesAddress = m_SysContext->pServicePool->Translate(state->lastSamples, sizeof(int32_t) * mixBufferCount);
        m_pCommandBuffer->GenerateMixRampGroupedCommand(mixBufferCount, voiceBufferIndex, mixBufferOffset, pMixVolume, pMixVolumePrev, lastSamplesAddress, nodeId);
    }
}

void CommandGenerator::GenerateBiquadFilterCommandForVoice(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, int channel, NodeId nodeId) NN_NOEXCEPT
{
    for (auto i = 0; i < VoiceType::BiquadFilterCountMax; ++i)
    {
        const auto& parameter = pVoice->_inParameter._biquadFilterParameter[i];
        if (!parameter.enable)
        {
            continue;
        }

        const auto input = static_cast<int8_t>(channel);
        const auto output = static_cast<int8_t>(channel);
        const auto needInitialization = !pVoice->isBiquadFilterEnabledPrev[i];

        m_pCommandBuffer->GenerateBiquadFilterCommand(mixBufferCount, &parameter, &state->biquadFilterState[i], input, output, needInitialization, nodeId);
    }
}

void CommandGenerator::GenerateVoiceCommand(server::VoiceInfo& voice) NN_NOEXCEPT
{
    const auto nodeId = voice._inParameter._nodeId;
    const auto channelCount = voice._inParameter._channelCount;

    // DataSource -> BiquadFilter(2) -> Volume -> VoiceMix(n)
    for (auto channel = 0; channel < channelCount; ++channel)
    {
        const auto channelResourceId = voice._inParameter._voiceChannelResourceIds[channel];
        auto& state = m_VoiceContext->GetDspSharedState(channelResourceId);
        auto& channelResources = m_VoiceContext->GetChannelResource(channelResourceId);
        const auto detailType = voice._inParameter._sampleFormat == SampleFormat_PcmInt16 ? PerformanceDetailType_PcmInt16DataSource : PerformanceDetailType_AdpcmDataSource;

        PerfDetail(this, PerformanceEntryType_Voice, nodeId, detailType)
            ->GenerateDataSourceCommand(voice, state, channel);


        if (voice._inParameter._shouldDepop)
        {
            voice._inParameter._volumePrev = 0.0f;
        }
        else if (voice.HasAnyConnection()) // Mix に接続されていないときはコマンドを発行しない
        {
            PerfDetail(this, PerformanceEntryType_Voice, nodeId, PerformanceDetailType_BiquadFilter)
                ->GenerateBiquadFilterCommandForVoice(&voice, &state, m_SysContext->mixBufferCount, channel, nodeId);
            PerfDetail(this, PerformanceEntryType_Voice, nodeId, PerformanceDetailType_Volume)
                ->m_pCommandBuffer->GenerateVolumeRampCommand(voice._inParameter._volumePrev, voice._inParameter._volume, m_SysContext->mixBufferCount + channel, nodeId);
            voice._inParameter._volumePrev = voice._inParameter._volume;

            if (voice.GetDestinationMixId() != Invalid_MixId)
            {
                auto& mixInfo = m_MixContext->GetInfo(voice.GetDestinationMixId());

                PerfDetail(this, PerformanceEntryType_Voice, nodeId, PerformanceDetailType_Mix)->
                    GenerateVoiceMixCommand(channelResources.GetMixVolume(), channelResources.GetMixVolumePrev(), &state, mixInfo.bufferOffset, mixInfo.bufferCount, m_SysContext->mixBufferCount + channel, nodeId);

                channelResources.UpdateInternalState();
            }
            else if (voice.GetSplitterInfoId() != common::InvalidSplitterInfoId)
            {
                int dstIdx = channel;
                while (auto* dstData = GetDestinationData(voice.GetSplitterInfoId(), dstIdx))
                {
                    dstIdx += channelCount;

                    // check if valid destination
                    if (dstData->IsConfigured() == false ||
                        dstData->GetMixId() >= m_MixContext->GetCount())
                    {
                        continue;
                    }
                    const auto* mixInfo = &m_MixContext->GetInfo(dstData->GetMixId());

                    GenerateVoiceMixCommand(dstData->GetMixVolume(), dstData->GetMixVolumePrev(), &state, mixInfo->bufferOffset, mixInfo->bufferCount, m_SysContext->mixBufferCount + channel, nodeId);
                    dstData->MarkAsNeedToUpdateInternalState();
                }
            }


            for (auto i = 0; i < VoiceType::BiquadFilterCountMax; ++i)
            {
                voice.isBiquadFilterEnabledPrev[i] = voice._inParameter._biquadFilterParameter[i].enable;
            }
        }
    } // for (channel)
}

void CommandGenerator::GenerateVoiceCommands() NN_NOEXCEPT
{
    for (auto i = 0; i < m_VoiceContext->GetCount(); ++i)
    {
        server::VoiceInfo& voice = m_VoiceContext->GetSortedInfo(i);

        if (voice.ShouldSkip() ||
            voice.UpdateForCommandGeneration(m_VoiceContext) == false)
        {
            continue;
        }

        PerfEntry(this, PerformanceEntryType_Voice, voice._inParameter._nodeId)
            ->GenerateVoiceCommand(voice);
    }

    m_SendContext->UpdateInternalState();
}

void CommandGenerator::GenerateBufferMixerCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(info.GetType(), ::nn::audio::EffectType_BufferMixer);
    auto param = reinterpret_cast<const BufferMixerParameter*>(info.GetParameter());
    if (info.IsEnabled())
    {
        const auto channelCountMax = param->_channelCountMax;

        for(int i = 0; i < channelCountMax; ++i)
        {
            const auto inputBufferIndex = param->_input[i] + mixBufferOffset;
            const auto outputBufferIndex = param->_output[i] + mixBufferOffset;
            const auto volume = param->_volume[i];
            if (volume != 0.0f)
            {
                m_pCommandBuffer->GenerateMixCommand(inputBufferIndex, outputBufferIndex, volume, nodeId);
            }
        }
    }
}

void CommandGenerator::GenerateDelayCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(info.GetType(), ::nn::audio::EffectType_Delay);

    DspAddr workBuffer = info.GetWorkBuffer();
    auto param = reinterpret_cast<const DelayParameter*>(info.GetParameter());
    auto state = info.GetStateBuffer();
    m_pCommandBuffer->GenerateDelayEffectCommand(mixBufferOffset, param, state, info.IsEnabled(), workBuffer, nodeId);
}

void CommandGenerator::GenerateReverbCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId, bool longSizePreDelaySupported) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(info.GetType(), ::nn::audio::EffectType_Reverb);

    DspAddr workBuffer = info.GetWorkBuffer();
    auto param = reinterpret_cast<const ReverbParameter*>(info.GetParameter());
    auto state = info.GetStateBuffer();
    m_pCommandBuffer->GenerateReverbEffectCommand(mixBufferOffset, param, state, info.IsEnabled(), longSizePreDelaySupported, workBuffer, nodeId);
}

void CommandGenerator::GenerateI3dl2ReverbEffectCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(info.GetType(), ::nn::audio::EffectType_I3dl2Reverb);

    DspAddr workBuffer = info.GetWorkBuffer();
    auto param = reinterpret_cast<const I3dl2ReverbParameter*>(info.GetParameter());
    auto state = info.GetStateBuffer();
    m_pCommandBuffer->GenerateI3dl2ReverbEffectCommand(mixBufferOffset, param, state, info.IsEnabled(), workBuffer, nodeId);
}

void CommandGenerator::GenerateAuxCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(info.GetType(), ::nn::audio::EffectType_Aux);

    auto param = reinterpret_cast<const AuxParameter*>(info.GetParameter());
    auto addresses = &info.m_AuxAddresses;

    if (info.IsEnabled())
    {
        // touch memory pools, no use buffer addresses itself.
        info.GetWorkBuffer(EffectInfoBase::Index_AuxSend);
        info.GetWorkBuffer(EffectInfoBase::Index_AuxReturn);
    }

    if (addresses->_sendBufferBase != NULL && addresses->_returnBufferBase != NULL)
    {
        const int channelCountMax = param->_channelCountMax;

        int offset = 0;
        for(int channel = 0; channel < channelCountMax; ++channel)
        {
            const auto input = param->_input[channel];
            const auto output = param->_output[channel];
            const auto isLastChannel = (channel == channelCountMax - 1);
            const auto sampleCount = m_pCommandListHeader->sampleCount;
            const auto updateCount = isLastChannel ? (offset + sampleCount) : 0;

            m_pCommandBuffer->GenerateAuxCommand(
                mixBufferOffset,
                input,
                output,
                addresses,
                info.IsEnabled(),
                param->_sampleCount,
                addresses->_sendBufferBase,
                addresses->_returnBufferBase,
                updateCount,
                offset,
                nodeId);

            offset += sampleCount;
        }
    }
}

void CommandGenerator::GenerateBiquadFilterEffectCommand(int mixBufferOffset, EffectInfoBase& info, NodeId nodeId) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(info.GetType(), ::nn::audio::EffectType_BiquadFilter);
    auto param = reinterpret_cast<const BiquadFilterEffectParameter*>(info.GetParameter());
    auto state = reinterpret_cast<BiquadFilterEffectState*>(info.GetStateBuffer());
    const auto channelCount = param->_channelCount;

    if (info.IsEnabled())
    {
        const auto needInitialization =
            param->_parameterStatus == EffectParameterStatus_Init ||
            param->_parameterStatus == EffectParameterStatus_UpdateParam;
        const BiquadFilterParameter filterParameter = {
            true, { param->_numerator[0], param->_numerator[1], param->_numerator[2] },
                  { param->_denominator[0], param->_denominator[1] }
        };

        for(int channel = 0; channel < channelCount; ++channel)
        {
            const auto input = param->_input[channel];
            const auto output = param->_output[channel];
            const auto pState = &state->filterStates[channel];

            m_pCommandBuffer->GenerateBiquadFilterCommand(mixBufferOffset, &filterParameter, pState, input, output, needInitialization, nodeId);
        }
    }
    else
    {
        for(int channel = 0; channel < channelCount; ++channel)
        {
            const auto input = param->_input[channel] + mixBufferOffset;
            const auto output = param->_output[channel] + mixBufferOffset;

            m_pCommandBuffer->GenerateCopyMixBufferCommand(input, output, nodeId);
        }
    }
}

void CommandGenerator::GenerateEffectCommand(const server::MixInfo& mixInfo) NN_NOEXCEPT
{
    const auto mixBufferOffset = mixInfo.bufferOffset;
    const auto longSizePreDelaySupported = mixInfo.longSizePreDelaySupported;
    const auto effectCount = m_EffectContext->GetCount();
    const auto effectOrder = reinterpret_cast<int32_t*>(mixInfo.effectProcessingOrder.GetPointer());
    NN_SDK_ASSERT(effectCount == 0 || effectOrder != nullptr);

    for (auto i = 0; i < effectCount; ++i)
    {
        const auto infoIndex = effectOrder[i];
        if (infoIndex == nn::audio::EffectInfo::InParameter::InvalidProcessingOrder)
        {
            break;
        }

        auto& info = m_EffectContext->GetInfo(infoIndex);
        NN_SDK_ASSERT(info.GetType() != EffectType_Invalid);
        NN_SDK_ASSERT(info.GetMixId() == mixInfo.mixId);

        if (info.ShouldSkip()) // has some error. (Contains unmapped etc.)
        {
            continue;
        }

        const auto entryType = mixInfo.mixId == MixInfo::FinalMixId ? PerformanceEntryType_FinalMix : PerformanceEntryType_SubMix;
        const auto nodeId = mixInfo.nodeId;

        switch (info.GetType())
        {
        case EffectType_BufferMixer:
            PerfDetail(this, entryType, nodeId, PerformanceDetailType_Mix)->GenerateBufferMixerCommand(mixBufferOffset, info, nodeId);
            break;
        case EffectType_Delay:
            PerfDetail(this, entryType, nodeId, PerformanceDetailType_Delay)->GenerateDelayCommand(mixBufferOffset, info, nodeId);
            break;
        case EffectType_Reverb:
            PerfDetail(this, entryType, nodeId, PerformanceDetailType_Reverb)->GenerateReverbCommand(mixBufferOffset, info, nodeId, longSizePreDelaySupported);
            break;
        case EffectType_I3dl2Reverb:
            PerfDetail(this, entryType, nodeId, PerformanceDetailType_I3dl2Reverb)->GenerateI3dl2ReverbEffectCommand(mixBufferOffset, info, nodeId);
            break;
        case EffectType_Aux:
            PerfDetail(this, entryType, nodeId, PerformanceDetailType_Aux)->GenerateAuxCommand(mixBufferOffset, info, nodeId);
            break;
        case EffectType_BiquadFilter:
            PerfDetail(this, entryType, nodeId, PerformanceDetailType_BiquadFilter)->GenerateBiquadFilterEffectCommand(mixBufferOffset, info, nodeId);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }

        info.UpdateForCommandGeneration();
    }
}

void CommandGenerator::GenerateMixCommands(server::MixInfo& mixInfo) NN_NOEXCEPT
{
    // Mix に接続されていないときはコマンドを発行しない
    if (mixInfo.HasAnyConnection() == false)
    {
        return;
    }

    if (mixInfo.destinationMixId != Invalid_MixId) // 従来の destination の場合
    {
        for (auto j = 0; j < mixInfo.bufferCount; ++j)
        {
            auto& pDestinationMixInfo = m_MixContext->GetInfo(mixInfo.destinationMixId);
            const auto nodeId = mixInfo.nodeId;

            for (auto k = 0; k < pDestinationMixInfo.bufferCount; ++k)
            {
                const auto volume = mixInfo.volume * mixInfo.mixVolume[j][k];
                const auto sourceIndex = j + mixInfo.bufferOffset;
                const auto destinationIndex = k + pDestinationMixInfo.bufferOffset;

                if (volume != 0.0f)
                {
                    m_pCommandBuffer->GenerateMixCommand(sourceIndex, destinationIndex, volume, nodeId);
                }
            }
        }
    }
    else if (mixInfo.splitterInfoId != common::InvalidSplitterInfoId)
    {
        int dstIdx = 0;
        while (const auto dstData = GetDestinationData(mixInfo.splitterInfoId, dstIdx++))
        {
            if (dstData->IsConfigured() == false)
            {
                continue;
            }

            const auto& destMix = m_MixContext->GetInfo(dstData->GetMixId());
            const auto src = (dstIdx - 1) % mixInfo.bufferCount + mixInfo.bufferOffset;
            for (auto k = 0; k < destMix.bufferCount; ++k)
            {
                const auto vol = mixInfo.volume * dstData->GetMixVolume(k);
                const auto dst = k + destMix.bufferOffset;
                if (vol != 0.0f)
                {
                    m_pCommandBuffer->GenerateMixCommand(src, dst, vol, mixInfo.nodeId);
                }
            }
        }
    }
    else
    {
        NN_SDK_ASSERT(0, "unexpected state\n");
    }
}

void CommandGenerator::GenerateSubMixCommand(server::MixInfo& subMixInfo) NN_NOEXCEPT
{
    // Depop
    m_pCommandBuffer->GenerateDepopForMixBuffersCommand(m_SysContext->depopBuffer, subMixInfo.bufferOffset, subMixInfo.bufferCount, subMixInfo.nodeId, subMixInfo.sampleRate);

    // Effect
    GenerateEffectCommand(subMixInfo);

    // Volume & Mix
    PerfDetail(this, PerformanceEntryType_SubMix, subMixInfo.nodeId, PerformanceDetailType_Mix)
        ->GenerateMixCommands(subMixInfo);
}

void CommandGenerator::GenerateSubMixCommands() NN_NOEXCEPT
{
    for (int i = 0; i < m_MixContext->GetCount(); ++i)
    {
        auto& subMixInfo = m_MixContext->GetSortedInfo(i);

        if (subMixInfo.isInUse == false || subMixInfo.mixId == MixInfo::FinalMixId)
        {
            continue;
        }
        PerfEntry(this, PerformanceEntryType_SubMix, subMixInfo.nodeId)
            ->GenerateSubMixCommand(subMixInfo);
    }
}

void CommandGenerator::GenerateFinalMixCommand() NN_NOEXCEPT
{
    // Depop
    auto& finalMixInfo = m_MixContext->GetFinalMixInfo();
    m_pCommandBuffer->GenerateDepopForMixBuffersCommand(m_SysContext->depopBuffer, finalMixInfo.bufferOffset, finalMixInfo.bufferCount, finalMixInfo.nodeId, finalMixInfo.sampleRate);

    NN_SDK_ASSERT(finalMixInfo.mixId == MixInfo::FinalMixId);
    GenerateEffectCommand(finalMixInfo);

    const auto nodeId = finalMixInfo.nodeId;

    for(auto i = 0; i < finalMixInfo.bufferCount; ++i)
    {
        PerfDetail(this, PerformanceEntryType_FinalMix, nodeId, PerformanceDetailType_Volume)
            ->m_pCommandBuffer->GenerateVolumeCommand(finalMixInfo.volume, finalMixInfo.bufferOffset + i, nodeId);
    }
}

void CommandGenerator::GenerateFinalMixCommands() NN_NOEXCEPT
{
    PerfEntry(this, PerformanceEntryType_FinalMix, m_MixContext->GetFinalMixInfo().nodeId)
        ->GenerateFinalMixCommand();
}

namespace {
bool IsUpsampleRequired(int sampleRate, const SinkInfoBase& pInfo) NN_NOEXCEPT
{
    return (pInfo.GetType() == common::SinkType_Device) &&
           (reinterpret_cast<const DeviceSinkState*>(pInfo.GetState())->pUpsamplerInfo == nullptr) &&
           (sampleRate != 48000);
}
}

void CommandGenerator::GenerateSinkCommands() NN_NOEXCEPT
{
    // Keep processing order CircularBufferSink -> DeviceSink, since need to perform DownMix in-place.
    const common::SinkType targetType[] = { common::SinkType_CircularBuffer, common::SinkType_Device };
    for (auto type : targetType)
    {
        for (auto i = 0; i < m_SinkContext->GetCount(); ++i)
        {
            auto& info = m_SinkContext->GetInfo(i);
            if (info.IsUsed() == false || info.GetType() != type)
            {
                continue;
            }

            if (IsUpsampleRequired(m_pCommandListHeader->sampleRate, info))
            {
                info.GetDeviceState()->pUpsamplerInfo = m_SysContext->pUpsamplerManager->Allocate();
            }
            PerfEntry(this, PerformanceEntryType_Sink, info.GetNodeId())
                ->GenerateSinkCommand(m_MixContext->GetFinalMixInfo().bufferOffset, &info);
        }
    }
}

void CommandGenerator::GenerateDeviceSinkCommand(int mixBufferOffset, server::SinkInfoBase* pSinkInfo) NN_NOEXCEPT
{
    const auto state = reinterpret_cast<const DeviceSinkState*>(pSinkInfo->GetState());
    const auto param = reinterpret_cast<const common::DeviceParameter*>(pSinkInfo->GetParameter());
    const auto deviceChannelCount = m_SysContext->channelCount;

    // Downmix
    if (param->useDownMixMatrix && deviceChannelCount == 2)
    {
        m_pCommandBuffer->GenerateDownMixCommand(mixBufferOffset, param->input, param->input, state->_downMixMatrixCoeff, InvalidNodeId);
    }
#ifdef NN_BUILD_CONFIG_OS_WIN
    else if (param->inputCount == 6 && deviceChannelCount == 2)
    {
        // Gmix default down mix parameter, See "Externals/odin2/hos/drivers/Raptor/Sources/Libraries/audio/gmix/adsp/gmix_Process.cpp"
        const int32_t defaultDownMixCoeff[4] = {0x10000, 0xb53c, 0x4000, 0xb53c};
        m_pCommandBuffer->GenerateDownMixCommand(mixBufferOffset, param->input, param->input, defaultDownMixCoeff, InvalidNodeId);
    }
#endif

    // Upsampler
    if (state->pUpsamplerInfo)
    {
        auto pUpsamplerInfo = state->pUpsamplerInfo;
        m_pCommandBuffer->GenerateUpsampleCommand(mixBufferOffset,
                                                 pUpsamplerInfo,
                                                 param->inputCount,
                                                 param->input,
                                                 m_pCommandListHeader->bufferCount,
                                                 m_pCommandListHeader->sampleCount,
                                                 m_pCommandListHeader->sampleRate,
                                                 InvalidNodeId);
    }

    // DeviceSink
    m_pCommandBuffer->GenerateDeviceSinkCommand(
        mixBufferOffset,
        pSinkInfo,
        m_SysContext->sessionId,
        m_pCommandListHeader->buffer,
        InvalidNodeId);
}

void CommandGenerator::GenerateCircularBufferSinkCommand(int mixBufferOffset, server::SinkInfoBase* pSinkInfo) NN_NOEXCEPT
{
    m_pCommandBuffer->GenerateCircularBufferSinkCommand(mixBufferOffset, pSinkInfo, InvalidNodeId);
}

void CommandGenerator::GenerateSinkCommand(int mixBufferOffset, server::SinkInfoBase* pSinkInfo) NN_NOEXCEPT
{
    if (pSinkInfo->ShouldSkip())
    {
        return;
    }

    switch (pSinkInfo->GetType())
    {
    case common::SinkType_Device:
        GenerateDeviceSinkCommand(mixBufferOffset, pSinkInfo);
        break;
    case common::SinkType_CircularBuffer:
        GenerateCircularBufferSinkCommand(mixBufferOffset, pSinkInfo);
        break;
    case common::SinkType_Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    pSinkInfo->UpdateForCommandGeneration();
}

void CommandGenerator::GeneratePerformanceCommand(const PerformanceEntryAddresses* entry, const PerformanceCommandType type, NodeId nodeId) NN_NOEXCEPT
{
    m_pCommandBuffer->GeneratePerformanceCommand(entry, type, nodeId);
}

server::SplitterDestinationData* CommandGenerator::GetDestinationData(common::SplitterInfoId infoId, int dataIndex) NN_NOEXCEPT
{
    if (infoId != common::InvalidSplitterInfoId)
    {
        return m_SendContext->GetDestinationData(infoId, dataIndex);
    }

    return nullptr; // no connection found.
}

size_t CommandGenerator::CalculateCommandBufferSize(const nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    // TODO: Replace std::max(std::max(...)) with std::max(xxx, yyy, ...) after we can use it.

    const auto voiceCount = parameter.voiceCount;
    const auto subMixCount = parameter.subMixCount;
    const auto effectCount = parameter.effectCount;
    const auto sinkCount = parameter.sinkCount;
    const auto splitterCount = parameter.splitterCount;
    const auto splitterSendChannelCount = parameter.splitterSendChannelCount;

    const auto voiceDataSourceCommandSizeMax = std::max(sizeof(PcmInt16DataSourceCommand), sizeof(AdpcmDataSourceCommand));
    const auto voiceBiquadFilterCommandSizeMax = sizeof(BiquadFilterCommand) * VoiceType::BiquadFilterCountMax;
    const auto voiceCommandSizeMax = (voiceDataSourceCommandSizeMax + voiceBiquadFilterCommandSizeMax + sizeof(VolumeRampCommand) + sizeof(MixRampGroupedCommand)) + sizeof(DepopPrepareCommand) * splitterCount;
    const auto voiceCommandsSize = voiceCommandSizeMax * voiceCount;

    const auto splitterCommandSizeMax = nn::audio::MixBufferCountMax * splitterSendChannelCount * sizeof(MixRampCommand);
    const auto splitterCommandsSize = splitterCommandSizeMax;

    const auto sinkCommandSizeMax = std::max(sizeof(CircularBufferSinkCommand), sizeof(DeviceSinkCommand));
    const auto sinkCommandsSize = sinkCommandSizeMax * sinkCount;

    const auto bufferMixerEffectCommandsSize = sizeof(MixCommand) * MixBufferCountMax;
    const auto auxEffectCommandSize = sizeof(AuxCommand) * MixBufferCountMax;
    const auto biquadFilterEffectCommandSize = std::max(sizeof(BiquadFilterCommand), sizeof(CopyMixBufferCommand)) * BiquadFilterEffectParameter::ChannelCountMax;
    const auto effectCommandSizeMax = std::max(std::max(bufferMixerEffectCommandsSize, std::max(std::max(std::max(sizeof(DelayCommand), sizeof(ReverbCommand)), sizeof(I3dl2ReverbCommand)), auxEffectCommandSize)), biquadFilterEffectCommandSize);
    const auto effectCommandsSize = effectCommandSizeMax * effectCount;

    const auto submixCommandsSize = (sizeof(DepopForMixBuffersCommand) + sizeof(MixCommand) * nn::audio::MixBufferCountMax * nn::audio::MixBufferCountMax) * subMixCount;
    const auto finalmixCommandsSize = sizeof(DepopForMixBuffersCommand) + sizeof(VolumeCommand) * MixBufferCountMax;

    const auto performanceCommandsSize = (common::GetRequiredEntryCount(parameter) + common::GetRequiredDetailCount(parameter)) * sizeof(PerformanceCommand);

    const auto size = voiceCommandsSize + splitterCommandsSize + effectCommandsSize + submixCommandsSize + finalmixCommandsSize + sinkCommandsSize + performanceCommandsSize;

    return size;
}


}}}  // namespace nn::audio::server
