﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>

#include <nn/nn_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/applet/applet_Types.h>

#include <nn/audio/detail/audio_AudioRendererTypesInternal.h>
#include <nn/audio/detail/audio_IAudioRendererManager.h>
#include <nn/audio/detail/audio_IAudioRenderer.h>
#include <nn/audio/detail/audio_IAudioDevice.h>

#include "../common/audio_AudioRendererSession.h"
#include "../common/audio_UpdateDataHeader.h"

#include "audio_AudioRenderSystemManager.h"

namespace nn { namespace audio { namespace server {

class AudioRendererManagerImpl : public nn::sf::ISharedObject
{
public:
    AudioRendererManagerImpl() NN_NOEXCEPT;
    virtual ~AudioRendererManagerImpl() NN_NOEXCEPT;
    nn::Result OpenAudioRenderer(nn::sf::Out<nn::sf::SharedPointer<::nn::audio::detail::IAudioRenderer>> outAudioRenderer, const nn::audio::detail::AudioRendererParameterInternal& parameter, nn::sf::NativeHandle workBufferHandle, nn::sf::NativeHandle processHandle, uint64_t workBufferSize, nn::applet::AppletResourceUserId appletResourceUserId) NN_NOEXCEPT;
    nn::Result GetWorkBufferSize(nn::sf::Out<std::int64_t> outSize, const nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT;
    nn::Result GetAudioDeviceService(nn::sf::Out<nn::sf::SharedPointer<::nn::audio::detail::IAudioDevice>> outAudioDevice, nn::applet::AppletResourceUserId& appletResourceUserId) NN_NOEXCEPT;
    nn::Result GetAudioDeviceServiceWithRevisionInfo(nn::sf::Out<nn::sf::SharedPointer<::nn::audio::detail::IAudioDevice>> outAudioDevice, nn::applet::AppletResourceUserId& appletResourceUserId, uint32_t revisionInfo) NN_NOEXCEPT;
    nn::Result OpenAudioRendererForManualExecution(nn::sf::Out<nn::sf::SharedPointer<nn::audio::detail::IAudioRenderer>> outAudioRenderer, const nn::audio::detail::AudioRendererParameterInternal& parameter, std::uint64_t workBufferAddress, nn::sf::NativeHandle&& processHandle, std::uint64_t size, nn::applet::AppletResourceUserId id) NN_NOEXCEPT;

    void Wake() NN_NOEXCEPT;
    void Sleep() NN_NOEXCEPT;
private:
    typedef nn::sf::ExpHeapAllocator MyAllocator;
    MyAllocator m_Allocator;
    static const int HeapBufferSize = 6 * 1024;
    std::aligned_storage<HeapBufferSize>::type m_HeapBuffer;
    nn::lmem::HeapHandle m_HeapHandle;
    std::array<int, nn::audio::common::AudioRendererSessionCountMax> m_SessionIdPool;
    int m_ActiveSessionCount;
    nn::os::Mutex m_ActiveSessionCountMutex;
    std::aligned_storage<sizeof(AudioRenderSystemManager)>::type m_AudioRendererSystemManagerBuffer;

    int AcquireSessionId() NN_NOEXCEPT;
    void ReleaseSessionId(int sessionId) NN_NOEXCEPT;

    class AudioRendererImpl;

public:
    // "IAudioDevice" service implementation
    class AudioDeviceImpl
    {
    public:
        AudioDeviceImpl(AudioRendererManagerImpl* pParent, nn::applet::AppletResourceUserId& id, common::RevisionInfo revisionInfo) NN_NOEXCEPT;
        ~AudioDeviceImpl() NN_NOEXCEPT;
        nn::Result ListAudioDeviceName(const nn::sf::OutBuffer& outDeviceNames, nn::sf::Out<std::int32_t> amountRet) NN_NOEXCEPT;
        nn::Result ListAudioDeviceNameAuto(const nn::sf::OutBuffer& outDeviceNames, nn::sf::Out<std::int32_t> amountRet) NN_NOEXCEPT;
        nn::Result SetAudioDeviceOutputVolume(const nn::sf::InBuffer& nameIn, float volume) NN_NOEXCEPT;
        nn::Result SetAudioDeviceOutputVolumeAuto(const nn::sf::InBuffer& nameIn, float volume) NN_NOEXCEPT;
        nn::Result GetAudioDeviceOutputVolume(const nn::sf::InBuffer& nameIn, nn::sf::Out<float> pOutVolume) NN_NOEXCEPT;
        nn::Result GetAudioDeviceOutputVolumeAuto(const nn::sf::InBuffer& nameIn, nn::sf::Out<float> pOutVolume) NN_NOEXCEPT;
        nn::Result GetActiveAudioDeviceName(const nn::sf::OutBuffer& outDeviceName) NN_NOEXCEPT;
        nn::Result GetActiveAudioDeviceNameAuto(const nn::sf::OutBuffer& outDeviceName) NN_NOEXCEPT;
        nn::Result QueryAudioDeviceSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT;
        nn::Result QueryAudioDeviceInputEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT;
        nn::Result QueryAudioDeviceOutputEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT;
        nn::Result GetActiveChannelCount(nn::sf::Out<std::int32_t> channelCount) NN_NOEXCEPT;
        nn::Result GetAudioSystemMasterVolumeSetting(nn::sf::Out<float> volume, const nn::sf::InBuffer& name) NN_NOEXCEPT;
    private:
        nn::Result ListAudioDeviceNameInternal(const nn::sf::OutBuffer& outDeviceNames, nn::sf::Out<std::int32_t> amountRet) NN_NOEXCEPT;
        nn::Result SetAudioDeviceOutputVolumeInternal(const nn::sf::InBuffer& nameIn, float volume) NN_NOEXCEPT;
        nn::Result GetAudioDeviceOutputVolumeInternal(const nn::sf::InBuffer& nameIn, nn::sf::Out<float> pOutVolume) NN_NOEXCEPT;
        nn::Result GetActiveAudioDeviceNameInternal(const nn::sf::OutBuffer& outDeviceName) NN_NOEXCEPT;

    private:
        nn::sf::SharedPointer<AudioRendererManagerImpl> m_Parent;
        nn::applet::AppletResourceUserId m_Id;
        common::RevisionInfo m_RevisionInfo;
    };
};

}}}  // namespace nn::audio::server
