﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>

#include <nn/audio/detail/audio_IAudioInManager.h>
#include <nn/audio/detail/audio_IAudioIn.h>

#include <nn/audio/audio_Result.h>
#include <nn/os/os_Thread.h>

namespace nn{ namespace audio { namespace server {

class AudioInManagerImpl : public nn::sf::ISharedObject
{
public:
    AudioInManagerImpl() NN_NOEXCEPT;
    virtual ~AudioInManagerImpl() NN_NOEXCEPT;
    nn::Result OpenAudioInProtocolSpecified(uint64_t protocolVersion, nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name, nn::audio::AudioInParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result OpenAudioIn(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name, nn::audio::AudioInParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result OpenAudioInAuto(nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name, nn::audio::AudioInParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result ListAudioIns(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT;
    nn::Result ListAudioInsAuto(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT;
    nn::Result ListAudioInsAutoFiltered(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet) NN_NOEXCEPT;
    void ReportCloseSession(int sessionId) NN_NOEXCEPT;
    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;
    nn::os::Mutex* GetMutex() NN_NOEXCEPT;

public:
    static const int NumberOfUacInSessions = 2; //UacManager::NumUacInSessions
    static const int NumberOfDefaultInSessions = 2; //nne::audio::gmix::NumNearVoiceSessions
    static const int NumberOfAudioInSessions = NumberOfDefaultInSessions + NumberOfUacInSessions;

private:
    static void ThreadFunc(void* arg) NN_NOEXCEPT;
    void ThreadFuncImpl() NN_NOEXCEPT;
    void FirstTimeInitialize();
    nn::Result OpenAudioInInternal(uint64_t protocolVersion, nn::sf::Out<nn::sf::SharedPointer<detail::IAudioIn>> outAudioIn, nn::sf::InBuffer name, nn::audio::AudioInParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::AudioInParameterInternal> audioInInter, nn::sf::OutBuffer nameOut, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT;
    nn::Result ListAudioInsInternal(nn::sf::OutBuffer outAudioInfo, nn::sf::Out<int> amountRet, bool filtered) NN_NOEXCEPT;

private:
    class AudioInImpl;
    typedef nn::sf::ExpHeapAllocator MyAllocator;
    MyAllocator m_Allocator;
    static const int SizeOfSession = 3 * 8 * 1024;
    static const int HeapBufferSize = NumberOfAudioInSessions * SizeOfSession;

    std::aligned_storage<HeapBufferSize>::type m_HeapBuffer;
    nn::lmem::HeapHandle m_HeapHandle;

    bool m_IsInitialized;
    os::Mutex m_Mutex;

    class SessionsCircularBuffer
    {
    public:
        SessionsCircularBuffer()
            : m_NextAvailableIndex(0)
            , m_NextFreeIndex(0)
            , m_CurrentAmount(NumberOfAudioInSessions)
        {
            for( int i = 0; i < NumberOfAudioInSessions; ++i )
            {
                m_AvailableSessions[i] = i;
            }
        }

        nn::Result GetNextSessions(int* sessions, bool isUac)
        {
            NN_RESULT_THROW_UNLESS(m_CurrentAmount > 0, ResultOutOfResource());

            //Ahub ids: [0-NumberOfDefaultInSessions) - [0,2)
            //Uac ids: [NumberOfDefaultInSessions-NumberOfAudioInSessions) - [2,4)
            auto startIndex = isUac ? NumberOfDefaultInSessions : 0;
            auto numSessions = isUac ? NumberOfAudioInSessions : NumberOfDefaultInSessions;
            int foundIndex = startIndex;
            for( ; foundIndex < numSessions; ++foundIndex )
            {
                if( m_AvailableSessions[foundIndex] == foundIndex )
                {
                    *sessions = foundIndex;
                    m_AvailableSessions[foundIndex] = -1;
                    break;
                }
            }
            if(foundIndex == (numSessions))
            {
                return ResultOutOfResource();
            }
            m_NextAvailableIndex = ( m_NextAvailableIndex + 1 ) % NumberOfAudioInSessions;
            --m_CurrentAmount;
            NN_RESULT_SUCCESS;
        }

        void InsertSession(int sessionId)
        {
            NN_SDK_ASSERT(sessionId >= 0);
            NN_SDK_ASSERT(m_AvailableSessions[sessionId] == -1);
            NN_SDK_ASSERT(sessionId < NumberOfAudioInSessions);
            NN_SDK_ASSERT(m_CurrentAmount < NumberOfAudioInSessions, "Can't hold more sessions (max %d)", NumberOfAudioInSessions);
            m_AvailableSessions[sessionId] = sessionId;
            m_NextFreeIndex = ( m_NextFreeIndex + 1 ) % NumberOfAudioInSessions;
            ++m_CurrentAmount;
        }
    private:

        int m_AvailableSessions[NumberOfAudioInSessions];
        int m_NextAvailableIndex;
        int m_NextFreeIndex;
        int m_CurrentAmount;
    };

    SessionsCircularBuffer m_Sessions;

    static const int SessionStackSize = 12 * 1024;
    char*            m_Stack;

    nn::os::SemaphoreType* m_pSemaphore;
    AudioInImpl*           m_Session[NumberOfAudioInSessions];
    nn::os::EventType      m_Event[NumberOfAudioInSessions];
    nn::applet::AppletResourceUserId m_AppletId[NumberOfAudioInSessions];
    nn::os::EventType       m_EventSemaphoreSignal;

    std::atomic<bool> m_IsThreadActive;
    os::ThreadType    m_Thread;
    bool              m_IsAwake;

};
}}} //namespace nn / audio / server
