﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dd/dd_Cache.h>

#include "audio_DspProfiler.h"
#include "../audio_AddrTypes.h"

#include "audio_dsp.h"
#include <nne/audio/audio.h>
#include <nn/audio/detail/audio_Log.h>
#include "audiodsp.h"

namespace nn { namespace audio { namespace dsp {

namespace {
nn::audio::SharedMem_t *g_pSharedMem;
nne::audio::adsp::MboxType g_RpcMailbox;
nne::audio::adsp::Session *g_pAdspSession;
uint64_t g_CpuAddr;
int32_t g_BufferSize;
int32_t g_DspAddr;
nn::dd::ProcessHandle g_ProcessHandle;
bool g_IsStarted = false;
bool g_IsInitialized = false;
} //anonymous namespace

void StartDspProfiler(nn::dd::ProcessHandle processHandle, uint64_t bufferAddr, uint32_t bufferSize) NN_NOEXCEPT
{
    if( !g_IsInitialized )
    {
        int16_t id;
        int32_t msg;

        id = NX_AUDIO_DSP_RPC_PROFILER_MBOXID;
        if (nne::audio::adsp::Mbox::MboxOpen(&g_RpcMailbox, &id, nullptr, nullptr, nullptr, nullptr, nullptr))
        {
            NN_DETAIL_AUDIO_ERROR("Unable to open the rpc intr mailbox\n");
            return;
        }
        nne::audio::adsp::OpenSession(&g_pAdspSession, "AudioDspProfiler");
        if( g_pAdspSession == nullptr )
        {
            NN_DETAIL_AUDIO_ERROR("Could not start adsp profiler app. Make sure to set \"ADSP_USE_PROFILER=1\" and rebuild ADSP firmware and audio process\n");
            return;
        }

        nne::audio::adsp::Mbox::MboxRecvMsg(&g_RpcMailbox, &msg, true, nn::TimeSpan::FromMilliSeconds(100));
        g_pSharedMem = (SharedMem_t*)g_pAdspSession->getSharedMemory();
        g_IsInitialized = true;
    }
    if( !g_IsStarted )
    {
        g_IsStarted = true;
        int32_t msg;

        g_ProcessHandle = processHandle;
        g_CpuAddr = bufferAddr;
        g_BufferSize = bufferSize;

        nne::audio::iova::Initialize();
        g_DspAddr = nne::audio::iova::Map(processHandle, bufferAddr, bufferSize);
        dd::FlushProcessDataCache(processHandle, bufferAddr, bufferSize);
        g_pSharedMem->rpcData.arg[0] = g_DspAddr;
        g_pSharedMem->rpcData.arg[1] = bufferSize;

        msg = NX_AUDIO_DSP_RPC_ENABLE_PROFILER;
        nne::audio::adsp::Mbox::MboxSendMsg(&g_RpcMailbox, msg, false, nn::TimeSpan::FromMilliSeconds(100));
        nne::audio::adsp::Mbox::MboxRecvMsg(&g_RpcMailbox, &msg, true, nn::TimeSpan::FromMilliSeconds(1000));
        NN_SDK_ASSERT(msg == NX_AUDIO_DSP_RPC_ENABLE_PROFILER_ACK);
    }
}

void StopDspProfiler() NN_NOEXCEPT
{
    if( g_IsStarted )
    {
        g_IsStarted = false;
        int32_t msg = NX_AUDIO_DSP_RPC_DISABLE_PROFILER;
        nne::audio::adsp::Mbox::MboxSendMsg(&g_RpcMailbox, msg, false, nn::TimeSpan::FromMilliSeconds(100));
        nne::audio::adsp::Mbox::MboxRecvMsg(&g_RpcMailbox, &msg, true, nn::TimeSpan::FromMilliSeconds(100));
        NN_SDK_ASSERT(msg == NX_AUDIO_DSP_RPC_DISABLE_PROFILER_ACK);

        dd::InvalidateProcessDataCache(g_ProcessHandle, g_CpuAddr, g_BufferSize);
        nne::audio::iova::Unmap(g_CpuAddr, g_BufferSize, g_DspAddr);
        nne::audio::iova::Finalize();
        os::CloseNativeHandle(g_ProcessHandle);
    }
}

}}}  // namespace nn::audio::dsp
