﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "audio_DeviceMixerManager.h"
#include "../common/audio_AudioRendererSession.h"

namespace nn { namespace audio {

namespace {
const int SessionCountMax = common::AudioRendererSessionCountMax;
} // anonymous namespace

Result GetPhysicalDeviceName(const char** outPhysicalName, const char* virtualName) NN_NOEXCEPT;

Result InitializeDevice(Device* p, const char* name, uint32_t sessionId)
{
    NN_SDK_ASSERT(sessionId < SessionCountMax);
    const char* physicalName;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetPhysicalDeviceName(&physicalName, name));
    NN_RESULT_DO(p->_client.Initialize(physicalName));
    std::strncpy(p->name, name, sizeof(p->name) - 1);
    p->pCurrent = nullptr;
    memset(&p->state[0], 0, sizeof(DownSamplerState) * p->InputCountMax);
    p->sessionId = sessionId;
    NN_RESULT_SUCCESS;
}

void FinalizeDevice(Device* p) NN_NOEXCEPT
{
    p->_client.Finalize();
}

DownSamplerState* GetDeviceDownsampleState(Device* p) NN_NOEXCEPT
{
    return &p->state[0];
}

int GetDeviceSampleRate(const Device* p) NN_NOEXCEPT
{
    return p->_client.GetSampleRate();
}

int GetDeviceChannelCount(const Device* p) NN_NOEXCEPT
{
    return p->_client.GetChannelCount();
}

bool IsDeviceReadyToFeedSamples(Device* p, int sampleCount)
{
    void* buffer = p->_client.AcquireBuffer(sampleCount);
    p->_client.ReleaseBuffer(0);
    return (buffer != nullptr);
}

void* GetDeviceBuffer(Device* p) NN_NOEXCEPT
{
    return p->pCurrent;
}

void WaitDevice(Device* p)
{
    return p->_client.Wait();
}

namespace {

const int DeviceCount = 1;
const int MainDeviceIndex = 0;
Device g_DeviceList[DeviceCount * SessionCountMax];
const char* g_DeviceName[DeviceCount] =
{
    "MainAudioOut",
};

} // anonymous namespace

void InitializeDeviceManager() NN_NOEXCEPT
{
    for (auto i = 0; i < DeviceCount; ++i)
    {
        for (auto j = 0; j < SessionCountMax; j++)
        {
            // TODO: needs to implement appropriate erro handling code
            NN_ABORT_UNLESS(
                InitializeDevice(&g_DeviceList[i * SessionCountMax + j], g_DeviceName[i], j).IsSuccess(),
                "No valid audio output is found on the system"
            );
        }
    }
}

void FinalizeDeviceManager() NN_NOEXCEPT
{
    for (auto i = 0; i < DeviceCount * SessionCountMax; ++i)
    {
        FinalizeDevice(&g_DeviceList[i]);
    }
}

Device* GetDevice(const char* name, uint32_t sessionId) NN_NOEXCEPT
{
    for (auto i = 0; i < DeviceCount * SessionCountMax; ++i)
    {
        if (strncmp(name, g_DeviceList[i].name, 256) == 0
            && sessionId == g_DeviceList[i].sessionId)
        {
            return &g_DeviceList[i];
        }
    }
    return nullptr;
}

void WaitDevice(int sessionId, int sampleCount) NN_NOEXCEPT
{
    while (!IsDeviceReadyToFeedSamples(&g_DeviceList[sessionId], sampleCount))
    {
        WaitDevice(&g_DeviceList[sessionId]);
    }
}

void BeginOutputToDevice(int sessionId, int sampleCount) NN_NOEXCEPT
{
    Device* p = &g_DeviceList[sessionId];
    p->pCurrent = p->_client.AcquireBuffer(sampleCount);
    NN_SDK_ASSERT(p->pCurrent);
    std::memset(p->pCurrent, 0, sizeof(int16_t) * GetDeviceChannelCount(p) * sampleCount);
}

void EndOutputToDevice(int sessionId, int sampleCount) NN_NOEXCEPT
{
    Device* p = &g_DeviceList[sessionId];
    p->_client.ReleaseBuffer(sampleCount);
}

}}  // namespace nn::audio
