﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "audio_EffectCommon.h" // f32
#include "audio_Qf.h"

namespace nn { namespace audio {

struct Axfx2DelayLine
{
    // The buffer pointers.  buffEnd is set to the last possible buffer sample.
    // (buffEnd - buff) is the "max delay" length. So the delay line is actually of length (maxDelay + 1)
    f32      *buff;
    f32      *buffEnd;
    int32_t   maxDelay; // Maximum delay allowed.
                         // The in/out pointers which wrap back to buff when they reach buffEnd
                         // maxDelay - (out - in) is the "current delay" length.
    f32      *in;
    f32      *out;
    int32_t   delay; // Actual delay configured (can be 0 through maxDelay)
                     // Used by allpass filter
    f32       coef;
};

/**
 * @brief           Zeros the delay line buffer
 * @param[in]       delay     delay-line structure
 * @details
 * Zeros the delay line buffer
 */
inline void Axfx2DelayClear(Axfx2DelayLine *delay)
{
    memset(delay->buff, 0, sizeof(f32) * delay->maxDelay);
}

/**
 * @brief           Changes the delay length of a delay line
 * @param[in]       delay     delay-line structure
 * @param[in]       new_delay number of samples delayed in the buffer
 * @return          FALSE if the new_delay is illegal bigger than maxDelay, TRUE otherwise.
 * @details
 * Changes the delay length of a delay line.  It can be no longer
 * than its maximum delay size set when initialized with
 * Axfx2DelayInitialize.
 */
inline bool Axfx2DelaySetDelay(Axfx2DelayLine *delay, int32_t newDelay)
{
    int32_t maxDelay = delay->maxDelay;

    if (delay->maxDelay < newDelay)
    {
        return false;
    }
    else
    {
        /* At max delay, the 'in' pointer is directly behind the (one less, to the left)
        'out' pointer.  A smaller delay results in the 'in' pointer shifting closer
        to the 'out' index by new_delay positions from the right.
        */
        delay->delay = newDelay;
        delay->in = (delay->buff + ((delay->out - delay->buff) + newDelay) % (maxDelay + 1));
    }

    return true;
}

/**
 * @brief           Get the next sample which will be returned by Axfx2DelayTick
 * @param[in]       delay     delay-line structure
 * @return          next sample (f32)
 * @details
 * Get the next sample which will be returned by Axfx2DelayTick
 */
inline f32 Axfx2DelayNextOut(const Axfx2DelayLine *delay)
{
    return *(delay->out);
}

/**
 * @brief           Get the last nth sample added to the delay line via Axfx2DelayTick.
 * @param[in]       delay     delay-line structure
 * @param[in]       indexBack nth last sample (0 = the most recently added)
 * @return          previous nth sample
 * @details
 * Get the last nth sample added to the delay line via Axfx2DelayTick.
 * If the index is larger than the delay size, wrapping will occur.
 * Only the last (maxDelay-1)th samples can be queried.
 */
inline f32 Axfx2DelayTapOut(const Axfx2DelayLine *delay, int32_t indexBack)
{
    const uint32_t maxDelayPlusOne = (delay->maxDelay + 1);
    f32 *p = delay->in - ((indexBack + 1));// % max_delay_plus_one);

    if (p < delay->buff)
    {
        p += maxDelayPlusOne;
    }

    return *p;
}

// Pass in the constants so they can be saved outside the big for loop.  Simpler code here as
// We don't need the + 1 adds and struct lookups
inline f32  Axfx2DelayTapOut(const f32 *in, const f32 *base, uint32_t indexBackPlusOne, const uint32_t maxDelayPlusOne)
{
    const f32 *p = in - indexBackPlusOne;

    if (p < base)
    {
        p += maxDelayPlusOne;
    }

    return *p;
}

/**
 * @brief           Add a new sample to the delay line wand return the next outgoing sample
 * @param[in]       delay    delay-line structure
 * @param[in]       input    sample to insert into the delay line
 * @return          outgoing sample
 * @details
 * Add a new sample to the delay line wand return the next outgoing sample.
 */
inline f32 Axfx2DelayTick(Axfx2DelayLine *delay, f32 input)
{
    const f32 *p = delay->buffEnd;

    // Write to in ptr and read from out ptr.
    *(delay->in++) = input;
    f32 samp = *(delay->out++);

    // Increment and wrap the pointers if at the end of buffer.
    if (p < (delay->in))
    {
        delay->in = delay->buff;
    }

    if (p < (delay->out))
    {
        delay->out = delay->buff;
    }

    return samp;
}

/// This version doesn't return a value so slightly faster.
inline void Axfx2DelayTickNoReturnVal(Axfx2DelayLine *delay, f32 input)
{
    const f32 *pEnd = delay->buffEnd;

    // Write to in ptr and read from out ptr.
    *(delay->in)++ = input;
    delay->out++;

    // Increment and wrap the pointers if at the end of buffer.
    if (pEnd < delay->in)
    {
        delay->in = delay->buff;
    }

    if (pEnd < delay->out)
    {
        delay->out = delay->buff;
    }
}

/**
 * @brief           Initialize Axfx2DelayLine
 * @param[in]       delay    delay-line structure
 * @param[in]       maxDelay
 * @param[in]       addr
 * @return          false if allocation failed
 * @details
 * Initialize an AXFX_DELAY_LINE structure with the maximum
 * number of samples allowed in the delay line.  The actual
 * delay is set to delay_length.  The delay line is cleared.
 */
inline bool Axfx2DelayInitialize(Axfx2DelayLine *delay, int32_t maxDelay, f32* addr)
{
    delay->buff = addr;

    // Set last sample pointer.  The difference equals the "max delay" length
    delay->buffEnd = (delay->buff + maxDelay);

    // Keep track of the max delay.
    delay->maxDelay = maxDelay;

    // Initialize the 'out' pointer to the start of the buffer.
    delay->out = delay->buff;

    // Indirectly set the 'in' tick pointer (should be the last sample of the buffer).
    Axfx2DelaySetDelay(delay, maxDelay);

    // Clear each sample in the buffer
    Axfx2DelayClear(delay);

    return true;
}

/**
 * @brief           Query the maximum allowed delay.
 * @param[in]       delay    delay-line structure
 * @return          delay size / sample count
 * @details
 * Query the maximum allowed delay.  This is the value passed
 * to Axfx2DelayInitialize. It should match the value of
 * delay->buffEnd - delay->buff.
 */
inline uint32_t Axfx2DelayGetMaxDelay(const Axfx2DelayLine *delay)
{
    return delay->maxDelay;
}

/**
 * @brief           Query the current delay length.
 * @param[in]       delay    delay-line structure
 * @return          delay size / sample count
 * @details
 * Query the current delay length.  This is the value passed
 * to Axfx2DelayInitialize or last call to Axfx2DelaySetDelay.
 */
inline uint32_t Axfx2DelayGetDelay(const Axfx2DelayLine *delay)
{
    return delay->delay;
}

// Calculate delay in terms of how many samples
// TODO: Replace T, U with f32.
template<typename T, typename U>
inline int32_t Axfx2DelayLineCalculateDelaySamples(T khz, U msec)
{
    return QfToS32(MultiQfQf(QF_CONST(khz), QF_CONST(msec)));
}

/**
 * @brief           Initialize an Axfx2DelayLine
 * @param[in]       allpass   delay-line structure
 * @param[in]       length    maximum number of samples to be delayed (0 is allowed)
 * @param[in]       coef      allpass filter coefficent
 * @details
 * Initialize an AXFX_DELAY_LINE structure as an allpass filter
 * with the maximum number of samples allowed in the delay line.
 * The actual delay is set to delay_length.
 */
inline bool Axfx2AllPassInitialize(Axfx2DelayLine *allpass, uint32_t length, f32 coef, f32* addr)
{
    bool result;

    result = Axfx2DelayInitialize(allpass, length, addr);

    if (result)
    {
        allpass->coef = coef;
    }

    return result;
}

/**
 * @brief           Reset the allpass filter coefficient value.
 * @param[in]       allpass  delay-line structure
 * @param[in]       coef     codfficient
 * @pre
 * - coef >= 0.0f && coef <= 1.0f
 * @details
 * Reset the allpass filter coefficient value.
 */
inline void Axfx2AllPassSetCoef(Axfx2DelayLine *allpass, f32 coef)
{
    allpass->coef = coef;
}

/**
 * @brief           Query the current delay length
 * @param[in]       allpass  delay-line structure
 * @return          delay size / sample count
 * @details
 * Query the current delay length. This is the value passed to AXFX_AllPass_Initialize.
 */
inline uint32_t Axfx2AllPassGetDelayLen(const Axfx2DelayLine *allpass)
{
    return Axfx2DelayGetDelay(allpass);
}

/**
 * @brief           Add a new sample to the allpass line while removing and returning the next outgoing sample
 * @param[in]       delay              delay-line structure
 * @param[in]       in                 incoming sample
 * @return          outgoing sample
 * @details
 * Add a new sample to the allpass line while removing and returning
 * the next outgoing sample.  The incoming and outgoing sample
 * is altered via the coef value.
 */
inline f32 Axfx2AllPassTick(Axfx2DelayLine *allpass, f32 in)
{
    f32 temp;
    f32 out;
    f32 coef;

    coef = allpass->coef;
    temp = in - coef *  *(allpass->out); // Axfx2DelayNextOut(allpass);
    out = coef * temp + Axfx2DelayTick(allpass, temp);

    return out;
}

// This version combines the work on the outside for loop to allow compiler to better help.
f32 Axfx2AllPassTick(Axfx2DelayLine *allpass, Axfx2DelayLine *allpass2,  Axfx2DelayLine *outLine, f32 in);

}} // nn::audio
