﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstdlib>
#include <cstring>

#include <nn/os/os_Cache.h>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_VoiceTypes.h>
#include "common/audio_Util.h"
#include "common/audio_WorkBufferAllocator.h"

#include "audio_VoiceInfoManager.h"

namespace nn {
namespace audio {

VoiceInfoManager::VoiceInfoManager() NN_NOEXCEPT
    : m_Mutex(true)
{
}

size_t VoiceInfoManager::GetWorkBufferSize(int voiceCount) NN_NOEXCEPT
{
    return (sizeof(VoiceInfo) + sizeof(VoiceChannelResource)) * voiceCount;
}

void VoiceInfoManager::Initialize(int voiceCount, void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT_GREATER_EQUAL(workBufferSize, GetWorkBufferSize(voiceCount));
    NN_SDK_ASSERT_GREATER_EQUAL(voiceCount, 0);

    common::WorkBufferAllocator allocator(pWorkBuffer, workBufferSize);

    // Setup VoiceInfo
    m_VoiceInfoCount = voiceCount;
    m_VoiceInfoUsedCount = 0;
    m_VoiceInfos = common::PrepareArray<VoiceInfo>(allocator.Allocate(sizeof(VoiceInfo) * voiceCount), voiceCount);
    m_VoiceInfoUsedList.clear();
    m_VoiceInfoFreeList.clear();
    for (int i = 0; i < voiceCount; ++i)
    {
        m_VoiceInfos[i]._inParameter._isInUse = false;
        m_VoiceInfos[i]._inParameter._id = i;
        m_VoiceInfos[i].nodeIdBase = i;
        m_VoiceInfos[i].nodeIdVariation = 0;

        for(int j = 0; j < VoiceType::ChannelCountMax; ++j)
        {
            m_VoiceInfos[i]._pVoiceChannelResources[j] = nullptr;
        }

        m_VoiceInfoFreeList.push_back(m_VoiceInfos[i]);
    }

    // Setup VoiceChannelResource
    m_VoiceChannelResourceCount = voiceCount;
    m_VoiceChannelResourceUsedCount = 0;
    m_VoiceChannelResources = common::PrepareArrayWithIndex<VoiceChannelResource>(allocator.Allocate(sizeof(VoiceChannelResource) * voiceCount), voiceCount);
    m_VoiceChannelResourceFreeList.clear();

    for (int i = 0; i < voiceCount; ++i)
    {
        m_VoiceChannelResourceFreeList.push_back(m_VoiceChannelResources[i]);
    }
}

VoiceInfo* VoiceInfoManager::GetFreeVoiceInfo() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_VoiceInfoFreeList.size() > 0)
    {
        auto p = &m_VoiceInfoFreeList.front();
        m_VoiceInfoFreeList.pop_front();
        return p;
    }
    return nullptr;
}

VoiceInfo* VoiceInfoManager::AcquireVoiceInfo(int priority, int channelCount) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT_MINMAX(priority, VoiceType::PriorityHighest, VoiceType::PriorityLowest);
    NN_SDK_ASSERT_MINMAX(channelCount, 1, VoiceType::ChannelCountMax);

    // Check VoiceInfo remain count
    if(m_VoiceInfoCount - m_VoiceInfoUsedCount < 1)
    {
        return nullptr;
    }

    // Check VoiceChannelResource remain count
    if(m_VoiceChannelResourceCount - m_VoiceChannelResourceUsedCount < channelCount)
    {
        return nullptr;
    }

    // Setup VoiceInfo
    auto p = GetFreeVoiceInfo();
    NN_SDK_ASSERT_NOT_NULL(p);

    p->_inParameter._isInUse = true;
    p->_inParameter._priority = priority;
    memset(p->_reserved, 0, sizeof(p->_reserved));

    InsertToUsedList(p);

    // Setup VoiceChannelResource
    for(int i = 0; i < channelCount; ++i)
    {
        auto pVoiceChannelResource = &m_VoiceChannelResourceFreeList.front();
        m_VoiceChannelResourceFreeList.pop_front();

        p->_pVoiceChannelResources[i] = pVoiceChannelResource;
        p->_inParameter._voiceChannelResourceIds[i] = pVoiceChannelResource->GetId();
        pVoiceChannelResource->SetInUse(true);
    }

    p->_inParameter._waveBufferCount = 0;
    p->_inParameter._waveBufferHead = 0;
    p->_waveBufferReleased = 0;
    std::memset(p->_inParameter._waveBuffer, 0, sizeof(p->_inParameter._waveBuffer));

    for (int i = 0; i < VoiceType::WaveBufferCountMax; ++i)
    {
        p->_inParameter._waveBuffer[i]._isSentToServer = true;
    }
    p->waveBufferAppended = 0;

    // Update used count
    ++m_VoiceInfoUsedCount;
    m_VoiceChannelResourceUsedCount += channelCount;

    return p;
}

void VoiceInfoManager::ReleaseVoiceInfo(VoiceInfo* entry) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES(IsVoiceInfoValid(entry));

    const auto channelCount = entry->_inParameter._channelCount;
    NN_SDK_ASSERT_MINMAX(channelCount, 1, VoiceType::ChannelCountMax);

    // Release VoiceInfo
    NN_SDK_ASSERT_GREATER(m_VoiceInfoUsedCount, 0);
    NN_SDK_ASSERT(entry->_inParameter._isInUse);

    --m_VoiceInfoUsedCount;
    m_VoiceInfoUsedList.erase(m_VoiceInfoUsedList.iterator_to(*entry));
    m_VoiceInfoFreeList.push_back(*entry);
    entry->_inParameter._isInUse = false;

    // Release VoiceChannelResource
    for(int i = 0; i < channelCount; ++i)
    {
        NN_SDK_ASSERT_GREATER(m_VoiceChannelResourceUsedCount, 0);
        NN_SDK_ASSERT_NOT_NULL(entry->_pVoiceChannelResources[i]);
        NN_SDK_ASSERT(entry->_pVoiceChannelResources[i]->IsInUse());

        --m_VoiceChannelResourceUsedCount;
        m_VoiceChannelResourceFreeList.push_back(*entry->_pVoiceChannelResources[i]);

        entry->_pVoiceChannelResources[i]->SetInUse(false);
        entry->_pVoiceChannelResources[i] = nullptr;
    }
}

bool VoiceInfoManager::IsVoiceInfoValid(const VoiceInfo* pVoiceInfo) const NN_NOEXCEPT
{
    // VoiceInfo 用に確保したメモリ範囲内かチェックした後、ポインタとして適切なアドレスかを確認
    return m_VoiceInfos <= pVoiceInfo
        && pVoiceInfo <= (m_VoiceInfos + m_VoiceInfoCount)
        && (reinterpret_cast<uintptr_t>(pVoiceInfo) - reinterpret_cast<uintptr_t>(m_VoiceInfos)) % sizeof(VoiceInfo) == 0;
}

void VoiceInfoManager::InsertToUsedList(VoiceInfo* entry) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(entry);

    for(auto itr = m_VoiceInfoUsedList.begin(); itr != m_VoiceInfoUsedList.end(); ++itr)
    {
        if(itr->_inParameter._priority >= entry->_inParameter._priority)
        {
            m_VoiceInfoUsedList.insert(itr, *entry);
            return;
        }
    }

    m_VoiceInfoUsedList.push_back(*entry);
}

void VoiceInfoManager::FlushVoiceDataCache(const VoiceInfo* entry) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    auto& param = entry->_inParameter;

    if (param._isNew && param._pAdditionalParameter.IsNullPtr() == false && param._additionalParameterSize > 0u)
    {
        os::FlushDataCache(param._pAdditionalParameter.GetPointer(),
                           param._additionalParameterSize);
    }

    for (auto waveBuffer : param._waveBuffer)
    {
        if (waveBuffer._isSentToServer)
        {
            continue;
        }
        if (waveBuffer.buffer.GetPointer() != nullptr && waveBuffer.size > 0u)
        {
            os::FlushDataCache(waveBuffer.buffer.GetPointer(), waveBuffer.size);
        }
        if (param._sampleFormat == nn::audio::SampleFormat_Adpcm &&
            waveBuffer.pContext.GetPointer() != nullptr && waveBuffer.contextSize > 0u)
        {
            os::FlushDataCache(waveBuffer.pContext.GetPointer(), waveBuffer.contextSize);
        }
    }
#else
    NN_UNUSED(entry);
#endif
}
void VoiceInfoManager::UpdateVoiceRelatedParameters(
    void* pOutParameter,
    size_t* pOutInParamDataSize,
    size_t* pOutChParamDataSize) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    uintptr_t ptr{ reinterpret_cast<uintptr_t>(pOutParameter) };
    *pOutChParamDataSize = UpdateVoiceChannelResourceInParameter(reinterpret_cast<VoiceChannelResource::InParameter*>(ptr));
    ptr += *pOutChParamDataSize;
    *pOutInParamDataSize = UpdateVoiceInParameter(reinterpret_cast<VoiceInfo::InParameter*>(ptr));
}

size_t VoiceInfoManager::UpdateVoiceInParameter(VoiceInfo::InParameter* pParameter) NN_NOEXCEPT
{
    // Update sortingOrder
    int sortingOrder = 0;
    for(auto& voiceInfo : m_VoiceInfoUsedList)
    {
        voiceInfo._inParameter._sortingOrder = sortingOrder;
        ++sortingOrder;
    }

    for (auto i = 0; i < m_VoiceInfoCount;++i)
    {
        auto& voiceInfo = m_VoiceInfos[i];

        if (voiceInfo._inParameter._isInUse)
        {
            FlushVoiceDataCache(&voiceInfo);
            pParameter[i] = voiceInfo._inParameter;
        }
        else
        {
            pParameter[i]._isInUse = false;
        }
    }

    return sizeof(VoiceInfo::InParameter) * m_VoiceInfoCount;

}

size_t VoiceInfoManager::UpdateVoiceOutStatus(VoiceInfo::OutStatus* pStatus) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    for (auto i = 0; i < m_VoiceInfoCount; ++i)
    {
        if(m_VoiceInfos[i]._inParameter._isInUse)
        {
            m_VoiceInfos[i]._outStatus = pStatus[i];
            m_VoiceInfos[i]._inParameter._isNew = false;
            m_VoiceInfos[i]._inParameter._isVoiceDroppedFlagClearRequested = false;
            m_VoiceInfos[i]._inParameter._waveBufferFlushRequestedCount = 0;
            for (int j = 0; j < VoiceType::WaveBufferCountMax; ++j)
            {
                m_VoiceInfos[i]._inParameter._waveBuffer[j]._isSentToServer = true;
            }
        }
    }
    return sizeof(VoiceInfo::OutStatus) * m_VoiceInfoCount;
}

size_t VoiceInfoManager::UpdateVoiceChannelResourceInParameter(VoiceChannelResource::InParameter* pParameter) NN_NOEXCEPT
{
    for (auto i = 0; i < m_VoiceChannelResourceCount; ++i)
    {
        m_VoiceChannelResources[i].Update(&pParameter[i]);
    }

    return sizeof(VoiceChannelResource::InParameter) * m_VoiceChannelResourceCount;
}

}  // namespace audio
}  // namespace nn

