﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <limits>

#include <nn/util/util_BitUtil.h>
#include <nn/nn_SdkAssert.h>

#include <nn/audio/audio_Common.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_VoiceApi.h>
#include <nn/audio/audio_VoiceTypes.h>
#include <nn/audio/audio_FinalMixApi.h>
#include <nn/audio/audio_FinalMixTypes.h>
#include <nn/audio/audio_SampleFormat.private.h>
#include <nn/audio/audio_SubMixTypes.h>
#include <nn/audio/audio_SubMixApi.h>
#include <nn/audio/audio_SplitterApi.h>

#include "audio_VoiceInfoManager.h"
#include "audio_SplitterInfoManager.h"
#include "audio_MixManager.h"
#include "common/audio_NodeIdManager.h"
#include "common/audio_CommonWaveBuffer.h"
#include "audio_ResourceExclusionChecker.h"
#include "audio_AudioCodecBufferManager.h"

#define NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessChecker(detail::FindResourceExclusionCheckerFromRegionInConfig(pVoice->_pVoiceInfo))

namespace nn {
namespace audio {

namespace detail {

bool AcquireVoiceSlot(AudioRendererConfig* pConfig, VoiceType* pOutVoice, int sampleRate, int channelCount, SampleFormat sampleFormat, int priority, const void* pParameter, size_t size, const VoiceType::BehaviorOptions* pBehaviorOptions) NN_NOEXCEPT
{
    uint32_t audioCodecBufferOffset = 0;
    uint32_t audioCodecBufferSize = 0;
    if (sampleFormat == static_cast<SampleFormat>(SampleFormat_Opus))
    {
        auto pExtraManager = pConfig->_pExtraManager;
        if (pExtraManager == nullptr)
        {
            return false;
        }
        const size_t OpusMonoWorkBufferSize = 64 * 1024;  // TODO: derive "64 * 1024" properly
        size_t offset;
        if (!pExtraManager->AcquireBuffer(&offset, OpusMonoWorkBufferSize))
        {
            return false;
        }
        audioCodecBufferOffset = static_cast<uint32_t>(offset);
        audioCodecBufferSize = static_cast<uint32_t>(OpusMonoWorkBufferSize);
    }

    auto pVoiceInfoManager = pConfig->_pVoiceInfoManager;
    auto pOutInfo = pVoiceInfoManager->AcquireVoiceInfo(priority, channelCount);
    pOutVoice->_pVoiceInfo = pOutInfo;
    if (pOutInfo == nullptr)
    {
        auto pExtraManager = pConfig->_pExtraManager;
        if (pExtraManager != nullptr && audioCodecBufferSize > 0)
        {
            pExtraManager->Release(audioCodecBufferOffset, audioCodecBufferSize);
        }
        return false;
    }

    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pOutVoice);

    pOutInfo->_inParameter._nodeId = common::NodeIdManager::GetNodeId(common::NodeIdManager::NodeIdType::Voice, pOutInfo->nodeIdBase, pOutInfo->nodeIdVariation);
    ++pOutInfo->nodeIdVariation;
    if(pOutInfo->nodeIdVariation > common::NodeIdManager::VariationMax)
    {
        pOutInfo->nodeIdVariation = 0;
    }

    pOutInfo->_inParameter._playState = VoiceType::PlayState_Stop;
    pOutInfo->_inParameter._priority = priority;

    pOutInfo->_inParameter._sampleRate = sampleRate;
    pOutInfo->_inParameter._sampleFormat = static_cast<uint8_t>(sampleFormat);
    pOutInfo->_inParameter._channelCount = channelCount;

    pOutInfo->_inParameter._pitch = 1.0f;
    pOutInfo->_inParameter._volume = 1.0f;

    pOutInfo->_inParameter._isNew = true;
    pOutInfo->_inParameter._isVoiceDroppedFlagClearRequested = false;
    pOutInfo->_inParameter._destinationMixId = nn::audio::Invalid_MixId;
    pOutInfo->_inParameter._splitterInfoId = common::InvalidSplitterInfoId;
    pOutInfo->_inParameter._waveBufferFlushRequestedCount = 0;

    // Set default value(0.0f) to mixVolume
    for(auto i = 0; i < channelCount; ++i)
    {
        auto pVoiceChannelResource = pOutInfo->_pVoiceChannelResources[i];
        NN_SDK_ASSERT_NOT_NULL(pVoiceChannelResource);
        pVoiceChannelResource->ClearMixVolumes();
    }

    for (auto i = 0; i < VoiceType::BiquadFilterCountMax; ++i)
    {
        pOutInfo->_inParameter._biquadFilterParameter[i].enable = false;
        std::memset(pOutInfo->_inParameter._biquadFilterParameter[i].numerator, 0, sizeof(pOutInfo->_inParameter._biquadFilterParameter[i].numerator));
        std::memset(pOutInfo->_inParameter._biquadFilterParameter[i].denominator, 0, sizeof(pOutInfo->_inParameter._biquadFilterParameter[i].denominator));
    }

    pOutInfo->_outStatus._waveBufferConsumed = 0;
    pOutInfo->_outStatus._playedSampleCount = 0;
    pOutInfo->_outStatus._voiceDroppedFlag = false;

    if (sampleFormat == SampleFormat_Adpcm)
    {
        NN_SDK_REQUIRES_NOT_NULL(pParameter);
    }
    pOutInfo->_inParameter._pAdditionalParameter = pParameter;
    pOutInfo->_inParameter._additionalParameterSize = size;

    pOutInfo->_inParameter._behaviorFlags.Clear();
    if(pBehaviorOptions)
    {
        pOutInfo->_inParameter._behaviorFlags.Set<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>(pBehaviorOptions->isPlayedSampleCountResetAtLoopPoint);
        pOutInfo->_inParameter._behaviorFlags.Set<VoiceBehaviorFlag::IsPitchAndSrcSkipped>(pBehaviorOptions->isPitchAndSrcSkipped);
    }

    pOutInfo->_inParameter._audioCodecBufferOffset = audioCodecBufferOffset;
    pOutInfo->_inParameter._audioCodecBufferSize = audioCodecBufferSize;

    return true;
}

} // anonymous namespace

bool AcquireVoiceSlot(AudioRendererConfig* pConfig, VoiceType* pOutVoice, int sampleRate, int channelCount, SampleFormat sampleFormat, int priority, const void* pParameter, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutVoice);
    NN_SDK_REQUIRES_GREATER(sampleRate, 0);
    NN_SDK_REQUIRES(channelCount == 1 || (sampleFormat == SampleFormat_PcmInt16 && channelCount >= 1 && channelCount <= VoiceType::ChannelCountMax));
    NN_SDK_REQUIRES_MINMAX(priority, VoiceType::PriorityHighest, VoiceType::PriorityLowest);
    NN_SDK_REQUIRES(pParameter == nullptr || nn::util::is_aligned(reinterpret_cast<uintptr_t>(pParameter), nn::audio::BufferAlignSize));
    NN_SDK_REQUIRES(pParameter == nullptr || size > 0);
    NN_SDK_REQUIRES(sampleFormat == nn::audio::SampleFormat_PcmInt16 || sampleFormat == nn::audio::SampleFormat_Adpcm || sampleFormat == static_cast<SampleFormat>(nn::audio::SampleFormat_Opus));

    return detail::AcquireVoiceSlot(pConfig, pOutVoice, sampleRate, channelCount, sampleFormat, priority, pParameter, size, nullptr);
}

bool AcquireVoiceSlot(AudioRendererConfig* pConfig, VoiceType* pOutVoice, int sampleRate, int channelCount, SampleFormat sampleFormat, int priority, const void* pParameter, size_t size, const VoiceType::BehaviorOptions* pBehaviorOptions) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutVoice);
    NN_SDK_REQUIRES_GREATER(sampleRate, 0);
    NN_SDK_REQUIRES(channelCount == 1 || (sampleFormat == SampleFormat_PcmInt16 && channelCount >= 1 && channelCount <= VoiceType::ChannelCountMax));
    NN_SDK_REQUIRES_MINMAX(priority, VoiceType::PriorityHighest, VoiceType::PriorityLowest);
    NN_SDK_REQUIRES(pParameter == nullptr || nn::util::is_aligned(reinterpret_cast<uintptr_t>(pParameter), nn::audio::BufferAlignSize));
    NN_SDK_REQUIRES(pParameter == nullptr || size > 0);
    NN_SDK_REQUIRES(sampleFormat == nn::audio::SampleFormat_PcmInt16 || sampleFormat == nn::audio::SampleFormat_Adpcm || sampleFormat == static_cast<SampleFormat>(nn::audio::SampleFormat_Opus));
    NN_SDK_REQUIRES_NOT_NULL(pBehaviorOptions);
    NN_SDK_REQUIRES(pBehaviorOptions->isPitchAndSrcSkipped == false || pConfig->_pMixManager->GetFinalMixSampleRate() == sampleRate);

    return detail::AcquireVoiceSlot(pConfig, pOutVoice, sampleRate, channelCount, sampleFormat, priority, pParameter, size, pBehaviorOptions);
}

bool IsVoiceValid(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    if (pVoiceInfo == nullptr)
    {
        return false;
    }

    return true;
}

bool IsVoiceDroppedFlagOn(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    NN_SDK_REQUIRES_NOT_NULL(pVoiceInfo);
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);

    return pVoiceInfo->_outStatus._voiceDroppedFlag && pVoiceInfo->_inParameter._isVoiceDroppedFlagClearRequested == false;
}

void ResetVoiceDroppedFlag(VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVoice);
    auto pVoiceInfo = pVoice->_pVoiceInfo;
    NN_SDK_REQUIRES_NOT_NULL(pVoiceInfo);
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);

    pVoiceInfo->_inParameter._isVoiceDroppedFlagClearRequested = true;
}

void SetVoiceDestination(AudioRendererConfig* pConfig, VoiceType* pSource, FinalMixType* pDestination) NN_NOEXCEPT
{
    auto const pSourceInfo = pSource->_pVoiceInfo;

    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES(IsVoiceValid(pSource));
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES(GetVoiceBehaviorOptions(pSource).isPitchAndSrcSkipped == false || GetVoiceSampleRate(pSource) == GetFinalMixSampleRate(pDestination));

    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);

    pSourceInfo->_inParameter._destinationMixId = pDestination->_pMixInfo->mixId;
    pSourceInfo->_inParameter._splitterInfoId = common::InvalidSplitterInfoId;
}

void SetVoiceDestination(AudioRendererConfig* pConfig, VoiceType* pSource, SubMixType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES(IsVoiceValid(pSource));
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES(GetVoiceBehaviorOptions(pSource).isPitchAndSrcSkipped == false || GetVoiceSampleRate(pSource) == GetSubMixSampleRate(pDestination));

    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);

    auto const pSourceInfo = pSource->_pVoiceInfo;
    pSourceInfo->_inParameter._destinationMixId = pDestination->_pMixInfo->mixId;
    pSourceInfo->_inParameter._splitterInfoId = common::InvalidSplitterInfoId;
}

void SetVoiceDestination(AudioRendererConfig* pConfig, VoiceType* pSource, SplitterType* pDestination) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pConfig->_pSplitterInfoManager);
    NN_SDK_REQUIRES(IsVoiceValid(pSource));
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_NOT_NULL(pDestination->_pSplitterInfo);
    NN_SDK_REQUIRES(GetVoiceBehaviorOptions(pSource).isPitchAndSrcSkipped == false || GetVoiceSampleRate(pSource) == GetSplitterSampleRate(pDestination));

    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pConfig);

    auto const pSourceInfo = pSource->_pVoiceInfo;
    pSourceInfo->_inParameter._destinationMixId = Invalid_MixId;
    pSourceInfo->_inParameter._splitterInfoId = pDestination->_pSplitterInfo->GetId();
}

void ReleaseVoiceSlot(AudioRendererConfig* pConfig, VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;

    if (!IsVoiceValid(pVoice))
    {
        return;
    }

    {
        auto offset = pVoiceInfo->_inParameter._audioCodecBufferOffset;
        auto size = pVoiceInfo->_inParameter._audioCodecBufferSize;

        auto pExtraManager = pConfig->_pExtraManager;
        if (pExtraManager != nullptr && size > 0)
        {
            pExtraManager->Release(offset, size);
        }
    }

    auto pVoiceInfoManager= pConfig->_pVoiceInfoManager;
    pVoiceInfoManager->ReleaseVoiceInfo(pVoiceInfo);
    pVoice->_pVoiceInfo = nullptr;
}

int GetVoiceSampleRate(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._sampleRate;
}

int GetVoiceChannelCount(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._channelCount;
}

SampleFormat GetVoiceSampleFormat(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return static_cast<SampleFormat>(pVoiceInfo->_inParameter._sampleFormat);
}

void SetVoicePriority(VoiceType* pVoice, int priority) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_MINMAX(priority, VoiceType::PriorityHighest, VoiceType::PriorityLowest);
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    pVoiceInfo->_inParameter._priority = priority;
}

int GetVoicePriority(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._priority;
}

void SetVoicePlayState(VoiceType* pVoice, VoiceType::PlayState playState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    pVoiceInfo->_inParameter._playState = static_cast<uint8_t>(playState);
}

VoiceType::PlayState GetVoicePlayState(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return static_cast<VoiceType::PlayState>(pVoiceInfo->_inParameter._playState);
}

void SetVoicePitch(VoiceType* pVoice, float pitch) NN_NOEXCEPT
{
    auto const pVoiceInfo = pVoice->_pVoiceInfo;

    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES(GetVoiceBehaviorOptions(pVoice).isPitchAndSrcSkipped == false);
    NN_SDK_REQUIRES_MINMAX(pitch, VoiceType::GetPitchMin(), VoiceType::GetPitchMax());

    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    pVoiceInfo->_inParameter._pitch = pitch;
}

float GetVoicePitch(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._pitch;
}

void SetVoiceVolume(VoiceType* pVoice, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_MINMAX(volume, VoiceType::GetVolumeMin(), VoiceType::GetVolumeMax());
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    pVoiceInfo->_inParameter._volume = volume;
}

float GetVoiceVolume(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._volume;
}

void SetVoiceMixVolume(VoiceType* pVoice, FinalMixType* pFinalMix, float volume, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_UNUSED(pFinalMix);
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_MINMAX(volume, VoiceType::GetVolumeMin(), VoiceType::GetVolumeMax());
    NN_SDK_REQUIRES_RANGE(sourceIndex, 0, GetVoiceChannelCount(pVoice));
    NN_SDK_REQUIRES_RANGE(destinationIndex, 0, GetFinalMixBufferCount(pFinalMix));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);

    auto const pSourceInfo = pVoice->_pVoiceInfo;
    NN_SDK_REQUIRES_EQUAL(pSourceInfo->_inParameter._destinationMixId, pFinalMix->_pMixInfo->mixId);

    auto pVoiceChannelResource = pSourceInfo->_pVoiceChannelResources[sourceIndex];
    NN_SDK_ASSERT_NOT_NULL(pVoiceChannelResource);
    pVoiceChannelResource->SetMixVolume(volume, destinationIndex);
}

float GetVoiceMixVolume(const VoiceType* pVoice, const FinalMixType* pFinalMix, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_UNUSED(pFinalMix);
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_NOT_NULL(pFinalMix);
    NN_SDK_REQUIRES_RANGE(sourceIndex, 0, GetVoiceChannelCount(pVoice));
    NN_SDK_REQUIRES_RANGE(destinationIndex, 0, GetFinalMixBufferCount(pFinalMix));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;

    auto pVoiceChannelResource = pVoiceInfo->_pVoiceChannelResources[sourceIndex];
    NN_SDK_ASSERT_NOT_NULL(pVoiceChannelResource);
    return pVoiceChannelResource->GetMixVolume(destinationIndex);
}

void SetVoiceMixVolume(VoiceType* pSource, SubMixType* pDestination, float volume, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pSource));
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_MINMAX(volume, VoiceType::GetVolumeMin(), VoiceType::GetVolumeMax());
    NN_SDK_REQUIRES_RANGE(sourceIndex, 0, GetVoiceChannelCount(pSource));
    NN_SDK_REQUIRES_RANGE(destinationIndex, 0, GetSubMixBufferCount(pDestination));
    auto const pSourceInfo = pSource->_pVoiceInfo;
    NN_SDK_REQUIRES_EQUAL(pSourceInfo->_inParameter._destinationMixId, pDestination->_pMixInfo->mixId);
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pSource);
    NN_UNUSED(pDestination);

    auto pVoiceChannelResource = pSourceInfo->_pVoiceChannelResources[sourceIndex];
    NN_SDK_ASSERT_NOT_NULL(pVoiceChannelResource);
    pVoiceChannelResource->SetMixVolume(volume, destinationIndex);
}

float GetVoiceMixVolume(const VoiceType* pSource, const SubMixType* pDestination, int sourceIndex, int destinationIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pSource));
    NN_SDK_REQUIRES_NOT_NULL(pDestination);
    NN_SDK_REQUIRES_RANGE(sourceIndex, 0, GetVoiceChannelCount(pSource));
    NN_SDK_REQUIRES_RANGE(destinationIndex, 0, GetSubMixBufferCount(pDestination));
    auto const pSourceInfo = pSource->_pVoiceInfo;
    NN_SDK_REQUIRES_EQUAL(pSourceInfo->_inParameter._destinationMixId, pDestination->_pMixInfo->mixId);
    NN_UNUSED(pSourceInfo);
    NN_UNUSED(pDestination);

    auto pVoiceChannelResource = pSourceInfo->_pVoiceChannelResources[sourceIndex];
    NN_SDK_ASSERT_NOT_NULL(pVoiceChannelResource);
    return pVoiceChannelResource->GetMixVolume(destinationIndex);
}

void SetVoiceBiquadFilterParameter(VoiceType* pVoice, int filterIndex, const BiquadFilterParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_RANGE(filterIndex, 0, ::nn::audio::VoiceType::BiquadFilterCountMax);
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    pVoiceInfo->_inParameter._biquadFilterParameter[filterIndex] = parameter;
}

void SetVoiceBiquadFilterParameter(VoiceType* pVoice, const BiquadFilterParameter& parameter) NN_NOEXCEPT
{
    SetVoiceBiquadFilterParameter(pVoice, 0, parameter);
}

BiquadFilterParameter GetVoiceBiquadFilterParameter(const VoiceType* pVoice, int filterIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_RANGE(filterIndex, 0, ::nn::audio::VoiceType::BiquadFilterCountMax);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._biquadFilterParameter[filterIndex];
}

const BiquadFilterParameter& GetVoiceBiquadFilterParameter(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._biquadFilterParameter[0];
}

bool AppendWaveBuffer(VoiceType* pVoice, const WaveBuffer* pWaveBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_SDK_REQUIRES_NOT_NULL(pWaveBuffer);
    NN_SDK_REQUIRES_ALIGNED(reinterpret_cast<uintptr_t>(pWaveBuffer->buffer), nn::audio::BufferAlignSize);
    NN_SDK_REQUIRES(GetVoiceSampleFormat(pVoice) != nn::audio::SampleFormat_Adpcm || pWaveBuffer->pContext == nullptr || nn::util::is_aligned(reinterpret_cast<uintptr_t>(pWaveBuffer->pContext), nn::audio::BufferAlignSize));
    NN_SDK_REQUIRES(GetVoiceSampleFormat(pVoice) != nn::audio::SampleFormat_Adpcm || pWaveBuffer->pContext == nullptr || pWaveBuffer->contextSize > 0);
    NN_SDK_REQUIRES(
        (GetVoiceSampleFormat(pVoice) == nn::audio::SampleFormat_Adpcm &&
            common::CalcBufferSizeOfAdpcm(pWaveBuffer->startSampleOffset) <= pWaveBuffer->size &&
            common::CalcBufferSizeOfAdpcm(pWaveBuffer->endSampleOffset) <= pWaveBuffer->size) ||
        (GetVoiceSampleFormat(pVoice) == nn::audio::SampleFormat_PcmInt16 &&
            common::CalcBufferSizeOfPcmInt16(pWaveBuffer->startSampleOffset) <= pWaveBuffer->size / GetVoiceChannelCount(pVoice) &&
            common::CalcBufferSizeOfPcmInt16(pWaveBuffer->endSampleOffset) <= pWaveBuffer->size / GetVoiceChannelCount(pVoice)) ||
        (GetVoiceSampleFormat(pVoice) == static_cast<SampleFormat>(nn::audio::SampleFormat_Opus)),
        "startSampleOffset or endSampleOffset exceeded buffer size");
    NN_SDK_REQUIRES_GREATER_EQUAL(pWaveBuffer->startSampleOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(pWaveBuffer->endSampleOffset, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(pWaveBuffer->endSampleOffset, pWaveBuffer->startSampleOffset);
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);

    auto pVoiceInfo = pVoice->_pVoiceInfo;

    if (pVoiceInfo->_inParameter._waveBufferCount == VoiceType::WaveBufferCountMax)
    {
        return false;
    }
    auto head = pVoiceInfo->_inParameter._waveBufferHead;
    auto count = pVoiceInfo->_inParameter._waveBufferCount;
    int index = head + count < VoiceType::WaveBufferCountMax ? head + count : head + count - VoiceType::WaveBufferCountMax;

    pVoiceInfo->_inParameter._waveBuffer[index].buffer =            pWaveBuffer->buffer;
    pVoiceInfo->_inParameter._waveBuffer[index].size =              pWaveBuffer->size;
    pVoiceInfo->_inParameter._waveBuffer[index].startSampleOffset = pWaveBuffer->startSampleOffset;
    pVoiceInfo->_inParameter._waveBuffer[index].endSampleOffset =   pWaveBuffer->endSampleOffset;
    pVoiceInfo->_inParameter._waveBuffer[index].loop =              pWaveBuffer->loop;
    pVoiceInfo->_inParameter._waveBuffer[index].isEndOfStream =     pWaveBuffer->isEndOfStream;
    pVoiceInfo->_inParameter._waveBuffer[index].pContext =          pWaveBuffer->pContext;
    pVoiceInfo->_inParameter._waveBuffer[index].contextSize =       pWaveBuffer->contextSize;

    pVoiceInfo->_waveBufferAddress[index] = pWaveBuffer;
    pVoiceInfo->_inParameter._waveBuffer[index]._isSentToServer = false;
    ++pVoiceInfo->_inParameter._waveBufferCount;
    ++pVoiceInfo->waveBufferAppended;
    return true;
}

const WaveBuffer* GetReleasedWaveBuffer(VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    if (pVoiceInfo->_outStatus._waveBufferConsumed != pVoiceInfo->_waveBufferReleased)
    {
        auto head = pVoiceInfo->_inParameter._waveBufferHead;
        pVoiceInfo->_inParameter._waveBufferHead = head + 1 < VoiceType::WaveBufferCountMax ? head + 1 : 0;
        --pVoiceInfo->_inParameter._waveBufferCount;
        ++pVoiceInfo->_waveBufferReleased;
        return pVoiceInfo->_waveBufferAddress[head];
    }
    return nullptr;
}

int GetWaveBufferCount(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);

    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    const auto consumedCount = static_cast<uint8_t>(pVoiceInfo->_outStatus._waveBufferConsumed);
    const auto appendedCount = static_cast<uint8_t>(pVoiceInfo->waveBufferAppended);
    auto count = static_cast<int>(appendedCount) - static_cast<int>(consumedCount);

    if(count < 0)
    {
        count = std::numeric_limits<uint8_t>::max() - consumedCount + appendedCount + 1;
    }

    return count;
}

int64_t GetVoicePlayedSampleCount(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_outStatus._playedSampleCount;
}

NodeId GetVoiceNodeId(const VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);
    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    return pVoiceInfo->_inParameter._nodeId;
}

void FlushWaveBuffers(VoiceType* pVoice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsVoiceValid(pVoice));
    NN_AUDIO_DETAIL_VOICE_EXCLUSION_SCOPED_CHEKCER(pVoice);

    auto const pVoiceInfo = pVoice->_pVoiceInfo;
    const auto appendedWaveCount = GetWaveBufferCount(pVoice);

    pVoiceInfo->_inParameter._waveBufferFlushRequestedCount = static_cast<uint8_t>(appendedWaveCount);
}

VoiceType::BehaviorOptions GetVoiceBehaviorOptions(const VoiceType* pVoice) NN_NOEXCEPT
{
    const auto pVoiceInfo = pVoice->_pVoiceInfo;

    const VoiceType::BehaviorOptions options = {
        pVoiceInfo->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>(),
        pVoiceInfo->_inParameter._behaviorFlags.Get<VoiceBehaviorFlag::IsPitchAndSrcSkipped>(),
    };

    return options;
}

}  // namespace audio
}  // namespace nn
