﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_MemoryPool.h>

#include "audio_MemoryPoolInfo.h"
#include "audio_MemoryPoolManager.h"
#include "audio_ResourceExclusionChecker.h"

#define NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pMemoryPool) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessCheckerForMemoryPool(detail::FindResourceExclusionCheckerFromRegionInConfig(pMemoryPool->_pMemoryPoolInfo))

namespace  {
#if !defined(NN_SDK_BUILD_RELEASE)
inline bool IsMemoryPoolInitialized(const nn::audio::MemoryPoolType* pPool) NN_NOEXCEPT
{
    return ( pPool->_pMemoryPoolInfo != nullptr ) &&
           ( pPool->_pMemoryPoolInfo->GetState() != nn::audio::MemoryPoolInfo::State_Invalid );
}

inline bool IsInRange(uintptr_t address, uintptr_t start, uintptr_t end) NN_NOEXCEPT
{
    return ( (start < address) && (address < end) );
}
#endif
} // anonymous namespace

namespace nn { namespace audio {

NN_DEFINE_STATIC_CONSTANT( const size_t MemoryPoolType::AddressAlignment );
NN_DEFINE_STATIC_CONSTANT( const size_t MemoryPoolType::SizeGranularity );

bool AcquireMemoryPool(AudioRendererConfig* pConfig, MemoryPoolType* pOutPool, void* address, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutPool);
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_ALIGNED(address, nn::os::MemoryPageSize);
    NN_SDK_REQUIRES_EQUAL(size % nn::os::MemoryPageSize, 0u);

    auto manager = pConfig->_pMemoryPoolManager;
    NN_SDK_ASSERT_NOT_NULL(manager);
    NN_SDK_REQUIRES(manager->IsOverlapped(address, size) != true, "passed memory region is already used on the other MemoryPool");

    auto pInfo = manager->Acquire(address, size);
    pOutPool->_pMemoryPoolInfo = pInfo;
    NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pOutPool);

    return (pOutPool->_pMemoryPoolInfo != nullptr);
}

void ReleaseMemoryPool(AudioRendererConfig* pConfig, MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    NN_SDK_REQUIRES(IsMemoryPoolAttached(pPool) == false, "pPool must be detached");
    NN_SDK_REQUIRES(pPool->_pMemoryPoolInfo->IsInTransition() == false, "pPool must not be in state transition");
    NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pPool);

    auto manager = pConfig->_pMemoryPoolManager;
    NN_SDK_ASSERT_NOT_NULL(manager);

    manager->Release(pPool->_pMemoryPoolInfo);
    pPool->_pMemoryPoolInfo = nullptr;
}

bool RequestAttachMemoryPool(MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pPool);
    return pPool->_pMemoryPoolInfo->Attach();
}

bool RequestDetachMemoryPool(MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pPool);
    return pPool->_pMemoryPoolInfo->Detach();
}

bool IsMemoryPoolAttached(const MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pPool);
    return pPool->_pMemoryPoolInfo->IsAttached();
}

int GetReleasedMemoryPoolCount(const AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);

    auto manager = pConfig->_pMemoryPoolManager;
    NN_SDK_ASSERT_NOT_NULL(manager);
    return manager->GetFreeCount();
}

void* GetMemoryPoolAddress(const MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    return pPool->_pMemoryPoolInfo->GetAddress();
}

size_t GetMemoryPoolSize(const MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    return pPool->_pMemoryPoolInfo->GetSize();
}

void* CopyMemoryPoolData(const MemoryPoolType* pPool, void* dst, const void* src, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");

    auto poolSize = pPool->_pMemoryPoolInfo->GetSize();
    auto poolStart = reinterpret_cast<uintptr_t>(pPool->_pMemoryPoolInfo->GetAddress());
    auto poolEnd = poolStart + poolSize;
    auto dstAddr = reinterpret_cast<uintptr_t>(dst);
    auto srcAddr = reinterpret_cast<uintptr_t>(src);

    NN_SDK_REQUIRES_NOT_NULL(dst);
    NN_SDK_REQUIRES_NOT_NULL(src);
    NN_SDK_REQUIRES(
        (poolStart <= dstAddr && dstAddr + size <= poolEnd) ||
        (poolStart <= srcAddr && srcAddr + size <= poolEnd),
        "dst / src must be in pPool");
    NN_SDK_REQUIRES((IsInRange(dstAddr, srcAddr, srcAddr + size) == false) &&
                    (IsInRange(dstAddr + size, srcAddr, srcAddr + size) == false) &&
                    (IsInRange(srcAddr, dstAddr, dstAddr + size) == false) &&
                    (IsInRange(srcAddr + size, dstAddr, dstAddr + size) == false),
                  "dst & src must not be overlapped");
    NN_UNUSED(poolSize);
    NN_UNUSED(poolStart);
    NN_UNUSED(poolEnd);
    NN_UNUSED(dstAddr);
    NN_UNUSED(srcAddr);

    return memcpy(dst, src, size);
}

MemoryPoolType::State GetMemoryPoolState(const MemoryPoolType* pPool) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPool);
    NN_SDK_REQUIRES(IsMemoryPoolInitialized(pPool), "pPool must be initialized");
    NN_AUDIO_DETAIL_MEMORYPOOL_EXCLUSION_SCOPED_CHEKCER(pPool);

    auto state = pPool->_pMemoryPoolInfo->GetState();
    switch(state)
    {
    case MemoryPoolInfo::State_Acquired:
    case MemoryPoolInfo::State_Released:
    case MemoryPoolInfo::State_Detached:
        return MemoryPoolType::State_Detached;
    case MemoryPoolInfo::State_RequestAttach:
        return MemoryPoolType::State_RequestAttach;
    case MemoryPoolInfo::State_Attached:
        return MemoryPoolType::State_Attached;
    case MemoryPoolInfo::State_RequestDetach:
        return MemoryPoolType::State_RequestDetach;
    case MemoryPoolInfo::State_Invalid:
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}} // namespace nn::audio
