﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_Device.h>

namespace nn { namespace audio {

#if defined(NN_BUILD_CONFIG_OS_WIN)
namespace {

const int NameLength = 255;
const int VirtualDeviceCount = 1;

const char* AvailableVirtualDeviceNames[] =
{
    "MainAudioOut",
};

char MappedDeviceNames[VirtualDeviceCount][NameLength + 1] = { 0 };

}  // namespace
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)

Result SetAudioDeviceMapping(const char* virtualName, const AudioDeviceName* pDeviceName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(virtualName);
    NN_SDK_ASSERT_NOT_NULL(pDeviceName);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    int index = -1;
    for (auto i = 0; i < VirtualDeviceCount; ++i)
    {
        const char* availableVirtualDeviceName = AvailableVirtualDeviceNames[i];
        if (std::strncmp(virtualName, availableVirtualDeviceName, NameLength) == 0)
        {
            index = i;
            break;
        }
    }
    if (index < 0)
    {
        NN_RESULT_THROW(ResultNotFound());
    }

    std::strncpy(MappedDeviceNames[index], pDeviceName->name, NameLength);

    NN_RESULT_SUCCESS;
#else  // defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(virtualName);
    NN_UNUSED(pDeviceName);
    NN_RESULT_THROW(ResultNotSupported());
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)
}

Result SetAudioDeviceMapping(const char* virtualName, const char* physicalName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(virtualName);
    NN_SDK_ASSERT_NOT_NULL(physicalName);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    int index = -1;
    for (auto i = 0; i < VirtualDeviceCount; ++i)
    {
        const char* availableVirtualDeviceName = AvailableVirtualDeviceNames[i];
        if (std::strncmp(virtualName, availableVirtualDeviceName, NameLength) == 0)
        {
            index = i;
            break;
        }
    }
    if (index < 0)
    {
        NN_RESULT_THROW(ResultNotFound());
    }

    std::strncpy(MappedDeviceNames[index], physicalName, NameLength);

    NN_RESULT_SUCCESS;
#else  // defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(virtualName);
    NN_UNUSED(physicalName);
    NN_RESULT_THROW(ResultNotSupported());
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)
}

Result GetPhysicalDeviceName(const char** outPhysicalName, const char* virtualName) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outPhysicalName);
    NN_SDK_ASSERT_NOT_NULL(virtualName);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    for (auto i = 0; i < VirtualDeviceCount; ++i)
    {
        const char* availableVirtualDeviceName = AvailableVirtualDeviceNames[i];
        if (std::strncmp(virtualName, availableVirtualDeviceName, NameLength) == 0)
        {
            *outPhysicalName = MappedDeviceNames[i];
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(ResultNotFound());
#else  // defined(NN_BUILD_CONFIG_OS_WIN)
    NN_UNUSED(virtualName);
    *outPhysicalName = nullptr;
    NN_RESULT_SUCCESS;
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)
}

}} // namespace nn::audio
