﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // std::memcpy
#include <mutex> // std::lock_guard
#include <algorithm> // std::min

#include <nn/nn_SdkAssert.h>
#include <nn/audio/detail/audio_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BytePtr.h>
#include <nn/util/util_PlacementArray.h>

#include <nn/nn_Result.h>
#include <nn/audio/audio_Result.h>

#include "common/audio_Util.h"
#include "audio_BehaviorManager.h"

namespace nn {
namespace audio {

BehaviorManager::BehaviorManager(nn::audio::common::RevisionInfo revision) NN_NOEXCEPT
    : m_Mutex(true)
    , m_Revision(revision)
{
}

size_t BehaviorManager::UpdateInParameter(void* pInParameter) const NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto inParam = reinterpret_cast<common::BehaviorParameter::InParameter*>(pInParameter);
    inParam->_revision = m_Revision;
    inParam->_parameter = m_Parameter;

    return sizeof(common::BehaviorParameter::InParameter);
}

size_t BehaviorManager::UpdateOutStatus(const void* pOutStatus) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto outStatus = reinterpret_cast<const common::BehaviorParameter::OutStatus*>(pOutStatus);
    for (auto i = 0; i < common::BehaviorParameter::ErrorInfoCountMax; ++i)
    {
        m_ErrorInfo[i] = outStatus->_infos[i];
    }
    return sizeof(common::BehaviorParameter::OutStatus);
}

void BehaviorManager::SetMemroyPoolForceMappingEnabled(bool enabled) NN_NOEXCEPT
{
    m_Parameter[common::BehaviorParameter::FlagIndex_MemoryPoolAssert] = enabled;
}

bool BehaviorManager::GetMemroyPoolErrorAssertEnabled() const NN_NOEXCEPT
{
    return m_Parameter[common::BehaviorParameter::FlagIndex_MemoryPoolAssert];
}

nn::Result BehaviorManager::CheckErrors() NN_NOEXCEPT
{
    Result result = ResultSuccess();
    for (auto& errorInfo : m_ErrorInfo)
    {
        if (errorInfo.IsError() == false)
        {
            break;
        }

        NN_RESULT_TRY(errorInfo._result)
            // MemoryPool Error Handle
            NN_RESULT_CATCH(ResultNoMemoryPoolEntry)
            {
                static bool MemoryPoolErrorFirstTime = true;
                if (MemoryPoolErrorFirstTime)
                {
                    NN_DETAIL_AUDIO_WARN(
                    "[nn::audio] ==============================================================================\n"
                    "[nn::audio] A memory pool has not been created. Using a memory pool is required.\n"
                    "[nn::audio] Please prepare nn::audio::MemoryPoolType. See the nn::audio documents.\n"
                    "[nn::audio] ==============================================================================\n");
                    MemoryPoolErrorFirstTime = false;
                }
                NN_DETAIL_AUDIO_WARN("[nn::audio] error : %p is not contained in nn::audio::MemoryPoolType.\n", reinterpret_cast<void*>(errorInfo._info));

                if (!GetMemroyPoolErrorAssertEnabled())
                {
                    result = ResultNoMemoryPoolEntry();
                }
            }

            // General purpose error handle
            NN_RESULT_CATCH(ResultInvalidUpdateInfo)
            {
                NN_DETAIL_AUDIO_WARN("[nn::audio] warning : %p related parameter is broken\n", reinterpret_cast<void*>(errorInfo._info));
            }
        NN_RESULT_END_TRY
    }

    return result;
}

nn::audio::common::RevisionInfo BehaviorManager::GetRevision() const NN_NOEXCEPT
{
    return m_Revision;
}

} // namespace audio
} // namespace nn
