﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset

#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/util/util_BitUtil.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_MemoryHeapCommon.h>

#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_AudioOutTypes.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_VoiceTypes.h>
#include <nn/audio/audio_AudioRendererApi.h>

#include "audio_AudioRendererUtil.h"
#include "audio_BehaviorManager.h"
#include "audio_EffectManager.h"
#include "audio_MemoryPoolManager.h"
#include "audio_MixManager.h"
#include "audio_PerformanceBufferManager.h"
#include "audio_SinkManager.h"
#include "audio_SplitterInfoManager.h"
#include "audio_VoiceInfoManager.h"
#include "audio_ResourceExclusionChecker.h"
#include "common/audio_PerformanceMetricsCommon.h"
#include "common/audio_MemoryPoolPolicy.h"
#include "common/audio_NodeIdManager.h"
#include "common/audio_Util.h"

#include "common/audio_WorkBufferAllocator.h"
#include "audio_RendererInfo.h"
#include "audio_AudioCodecBufferManager.h"

namespace nn {
namespace audio {

namespace detail {

extern std::atomic<uint32_t> g_AudioRendererCount;

size_t GetConfigInParameterSize(const nn::audio::AudioRendererParameter& parameter) NN_NOEXCEPT
{
    const auto inParameterSize =
          sizeof(common::UpdateDataHeader)
        + sizeof(VoiceChannelResource::InParameter) * parameter.voiceCount
        + sizeof(VoiceInfo::InParameter) * parameter.voiceCount
        + sizeof(MixInfo) * (parameter.subMixCount + 1)
        + sizeof(EffectInfo::InParameter) * parameter.effectCount
        + sizeof(common::SinkInParameter) * parameter.sinkCount
        + sizeof(MemoryPoolInfo::InParameter) * common::GetMemoryPoolCount(detail::ConvertToAudioRendererParameterInternal(parameter))
        + sizeof(PerformanceBufferInfo::InParameter)
        + sizeof(common::BehaviorParameter::InParameter)
        + SplitterInfoManager::GetInParamterBufferSize(parameter.splitterCount, parameter.splitterSendChannelCount, parameter.subMixCount);

    return inParameterSize;
}

size_t GetConfigOutStatusSize(const nn::audio::AudioRendererParameter& parameter) NN_NOEXCEPT
{
    const auto outStatusSize =
          sizeof(common::UpdateDataHeader)
        + sizeof(VoiceInfo::OutStatus) * parameter.voiceCount
        + sizeof(EffectInfo::OutStatus) * parameter.effectCount
        + sizeof(common::SinkOutStatus) * parameter.sinkCount
        + sizeof(MemoryPoolInfo::OutStatus) * common::GetMemoryPoolCount(detail::ConvertToAudioRendererParameterInternal(parameter))
        + sizeof(PerformanceBufferInfo::OutStatus)
        + sizeof(RendererInfo)
        + sizeof(common::BehaviorParameter::OutStatus);

    return outStatusSize;
}

} // namespace detail


NN_DEFINE_STATIC_CONSTANT( const int32_t AudioRendererParameter::MixBufferCountMax );
NN_DEFINE_STATIC_CONSTANT( const int32_t AudioRendererParameter::SubMixCountMax );
NN_DEFINE_STATIC_CONSTANT( const int32_t AudioRendererParameter::VoiceCountMax );
NN_DEFINE_STATIC_CONSTANT( const int32_t AudioRendererParameter::SinkCountMax );
NN_DEFINE_STATIC_CONSTANT( const int32_t AudioRendererParameter::EffectCountMax );
NN_DEFINE_STATIC_CONSTANT( const int32_t AudioRendererParameter::PerformanceFrameCountMax );

bool IsValidMixBufferSampleCount(int sampleCount, int sampleRate) NN_NOEXCEPT
{
    return (sampleCount == 240 || sampleCount == 160) && sampleRate / 200 == sampleCount;
}

bool IsValidAudioRendererParameter(const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    // ドキュメントではボイスの個数を 1024 と即値で書いているためスタティックアサートでチェック
    NN_STATIC_ASSERT(1024 <= (1 << common::NodeIdManager::BaseWidth));

#if defined(NN_BUILD_CONFIG_OS_WIN)
    const auto IsRenderingMethodSupported = (parameter.renderingDevice == AudioRendererRenderingDevice_Cpu);
#else
    const auto IsRenderingMethodSupported = (parameter.renderingDevice == AudioRendererRenderingDevice_Cpu && parameter.executionMode == AudioRendererExecutionMode_ManualExecution)
                                         || (parameter.renderingDevice == AudioRendererRenderingDevice_AudioCoprocessor && parameter.executionMode == AudioRendererExecutionMode_AutoExecution);
#endif


    return (parameter._magic == common::GetCurrentRevision()) &&
        (parameter.sampleRate == 32000 || parameter.sampleRate == 48000) &&
        (IsValidMixBufferSampleCount(parameter.sampleCount, parameter.sampleRate)) &&
        (parameter.mixBufferCount > 0) &&
        (parameter.voiceCount > 0)     &&
        (parameter.sinkCount > 0)      &&
        (parameter.effectCount >= 0)   &&
        (parameter.subMixCount >= 0)   &&
        (parameter.mixBufferCount <= AudioRendererParameter::MixBufferCountMax) &&
        (parameter.voiceCount <= AudioRendererParameter::VoiceCountMax)         &&
        (parameter.sinkCount <= AudioRendererParameter::SinkCountMax)           &&
        (parameter.effectCount <= AudioRendererParameter::EffectCountMax)       &&
        (parameter.subMixCount <= AudioRendererParameter::SubMixCountMax)       &&
        IsRenderingMethodSupported;
}

void InitializeAudioRendererParameter(AudioRendererParameter* pOutParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParameter);
    memset(pOutParameter, 0, sizeof(AudioRendererParameter));
    pOutParameter->_magic = common::GetCurrentRevision();
    pOutParameter->sampleRate = 48000;
    pOutParameter->sampleCount = 240;
    pOutParameter->mixBufferCount = 1;
    pOutParameter->subMixCount = 0;
    pOutParameter->voiceCount = 1;
    pOutParameter->sinkCount = 1;
    pOutParameter->effectCount = 0;
    pOutParameter->performanceFrameCount = 0;
    pOutParameter->isVoiceDropEnabled = false;
    pOutParameter->splitterCount = 0;
    pOutParameter->splitterSendChannelCount = 0;
    pOutParameter->executionMode = AudioRendererExecutionMode_AutoExecution;

#if defined(NN_BUILD_CONFIG_OS_WIN)
    pOutParameter->renderingDevice = AudioRendererRenderingDevice_Cpu;
#else
    pOutParameter->renderingDevice = AudioRendererRenderingDevice_AudioCoprocessor;
#endif

    pOutParameter->_extra = 0;
}

size_t GetAudioRendererConfigWorkBufferSize(const AudioRendererParameter& parameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsValidAudioRendererParameter(parameter));

    // align up by FieldAlignSize because WorkBufferAllocator's default alignment size is it
    size_t size = 0;
    size += nn::util::align_up(sizeof(VoiceInfoManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(VoiceInfoManager::GetWorkBufferSize(parameter.voiceCount), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(EffectInfo) * parameter.effectCount, nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(EffectManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(MixInfo) * (parameter.subMixCount + 1), nn::audio::common::InfoTypeFieldAlignSize); // SubMixCount + FinalMixCount(1)
    size += nn::util::align_up(sizeof(MixManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(SinkInfo) * parameter.sinkCount, nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(SinkManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(PerformanceBufferManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(MemoryPoolManager::GetWorkBufferSize(parameter), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(BehaviorManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(SplitterInfoManager), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(SplitterInfoManager::GetSplitterInfoWorkBufferSize(parameter.splitterCount), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(SplitterInfoManager::GetSplitterDestinationWorkBufferSize(parameter.splitterSendChannelCount), nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(RendererInfo), nn::audio::common::InfoTypeFieldAlignSize);
    if (parameter._extra > 0)
    {
        size += nn::util::align_up(sizeof(ExtraManager), nn::audio::common::InfoTypeFieldAlignSize);
        size += ExtraManager::GetRequiredMemorySize(parameter._extra);
    }

    // Note: In nn::audio, a memory region used for IPC has been aligned address and size to nn::os::MemoryPageSize because of historical reasons.
    //       Before MemoryPool was introduced, audio process aborts when it try to map a memory region as device shared memory attribute if a memory region has already been mapped as IPC shared memory attribute.
    //       To avoid above case in nn::audio world, we added padding region for IPC buffer.
    //       We can remove it because MemoryPool has been introduced but we keep it because of compatibility and support cost reducing.

    // Align up for a buffer used for IPC
    size = nn::util::align_up(size, nn::os::MemoryPageSize);

    // for InParameter
    size += detail::GetConfigInParameterSize(parameter);
    size = nn::util::align_up(size, nn::audio::common::InfoTypeFieldAlignSize);

    // for OutStatus
    size += detail::GetConfigOutStatusSize(parameter);
    size = nn::util::align_up(size, nn::audio::common::InfoTypeFieldAlignSize);

    // Align up for a buffer used for IPC
    size = nn::util::align_up(size, nn::os::MemoryPageSize);

    return size;
}

void InitializeAudioRendererConfig(AudioRendererConfig* pOutConfig, const AudioRendererParameter& parameter, void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutConfig);
    NN_SDK_REQUIRES(IsValidAudioRendererParameter(parameter));
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, nn::os::MemoryPageSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, GetAudioRendererConfigWorkBufferSize(parameter));

    common::WorkBufferAllocator allocator(buffer, size);
    memset(buffer, 0, size);

    auto pVoiceInfoManager = new (allocator.Allocate(sizeof(VoiceInfoManager))) VoiceInfoManager();
    NN_ABORT_UNLESS(pVoiceInfoManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    const auto voiceInfoManagerWorkBufferSize = VoiceInfoManager::GetWorkBufferSize(parameter.voiceCount);
    auto pVoiceInfoManagerWorkBuffer = allocator.Allocate(voiceInfoManagerWorkBufferSize);
    NN_ABORT_UNLESS(pVoiceInfoManagerWorkBuffer != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    auto ei = common::PrepareArray<EffectInfo>(allocator.Allocate(sizeof(EffectInfo) * parameter.effectCount), parameter.effectCount);
    NN_ABORT_UNLESS(parameter.effectCount == 0 || ei != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");
    auto pEffectManager = new (allocator.Allocate(sizeof(EffectManager))) EffectManager(parameter.effectCount, ei);
    NN_ABORT_UNLESS(pEffectManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    const size_t mixInfosSize = sizeof(MixInfo) * (parameter.subMixCount + 1); // SubMixCount + FinalMixCount(1)
    auto pMixInfos = static_cast<MixInfo*>(allocator.Allocate(mixInfosSize));
    NN_ABORT_UNLESS(pMixInfos != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    auto pMixManager = new (allocator.Allocate(sizeof(MixManager))) MixManager(pMixInfos, parameter.subMixCount + 1, parameter.sampleRate, parameter.mixBufferCount, parameter.sampleCount);
    NN_ABORT_UNLESS(pMixManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    auto pSinkInfo = common::PrepareArray<SinkInfo>(allocator.Allocate(sizeof(SinkInfo) * parameter.sinkCount), parameter.sinkCount);
    NN_ABORT_UNLESS(pSinkInfo != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");
    auto pSinkManager = new(allocator.Allocate(sizeof(SinkManager))) SinkManager(parameter.sinkCount, pSinkInfo, parameter.sampleCount, parameter.sampleRate);
    NN_ABORT_UNLESS(pSinkManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    PerformanceBufferManager* pPerformanceHistory = new(allocator.Allocate(sizeof(PerformanceBufferManager))) PerformanceBufferManager();
    NN_ABORT_UNLESS(pPerformanceHistory != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    MemoryPoolManager* pMemoryPoolManager = MemoryPoolManager::CreateManager(allocator, parameter);
    NN_ABORT_UNLESS(pMemoryPoolManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    BehaviorManager* pConfigManager = new(allocator.Allocate(sizeof(BehaviorManager))) BehaviorManager(common::GetCurrentRevision());
    NN_ABORT_UNLESS(pConfigManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    SplitterInfoManager* pSplitterInfoManager = new(allocator.Allocate(sizeof(SplitterInfoManager))) SplitterInfoManager();
    NN_ABORT_UNLESS(pSplitterInfoManager != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");
    auto splitterInfoSize = SplitterInfoManager::GetSplitterInfoWorkBufferSize(parameter.splitterCount);
    auto pSplitterInfoBuffer = common::PrepareArrayWithIndex<SplitterInfo>(allocator.Allocate(splitterInfoSize), parameter.splitterCount);
    NN_ABORT_UNLESS(parameter.splitterCount == 0 || pSplitterInfoBuffer != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");
    auto splitterDestinationDataSize = SplitterInfoManager::GetSplitterDestinationWorkBufferSize(parameter.splitterSendChannelCount);
    auto pSplitterDestinationBuffer = common::PrepareArrayWithIndex<SplitterDestinationData>(allocator.Allocate(splitterDestinationDataSize), parameter.splitterSendChannelCount);
    NN_ABORT_UNLESS(parameter.splitterSendChannelCount == 0 || pSplitterDestinationBuffer != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");
    pSplitterInfoManager->Initialize(
        parameter.splitterCount,
        parameter.splitterSendChannelCount,
        parameter.subMixCount,
        pSplitterInfoBuffer, splitterInfoSize,
        pSplitterDestinationBuffer, splitterDestinationDataSize);

    auto pRendererInfo = reinterpret_cast<RendererInfo*>(allocator.Allocate(sizeof(RendererInfo)));
    NN_ABORT_UNLESS(pRendererInfo != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    ExtraManager* pExtraManager = nullptr;
    if (parameter._extra > 0)
    {
        pExtraManager = new (allocator.Allocate(sizeof(ExtraManager))) ExtraManager();
        auto extraSize = pExtraManager->GetRequiredMemorySize(parameter._extra);
        auto extraBuffer = allocator.Allocate(extraSize);
        pExtraManager->Initialize(extraBuffer, parameter._extra);
    }

    const auto inParameterSize = detail::GetConfigInParameterSize(parameter);
    auto pInParameter = allocator.Allocate(inParameterSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS(pInParameter != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    const auto outStatusSize = detail::GetConfigOutStatusSize(parameter);
    auto pOutStatus = allocator.Allocate(outStatusSize);
    NN_ABORT_UNLESS(pOutStatus != nullptr, "Failed to initialize AudioRendererConfig due to lack of work buffer");

    pVoiceInfoManager->Initialize(parameter.voiceCount, pVoiceInfoManagerWorkBuffer, voiceInfoManagerWorkBufferSize);
    pOutConfig->_pVoiceInfoManager= pVoiceInfoManager;
    pOutConfig->_pEffectManager = pEffectManager;
    pOutConfig->_pSinkManager = pSinkManager;
    pOutConfig->_pMixManager = pMixManager;
    pOutConfig->_pPerformanceBufferManager = pPerformanceHistory;
    pOutConfig->_pMemoryPoolManager = pMemoryPoolManager;
    pOutConfig->_pInParameter = pInParameter;
    pOutConfig->_pInParameterSize = inParameterSize;
    pOutConfig->_pOutStatus = pOutStatus;
    pOutConfig->_pOutStatusSize = outStatusSize;
    pOutConfig->_pBehaviorManager = pConfigManager;
    pOutConfig->_pSplitterInfoManager = pSplitterInfoManager;
    pOutConfig->_pRendererInfo = pRendererInfo;
    pOutConfig->_pExtraManager = pExtraManager;
    pOutConfig->_pConfigBuffer = buffer;
    pOutConfig->_configBufferSize = size;

    NN_SDK_ASSERT_EQUAL(GetAudioRendererConfigWorkBufferSize(parameter), nn::util::align_up(allocator.GetUsedSize(), nn::os::MemoryPageSize));
}

void SetMemoryPoolErrorCheckEnabled(AudioRendererConfig* pOutConfig, bool enabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutConfig);
    NN_SDK_ASSERT_NOT_NULL(pOutConfig->_pBehaviorManager);
    pOutConfig->_pBehaviorManager->SetMemroyPoolForceMappingEnabled(!enabled);
}

bool IsMemoryPoolErrorCheckEnabled(const AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pConfig->_pBehaviorManager);
    return !pConfig->_pBehaviorManager->GetMemroyPoolErrorAssertEnabled();
}

void SetAudioRendererExclusiveControlLeakageCheckEnabled(bool enabled) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(detail::g_AudioRendererCount, 0u);
    detail::SetResourceExclusionCheckEnabled(enabled);
}

bool IsAudioRendererExclusiveControlLeakageCheckEnabled() NN_NOEXCEPT
{
    return detail::IsResourceExclusionCheckEnabled();
}

int64_t GetAudioRendererElapsedFrameCount(const AudioRendererConfig* pConfig) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pConfig);
    NN_SDK_ASSERT_NOT_NULL(pConfig->_pRendererInfo);
    return pConfig->_pRendererInfo->elapsedFrameCount;
}

}  // namespace audio
}  // namespace nn

