﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/apm/apm_SystemTypes.h>

namespace nn { namespace apm { namespace server {

// オブジェクト内部の配列のインデックスに流用している為、連番定義が必須。
enum OcAlarm
{
    OcAlarm_Oc1 = 0,
    OcAlarm_Oc2 = 1,
};

const int OcAlarmMax = 2;

enum Polarity
{
    Polarity_AssertHigh,
    Polarity_AssertLow,
};

class SocthermManager
{
public:
    SocthermManager() NN_NOEXCEPT
    : m_State(State_NotInitialized)
    , m_IsThrottlingForUndockEnabled(false)
    , m_IsThrottlingForSmpdEnabled(false)
    , m_pSettingsHolder(nullptr)
    {
        for ( int index = 0; index < OcAlarmMax; index++ )
        {
            m_ThrottlingEnabled[index] = false;
            m_ThrottlingInvoked[index] = false;
            m_ThrottlingStartTick[index] = nn::os::Tick(0);
            m_LastThrottlingState[index].Clear();
            m_Polarity[index] = false;
        }
    }

    // 起動時に呼ばれる想定。
    void Initialize(SettingsHolder* pSettingsHolder) NN_NOEXCEPT;

    // 終了時に呼ばれる想定。
    void Finalize() NN_NOEXCEPT;

    // スリープ突入時に呼ばれる想定。
    void Suspend() NN_NOEXCEPT;

    // スリープ復帰時に呼ばれる想定。
    void Resume() NN_NOEXCEPT;

    // スロットリングを有効にする。
    void EnableThrottling(OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリングを無効にする。
    void DisableThrottling(OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリングの有効かを取得する。
    bool IsThrottlingEnabled(OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリング状態をクリアする。
    void ClearThrottling(OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリングの割り込みフラグの確認。
    bool IsInterruptedBy(OcAlarm ocAlarm) NN_NOEXCEPT;

    // 現在のスロットリング状態を取得する。
    void GetThrottlingState(ThrottlingState* pOutThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT;

    // SocThermManager の保持する最後のスロットリング状態を取得する。
    void GetLastThrottlingState(ThrottlingState* pOutThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT;

    // SocThermManager の保持する最後のスロットリング状態をクリアする。
    void ClearLastThrottlingState(OcAlarm ocAlarm) NN_NOEXCEPT;

    // 極性を変更する。
    void SetPolarity(OcAlarm ocAlarm, Polarity polarity) NN_NOEXCEPT;

    // 極性を取得する。
    Polarity GetPolarity(OcAlarm ocAlarm) NN_NOEXCEPT;

    // ステータスレジスタをログに出力する。
    void DumpStatus() NN_NOEXCEPT;

private:
    // スロットリングの開始を知らせスロットリング時間の取得を開始する。
    void StartThrottlingDuration(OcAlarm ocAlarm) NN_NOEXCEPT;

    // 最後のスロットリング状態を更新する。
    void UpdateLastThrottlingState(OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリングの完了を知らせスロットリング時間の取得を終了する。
    void EndThrottlingDuration(OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリング状態を取得する内部関数。
    void GetThrottlingStateInternal(ThrottlingState& outThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT;

    // スロットリング時間を取得する内部関数。
    void GetThrottlingDurationInternal(ThrottlingState& outThrottlingState, OcAlarm ocAlarm) NN_NOEXCEPT;

public:
    enum State
    {
        State_NotInitialized,
        State_Initialized,
        State_Suspended,
    };

private:
    State m_State;
    bool m_ThrottlingEnabled[OcAlarmMax];

    bool m_ThrottlingInvoked[OcAlarmMax];
    nn::os::Tick m_ThrottlingStartTick[OcAlarmMax];
    ThrottlingState m_LastThrottlingState[OcAlarmMax];

    bool m_Polarity[OcAlarmMax];

    bool m_IsThrottlingForUndockEnabled;
    bool m_IsThrottlingForSmpdEnabled;

    //! 既に有効な値を持つ SettingHolder へのポインタ。
    SettingsHolder* m_pSettingsHolder;
};

}}} // namespace nn::apm::server
