﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/apm/apm_Types.h>
#include <nn/fgm/fgm.h>

namespace nn { namespace apm { namespace server {

class RequestManager final
{
    NN_DISALLOW_COPY(RequestManager);
    NN_DISALLOW_MOVE(RequestManager);

private:
    //! CPU のデフォルトクロックレート。
    static const nn::fgm::Setting DefaultCpuSetting;

    //! GPU のデフォルトクロックレート。
    static const nn::fgm::Setting DefaultGpuSetting;

    //! EMC のデフォルトクロックレート。
    static const nn::fgm::Setting DefaultEmcSetting;

public:
    RequestManager() NN_NOEXCEPT;

    ~RequestManager() NN_NOEXCEPT;

    //! 各クロックレートのリクエストを初期化します。
    void Initialize() NN_NOEXCEPT;

    //! 各クロックレートのリクエストの終了処理をします。
    void Finalize() NN_NOEXCEPT;

    //! CPU 固定クロックレート設定を有効にします。
    void BeginCpuFixedSetting(nn::fgm::Setting setting) NN_NOEXCEPT;

    //! CPU 固定クロックレート設定を無効にします。
    void EndCpuFixedSetting() NN_NOEXCEPT;

    //! 各クロックレートのリクエストを指定された PerformanceConfiguration に対応するクロックレートで更新します。
    void Update(nn::apm::PerformanceConfiguration config) NN_NOEXCEPT;

    //! 現在のクロックレートの組み合わせでスロットリングが必要か返します。
    bool IsThrottlingNeeded(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT;

    //! 現在の CPU クロックレートの下限値を返します。
    nn::fgm::Setting GetCpuMinSetting(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT;

    //! 現在の GPU クロックレートの下限値を返します。
    nn::fgm::Setting GetGpuMinSetting(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT;

    //! 現在の EMC クロックレートの下限値を返します。
    nn::fgm::Setting GetEmcMinSetting(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT;

private:
    //! CPU のクロックレートのリクエスト。
    nn::fgm::Request m_RequestCpu;

    //! GPU のクロックレートのリクエスト。
    nn::fgm::Request m_RequestGpu;

    //! EMC のクロックレートのリクエスト。
    nn::fgm::Request m_RequestEmc;

    //! CPU 固定クロックレート設定の有効・無効フラグ。
    bool m_CpuFixedSettingEnabled;

    //! CPU 固定クロックレート設定時のクロックレート。
    nn::fgm::Setting m_CpuFixedSetting;
};

}}} // namespace nn::apm::server
