﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>

#include <nn/sf/sf_NativeHandle.h> // for nn::sf::NativeHandle
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer

#include <nn/os.h>
#include <nn/os/os_Mutex.h>

#include <nn/apm/apm.h>
#include <nn/apm/apm_IManager.sfdl.h> // for IManager
#include <nn/apm/apm_IManagerPrivileged.sfdl.h> // for IManagerPrivileged
#include <nn/apm/apm_ISession.sfdl.h> // for ISession
#include <nn/apm/apm_ISystemManager.sfdl.h> // for ISystemManager
#include <nn/apm/apm_ResultPrivate.h>
#include <nn/apm/apm_ServiceName.h> // for SfsmplCalculatorServiceName
#include <nn/apm/apm_SystemServiceName.h> // for SfsmplCalculatorServiceName

#include "apm_SessionData.h"
#include "apm_PerformanceList.h"

namespace nn { namespace apm {

namespace {

const int NumberOfManagers = 3;
const int NumberOfSessions = 1;
const int NumberOfClients = NumberOfManagers + NumberOfSessions;

sf::SimpleAllInOneHipcClientManager<NumberOfClients> g_ClientManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;

sf::ShimLibraryObjectHolder<IManager>           g_Holder           = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
sf::ShimLibraryObjectHolder<IManagerPrivileged> g_HolderPrivileged = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
sf::ShimLibraryObjectHolder<ISystemManager>     g_SystemHolder     = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

nn::sf::SharedPointer<nn::apm::ISession> g_SessionImpl = nullptr;
bool g_ManagerPrivilegedInitializedInternally = false;

nn::os::Mutex g_SessionImplMutex(false);

} // namespace

void InitializePrivileged() NN_NOEXCEPT;
void FinalizePrivileged() NN_NOEXCEPT;

Result Initialize() NN_NOEXCEPT
{
    return g_ClientManager.InitializeShimLibraryHolder(&g_Holder, nn::apm::ApmServiceName);
}

void InitializeWith(sf::SharedPointer<IManager>&& manager) NN_NOEXCEPT
{
    g_Holder.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<IManager> GetInternalManager() NN_NOEXCEPT
{
    return g_Holder.GetObject();
}

void Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_SessionImplMutex);

    if ( g_SessionImpl != nullptr )
    {
        g_SessionImpl = nullptr;
    }

    if ( g_ManagerPrivilegedInitializedInternally )
    {
        FinalizePrivileged();
        g_ManagerPrivilegedInitializedInternally = false;
    }

    g_Holder.FinalizeHolder();
}

void SetPerformanceConfiguration(PerformanceMode performanceMode, PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    PerformanceType type = CheckPerformanceList(performanceMode, performanceConfiguration);

    NN_ABORT_UNLESS((type == PerformanceType_Privileged) || (type == PerformanceType_Public),
        "Performance mode: %x, PerformanceConfiguration: %x!", performanceMode, performanceConfiguration);

    std::lock_guard<nn::os::Mutex> lock(g_SessionImplMutex);

    // 必要な時だけ非特権開発者向けのセッションを開く。
    if ( g_SessionImpl == nullptr )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder.GetObject()->OpenSession(&g_SessionImpl));
    }

    // クロックレートの設定。
    auto result = g_SessionImpl->SetPerformanceConfiguration(performanceMode, performanceConfiguration);

    if ( ResultPrivilegedConfiguration::Includes(result)
        && type == PerformanceType_Privileged && !g_ManagerPrivilegedInitializedInternally )
    {
        InitializePrivileged();
        g_ManagerPrivilegedInitializedInternally = true;

        // 非特権開発者向けのセッションの情報をバックアップするコンテナを作る。
        SessionDataContainer container;

        // 非特権開発者向けのセッションの情報をバックアップ。
        container.Backup(g_SessionImpl);

        // 非特権開発者向けのセッションの解放。
        g_SessionImpl = nullptr;

        // 特権開発者向けのセッションを開く。
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_HolderPrivileged.GetObject()->OpenSession(&g_SessionImpl));

        // 非特権開発者向けのセッションの情報を復旧。
        container.Restore(g_SessionImpl);

        // クロックレートの設定をやり直す。
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_SessionImpl->SetPerformanceConfiguration(performanceMode, performanceConfiguration));
    }
    else
    {
        // Privileged を試す条件を満たしていない場合 result を評価して終わる。
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
}

nn::apm::PerformanceConfiguration GetPerformanceConfiguration(nn::apm::PerformanceMode performanceMode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_SessionImplMutex);

    // 必要な時だけ非特権開発者向けの Session を開く。
    if ( g_SessionImpl == nullptr )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Holder.GetObject()->OpenSession(&g_SessionImpl));
    }

    nn::apm::PerformanceConfiguration performanceConfiguration;

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_SessionImpl->GetPerformanceConfiguration(&performanceConfiguration, performanceMode));

    return performanceConfiguration;
}

PerformanceMode GetPerformanceMode() NN_NOEXCEPT
{
    PerformanceMode performanceMode = PerformanceMode_Invalid;
    auto result = g_Holder.GetObject()->GetPerformanceMode(&performanceMode);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return performanceMode;
}

void InitializeForSystem() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_SystemHolder, nn::apm::ApmServiceNameForSystem));
}

void InitializeForSystemWith(sf::SharedPointer<ISystemManager>&& manager) NN_NOEXCEPT
{
    g_SystemHolder.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<ISystemManager> GetInternalManagerForSystem() NN_NOEXCEPT
{
    return g_SystemHolder.GetObject();
}

void FinalizeForSystem() NN_NOEXCEPT
{
    g_SystemHolder.FinalizeHolder();
}

void RequestPerformanceMode(PerformanceMode performanceMode) NN_NOEXCEPT
{
    auto result = g_SystemHolder.GetObject()->RequestPerformanceMode(performanceMode);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void GetPerformanceEvent(nn::os::SystemEventType* pOutSystemEvent, EventTarget eventTarget, nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutSystemEvent);

    nn::sf::NativeHandle sfHandle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_SystemHolder.GetObject()->GetPerformanceEvent(&sfHandle, eventTarget));

    nn::os::AttachReadableHandleToSystemEvent(pOutSystemEvent, sfHandle.GetOsHandle(), sfHandle.IsManaged(), clearMode);

    sfHandle.Detach();
}

void InitializePrivileged() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ClientManager.InitializeShimLibraryHolder(&g_HolderPrivileged, nn::apm::ApmServiceNamePrivileged));
}

void InitializePrivilegedWith(sf::SharedPointer<IManagerPrivileged>&& manager) NN_NOEXCEPT
{
    g_HolderPrivileged.InitializeHolderDirectly(std::move(manager));
}

nn::sf::SharedPointer<IManagerPrivileged> GetInternalManagerPrivileged() NN_NOEXCEPT
{
    return g_HolderPrivileged.GetObject();
}

void FinalizePrivileged() NN_NOEXCEPT
{
    g_HolderPrivileged.FinalizeHolder();
}

void GetThrottlingState(nn::apm::ThrottlingState* pOutThrottlingState) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutThrottlingState);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_SystemHolder.GetObject()->GetThrottlingState(pOutThrottlingState));
}

void GetLastThrottlingState(nn::apm::ThrottlingState* pOutThrottlingState) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pOutThrottlingState);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_SystemHolder.GetObject()->GetLastThrottlingState(pOutThrottlingState));
}

void ClearLastThrottlingState() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_SystemHolder.GetObject()->ClearLastThrottlingState());
}

void LoadAndApplySettings() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_SystemHolder.GetObject()->LoadAndApplySettings());
}

}} // namespace nn::apm
