﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/os/os_NativeHandle.h>

#include <nn/aoc.h>
#include <nn/aoc/aoc_SystemApi.h>
#include <nn/aoc/detail/aoc_Service.h>
#include <nn/aocsrv/detail/aocsrv_IAddOnContentManager.sfdl.h>
#include <nn/ncm/ncm_ContentMetaId.h>

#include <nn/sf/sf_ShimLibraryUtility.h>

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN)
#include <nn/aocsrv/detail/aocsrv_AddOnContentManagerImpl.h>
#endif

namespace nn { namespace aoc {
    const nn::Bit64 PseudoProcessId = 0;

    void Initialize() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN)
//      TODO: Windows 版対応時に復活させる
//        detail::InitializeForDfc();
        NN_ABORT("nn::aoc::Initialize is not implemented for windows.\n");
#elif defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
        detail::InitializeForHipc();
#else
#error("Unsuppored os type specified")
#endif
    }

    void Finalize() NN_NOEXCEPT
    {
        detail::Finalize();
    }

    int CountAddOnContent(nn::ApplicationId targetApplication) NN_NOEXCEPT
    {
        ncm::ApplicationId id = { targetApplication.value };
        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            detail::GetObjectHolder()->CountAddOnContentByApplicationId(&count, id));

        return count;
    }

    int CountAddOnContent() NN_NOEXCEPT
    {
        int count;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            detail::GetObjectHolder()->CountAddOnContent(&count, PseudoProcessId));

        return count;
    }

    int ListAddOnContent(AddOnContentIndex*  outIndices,
        int                 offset,
        int                 count,
        ApplicationId       targetApplication) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(offset >= 0);
        NN_SDK_ASSERT_NOT_NULL(outIndices);

        ncm::ApplicationId id = { targetApplication.value };
        int outCount;
        sf::OutArray<nn::aoc::AddOnContentIndex> indices(outIndices, count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            detail::GetObjectHolder()->ListAddOnContentByApplicationId(&outCount, indices, offset, count, id));

        return outCount;
    }

    int ListAddOnContent(AddOnContentIndex*  outIndices,
        int                 offset,
        int                 count) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(offset >= 0);
        NN_SDK_ASSERT_NOT_NULL(outIndices);

        int outCount;
        sf::OutArray<nn::aoc::AddOnContentIndex> indices(outIndices, count);
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            detail::GetObjectHolder()->ListAddOnContent(&outCount, indices, PseudoProcessId, offset, count));

        return outCount;
    }

    nn::Bit64 GetAddOnContentBaseId() NN_NOEXCEPT
    {
        nn::Bit64 out;
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetObjectHolder()->GetAddOnContentBaseId(&out, PseudoProcessId));

        return out;
    }

    nn::Bit64 GetAddOnContentBaseId(nn::ApplicationId targetApplication) NN_NOEXCEPT
    {
        nn::Bit64 out;
        nn::ncm::ApplicationId appId = { targetApplication.value };
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetObjectHolder()->GetAddOnContentBaseIdByApplicationId(&out, appId));

        return out;
    }

    nn::Result PrepareAddOnContent(nn::aoc::AddOnContentIndex targetIndex) NN_NOEXCEPT
    {
        return detail::GetObjectHolder()->PrepareAddOnContent(targetIndex, PseudoProcessId);
    }

    nn::Result PrepareAddOnContent(nn::aoc::AddOnContentIndex targetIndex, nn::ApplicationId targetApplication) NN_NOEXCEPT
    {
        return detail::GetObjectHolder()->PrepareAddOnContentByApplicationId(targetIndex, { targetApplication.value });
    }

    void GetAddOnContentListChangedEvent(nn::os::SystemEvent* outEvent) NN_NOEXCEPT
    {
        sf::NativeHandle handle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetObjectHolder()->GetAddOnContentListChangedEvent(sf::Out<sf::NativeHandle>(&handle)));
        outEvent->AttachReadableHandle(handle.GetOsHandle(), handle.IsManaged(), os::EventClearMode_AutoClear);
        handle.Detach();
    }

}} // namespace nn::aoc
