﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nn;
using nn.sf;
using std;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;
using Nintendo.ServiceFramework.Applet;

namespace nn.am
{
    /// <summary>
    /// アプレットリソースユーザ ID
    /// </summary>
    /// <remarks>
    /// リソースユーザに付与される ID
    /// </remarks>
    [CppRefPath(Location.CommonTypesPath)]
    [ExternalStruct(8, 8)]
    public struct AppletResourceUserId
    {
    }

    [CheckAppletResourceUserId]
    [CppRefPath(Location.CommonTypesPath)]
    [ExternalStruct(8, 8)]
    [CppFullName(@"nn::applet::AppletResourceUserId")]
    public struct CheckedAppletResourceUserId
    {
    }
}

namespace nn.am.service
{
    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface IDisplayController : IServiceObject
    {
        /// <summary>
        /// 直前アプレットの最終 FG 出画画像イメージを取得する
        /// </summary>
        /// <param name="pOutBuffer">コピー先領域</param>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレット（アプリ含む）の最終 FG->BG 遷移時の出画画像の
        /// イメージデータを指定領域にコピーします。
        /// </remarks>
        [MethodId(0)]
        Result GetLastForegroundCaptureImage(OutBuffer pOutBuffer);

        /// <summary>
        /// 直前アプレットの最終 FG 出画画像イメージを現時点の出画画像に更新する
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレット（アプリ含む）の最終 FG->BG 遷移時の出画画像を
        /// 現在出画している画像データで上書き更新します。
        /// </remarks>
        [MethodId(1)]
        Result UpdateLastForegroundCaptureImage();

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージを取得する
        /// </summary>
        /// <param name="pOutBuffer">コピー先領域</param>
        /// <returns></returns>
        /// <remarks>
        /// アプリケーションが最後に FG->BG 遷移したときの出画画像の
        /// イメージデータを指定領域にコピーします。
        /// </remarks>
        [MethodId(2)]
        Result GetLastApplicationCaptureImage(OutBuffer pOutBuffer);

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージを取得する
        /// </summary>
        /// <param name="pOutBuffer">コピー先領域</param>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元がキャプチャした画像イメージを指定領域にコピーします。
        /// </remarks>
        [MethodId(3)]
        Result GetCallerAppletCaptureImage(OutBuffer pOutBuffer);

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージを更新する
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元のキャプチャ画像イメージを
        /// 現在出画している画像データで上書き更新します。
        /// </remarks>
        [MethodId(4)]
        Result UpdateCallerAppletCaptureImage();

        /// <summary>
        /// 直前アプレットの最終 FG 出画画像イメージを取得する
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOutBuffer">コピー先領域</param>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレット（アプリ含む）の最終 FG->BG 遷移時の出画画像の
        /// イメージデータを指定領域にコピーします。
        /// </remarks>
        [MethodId(5)]
        Result GetLastForegroundCaptureImageEx(Out<bool> pOutIsScreenShotEnabled, OutBuffer pOutBuffer);

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージを取得する
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOutBuffer">コピー先領域</param>
        /// <returns></returns>
        /// <remarks>
        /// アプリケーションが最後に FG->BG 遷移したときの出画画像の
        /// イメージデータを指定領域にコピーします。
        /// </remarks>
        [MethodId(6)]
        Result GetLastApplicationCaptureImageEx(Out<bool> pOutIsScreenShotEnabled, OutBuffer pOutBuffer);

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージを取得する
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOutBuffer">コピー先領域</param>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元がキャプチャした画像イメージを指定領域にコピーします。
        /// </remarks>
        [MethodId(7)]
        Result GetCallerAppletCaptureImageEx(Out<bool> pOutIsScreenShotEnabled, OutBuffer pOutBuffer);

        /// <summary>
        /// 自アプレットの出画レイヤーのキャプチャ
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// 自アプレットで出画レイヤー画像をキャプチャし、
        /// index で指定されたキャプチャバッファに格納します。
        /// </remarks>
        [MethodId(8)]
        Result TakeScreenShotOfOwnLayer(int index, bool isScreenShotPermitted);

        /// <summary>
        /// キャプチャバッファ間のコピー
        /// </summary>
        /// <param name="dstIndex">転送先のキャプチャバッファインデックス番号</param>
        /// <param name="srcIndex">転送元のキャプチャバッファインデックス番号</param>
        /// <returns></returns>
        /// <remarks></remarks>
        [MethodId(9)]
        Result CopyBetweenCaptureBuffers(int dstIndex, int srcIndex);

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージの TransferMemory 要求
        /// </summary>
        /// <param name="pOut">TransferMemory のハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// アプリケーションが最後に FG->BG 遷移したときの出画画像の
        /// イメージデータを TransferMemory 化したハンドルを取得します。
        /// </remarks>
        [MethodId(10)]
        Result AcquireLastApplicationCaptureBuffer(Out<NativeHandle> pOut);

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージの TransferMemory 解除
        /// </summary>
        /// <param></param>
        /// <returns></returns>
        /// <remarks>
        /// アプリケーションが最後に FG->BG 遷移したときの出画画像を
        /// クライアント上からアンマップしたことを通知します。
        /// </remarks>
        [MethodId(11)]
        Result ReleaseLastApplicationCaptureBuffer();

        /// <summary>
        /// 直前アプレットの最終出画画像イメージの TransferMemory 要求
        /// </summary>
        /// <param name="pOut">TransferMemory のハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレットの最終 FG->BG 遷移時の出画画像の
        /// イメージデータを TransferMemory 化したハンドルを取得します。
        /// </remarks>
        [MethodId(12)]
        Result AcquireLastForegroundCaptureBuffer(Out<NativeHandle> pOut);

        /// <summary>
        /// 直前アプレットの最終出画画像イメージの TransferMemory 解除
        /// </summary>
        /// <param></param>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレットの最終 FG->BG 遷移時の出画画像を
        /// クライアント上からアンマップしたことを通知します。
        /// </remarks>
        [MethodId(13)]
        Result ReleaseLastForegroundCaptureBuffer();

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージの TransferMemory 要求
        /// </summary>
        /// <param name="pOut">TransferMemory のハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元でのキャプチャ画像のメモリを
        /// TransferMemory 化したハンドルを取得します。
        /// </remarks>
        [MethodId(14)]
        Result AcquireCallerAppletCaptureBuffer(Out<NativeHandle> pOut);

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージの TransferMemory 解除
        /// </summary>
        /// <param></param>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元でのキャプチャ画像のメモリを
        /// クライアント上からアンマップしたことを通知します。
        /// </remarks>
        [MethodId(15)]
        Result ReleaseCallerAppletCaptureBuffer();

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージの TransferMemory 要求
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOut">TransferMemory のハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// アプリケーションが最後に FG->BG 遷移したときの出画画像の
        /// イメージデータを TransferMemory 化したハンドルを取得します。
        /// </remarks>
        [MethodId(16)]
        Result AcquireLastApplicationCaptureBufferEx(Out<bool> pOutIsScreenShotEnabled, Out<NativeHandle> pOut);

        /// <summary>
        /// 直前アプレットの最終出画画像イメージの TransferMemory 要求
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOut">TransferMemory のハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレットの最終 FG->BG 遷移時の出画画像の
        /// イメージデータを TransferMemory 化したハンドルを取得します。
        /// </remarks>
        [MethodId(17)]
        Result AcquireLastForegroundCaptureBufferEx(Out<bool> pOutIsScreenShotEnabled, Out<NativeHandle> pOut);

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージの TransferMemory 要求
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOut">TransferMemory のハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元でのキャプチャ画像のメモリを
        /// TransferMemory 化したハンドルを取得します。
        /// </remarks>
        [MethodId(18)]
        Result AcquireCallerAppletCaptureBufferEx(Out<bool> pOutIsScreenShotEnabled, Out<NativeHandle> pOut);

        /// <summary>
        /// 指定されたキャプチャバッファの塗り潰し
        /// </summary>
        /// <param name="index">対象のキャプチャバッファのインデックス</param>
        /// <param name="isScreenShotEnabled">キャプチャ撮影許可フラグ</param>
        /// <param name="clearColor">塗り潰しするカラー値</param>
        /// <returns></returns>
        /// <remarks>
        /// index で指定されたキャプチャバッファを clearColor で塗り潰します。
        /// </remarks>
        [MethodId(20)]
        Result ClearCaptureBuffer(int index, bool isScreenShotPermitted, Bit32 clearColor);

        /// <summary>
        /// AmLayer 用のテクスチャバッファの塗り潰し
        /// </summary>
        /// <param name="clearColor">塗り潰しするカラー値</param>
        /// <returns></returns>
        /// <remarks>
        /// AmLayer 用のテクスチャバッファを clearColor で塗り潰します。
        /// </remarks>
        [MethodId(21)]
        Result ClearAppletTransitionBuffer(Bit32 clearColor);

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージの共有バッファの使用権要求
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOut">共有バッファインデックスのハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// アプリケーションが最後に FG->BG 遷移したときの出画画像の
        /// イメージデータを保持している共有バッファのインデックスを取得します。
        /// </remarks>
        [MethodId(22)]
        Result AcquireLastApplicationCaptureSharedBuffer(Out<bool> pOutIsScreenShotEnabled, Out<int> pOut);

        /// <summary>
        /// アプリケーションの最終 FG 出画画像イメージの共有バッファの使用権解放
        /// </summary>
        [MethodId(23)]
        Result ReleaseLastApplicationCaptureSharedBuffer();

        /// <summary>
        /// 直前アプレットの最終出画画像イメージの共有バッファの使用権要求
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOut">共有バッファインデックスのハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// 直前アプレットの最終 FG->BG 遷移時の出画画像の
        /// イメージデータを保持している共有バッファのインデックスを取得します。
        /// </remarks>
        [MethodId(24)]
        Result AcquireLastForegroundCaptureSharedBuffer(Out<bool> pOutIsScreenShotEnabled, Out<int> pOut);

        /// <summary>
        /// 直前アプレットの最終出画画像イメージの共有バッファの使用権解放
        /// </summary>
        [MethodId(25)]
        Result ReleaseLastForegroundCaptureSharedBuffer();

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージの共有バッファの使用権要求
        /// </summary>
        /// <param name="pOutIsScreenShotEnabled">キャプチャ撮影許可状態</param>
        /// <param name="pOut">共有バッファインデックスのハンドル書込先</param>
        /// <returns></returns>
        /// <remarks>
        /// LA 呼出元でのキャプチャ画像のメモリを
        /// イメージデータを保持している共有バッファのインデックスを取得します。
        /// </remarks>
        [MethodId(26)]
        Result AcquireCallerAppletCaptureSharedBuffer(Out<bool> pOutIsScreenShotEnabled, Out<int> pOut);

        /// <summary>
        /// LA 呼出元のキャプチャ画像イメージの共有バッファの使用権解放
        /// </summary>
        [MethodId(27)]
        Result ReleaseCallerAppletCaptureSharedBuffer();

        /// <summary>
        /// 自アプレットの出画レイヤーのキャプチャ
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// 自アプレットで出画レイヤー画像をキャプチャし、
        /// index で指定されたキャプチャバッファに格納します。
        /// </remarks>
        [MethodId(28)]
        Result TakeScreenShotOfOwnLayerEx(int index, bool isScreenShotPermitted, bool isImmediate);

    }

    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface IAudioController : IServiceObject
    {
        /// <summary>
        /// メインアプレットとしての期待値音量を設定する
        /// </summary>
        /// <param name="mainAppletVolume">メインアプレット音量</param>
        /// <param name="libraryAppletVolume">ライブラリアプレット音量</param>
        /// <returns></returns>
        /// <remarks>
        /// MA->LA 起動時に MA が期待する、MA 音量と LA 音量の期待値を設定する。
        /// この期待値パラメータは、単に AM 内で管理されるのみで、
        /// 本メソッドによって何かの音量が実際に変更されるわけではありません。
        /// </remarks>
        [MethodId(0)]
        Result SetExpectedMasterVolume(float mainAppletVolume, float libraryAppletVolume);

        /// <summary>
        /// メインアプレットが期待するメインアプレットの期待値音量を取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        /// <remarks>
        /// MA が設定した MA 音量期待値を取得します。
        /// 本メソッドは MA および LA から呼ばれる想定です。
        /// </remarks>
        [MethodId(1)]
        Result GetMainAppletExpectedMasterVolume(Out<float> pOutVolume);

        /// <summary>
        /// メインアプレットが期待するライブラリアプレットの期待値音量を取得する
        /// </summary>
        /// <param name="pOut"></param>
        /// <returns></returns>
        /// <remarks>
        /// MA が設定した LA 音量期待値を取得します。
        /// 本メソッドは MA および LA から呼ばれる想定です。
        /// </remarks>
        [MethodId(2)]
        Result GetLibraryAppletExpectedMasterVolume(Out<float> pOutVolume);

        /// <summary>
        /// メインアプレットのマスター音量を変更する
        /// </summary>
        /// <param name="volume">ボリューム値</param>
        /// <param name="fadeTime">フェード時間</param>
        /// <returns></returns>
        /// <remarks>
        /// MA のマスター音量を LA から設定するためのものです。
        /// </remarks>
        [MethodId(3)]
        Result ChangeMainAppletMasterVolume(float volume, int64_t fadeTimeInNanoSeconds);

        /// <summary>
        /// 自アプレットウィンドウより下位のボリューム透過率を指定する
        /// </summary>
        /// <param name="transparentRate">透過率</param>
        /// <returns></returns>
        /// <remarks>
        /// OA が他の全アプレットの音量を絞るためのものですが、
        /// 機能的にはどのアプレットからも自身以降のウィンドウに適用可能です。
        /// </remarks>
        [MethodId(4)]
        Result SetTransparentVolumeRate(float transparentRate);
    }

    [CppRefPath(Location.FunctionsSfdlPath)]
    public struct WindowCreationOption
    {
        public int reserved;
    };

    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface IWindowController : IServiceObject
    {
        [MethodId(0)]
        Result CreateWindow(Out<IWindow> pOut, WindowCreationOption option);

        /// <summary>
        /// アプレットのリソースユーザに付与される ID を取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// 各アプレットに付与される AppletResourceUserId を返します。
        /// </remarks>
        [MethodId(1)]
        Result GetAppletResourceUserId(Out<nn.am.AppletResourceUserId> pOut);

        /// <summary>
        /// 呼出元アプレットのリソースユーザ ID を取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// 呼出元アプレットに付与されている AppletResourceUserId を返します。
        /// </remarks>
        [MethodId(2)]
        Result GetAppletResourceUserIdOfCallerApplet(Out<nn.am.AppletResourceUserId> pOut);

        /// <summary>
        /// FG 権限を取得する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// TODO: 以前は IForegroundAccessor を生成していた
        ///
        /// Message_ChangeIntoForeground 通知に対して、FG 権限を取得します。
        /// 現状では本メソッドの shim 側で nn::vi::SetLayerVisibility(true) を
        /// 行ないます。
        /// </remarks>
        [MethodId(10)]
        Result AcquireForegroundRights();

        /// <summary>
        /// FG 権限を開放する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// TODO: 以前は IForegroundAccessor を生成していた
        ///
        /// Message_ChangeIntoBackground 通知に対して、FG 権限を開放します。
        /// 現状では本メソッドの shim 側で nn::vi::SetLayerVisibility(false) を
        /// 行ないます。
        /// </remarks>
        [MethodId(11)]
        Result ReleaseForegroundRights();

        /// <summary>
        /// BG への遷移要求を拒否する
        /// </summary>
        /// <returns>
        /// </returns>
        /// <remarks>
        /// Message_ChangeIntoBackground 通知に対して、BG 遷移を拒絶します。
        /// 通常は RequestToGetForeground() を発行した SA に対して、
        /// Message_RejectedChangedIntoBackground を通知します。
        /// </remarks>
        [MethodId(12)]
        Result RejectToChangeIntoBackground();
    }

    [CppRefPath(Location.FunctionsSfdlPath)]
    public interface IWindow : IServiceObject
    {
    }
}
