﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/process/am_WindableApplet.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <utility>
#include <nn/util/util_ScopeExit.h>
#include <mutex>
#include <nn/am/service/am_ServiceStaticAllocator.h>
#include <nn/am/am_ResultPrivate.h>

namespace nn { namespace am { namespace service { namespace process {

class WindableApplet::CleanupLockImpl
{
public:
    explicit CleanupLockImpl(WindableApplet* p)
        : m_P(p)
    {
    }
    ~CleanupLockImpl() NN_NOEXCEPT
    {
        // すべて解放されたら Cleanup を走らせるためにシグナルする
        m_P->m_CleaupEvent.Signal();
    }
private:
    WindableApplet* m_P;
};

WindableApplet::CleanupLock WindableApplet::AcquireCleanupLock() NN_NOEXCEPT
{
    return m_WeakCleanupLock.lock();
}

bool WindableApplet::IsLiving() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    return m_Living;
}

#define NN_AM_CHECK_LIVING NN_RESULT_THROW_UNLESS(IsLiving(), ::nn::am::ResultAppletTerminatedInternal())

Result WindableApplet::Run() NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        // このスレッド上で、子の終了を待つ
        this->CleanAllChildren();
    };
    NN_AM_CHECK_LIVING;
    NN_AM_SERVICE_APPLET_LOG(call, this, "begin Before()");
    NN_RESULT_DO(this->Before());
    NN_AM_SERVICE_APPLET_LOG(call, this, "end Before()");
    auto cleanupLock = MakeShared<CleanupLockImpl>(this);
    this->m_WeakCleanupLock = cleanupLock;
    NN_UTIL_SCOPE_EXIT
    {
        // Before が成功した場合には、必ず Cleanup を実行する。
        // cleanup ロックの解放を待ってから
        cleanupLock.reset();
        NN_AM_SERVICE_APPLET_LOG(seq, this, "wait for cleanup lock");
        this->m_CleaupEvent.Wait();
        NN_AM_SERVICE_APPLET_LOG(call, this, "begin Cleanup()");
        this->Cleanup();
        NN_AM_SERVICE_APPLET_LOG(call, this, "end Cleanup()");
        NN_AM_SERVICE_APPLET_LOG(seq, this, "Cleanup()");
    };
    auto doContinue = true;
    while (doContinue)
    {
        // プロセスの起動と待機
        {
            // プロセスの作成
            // TODO: ns が対応したら、Create と Start を分離する
            std::shared_ptr<NsProcess> pProcess;
            NN_RESULT_DO(this->CreateProcess(&pProcess));
            {
                // プロセス起動中は Terminate から殺せるようにメンバ変数に登録
                std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
                if (!m_Living)
                {
                    pProcess->Terminate();
                    pProcess->Join();
                    break;
                }
                this->m_CurrentProcess = pProcess;
            }
            NN_UTIL_SCOPE_EXIT
            {
                // Wait 後に登録解除
                std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
                this->m_CurrentProcess.reset();
            };

            // プロセスごとの情報の設定
            // プロセス ID とプロキシのマップの登録を想定
            auto processId = pProcess->GetProcessId();
            NN_AM_SERVICE_APPLET_LOG(seq, this, "process created: id = %lld", processId);
            NN_AM_SERVICE_APPLET_LOG(call, this, "begin OnProcessBegin()");
            this->OnProcessBegin(processId);
            NN_AM_SERVICE_APPLET_LOG(call, this, "end OnProcessBegin()");
            NN_UTIL_SCOPE_EXIT
            {
                NN_AM_SERVICE_APPLET_LOG(call, this, "begin OnProcessEnd()");
                this->OnProcessEnd(processId);
                NN_AM_SERVICE_APPLET_LOG(call, this, "end OnProcessEnd()");
            };

            // このスレッド上で、プロセスと並行して何か処理をする場合には OnProcessExecuting をオーバライドする。
            // ただし、プロセスが終了したら抜けること。
            this->RunBehindProcess(pProcess.get());

            // 終了を待機
            NN_AM_SERVICE_APPLET_LOG(call, this, "begin Join()");
            pProcess->Join();
            NN_AM_SERVICE_APPLET_LOG(call, this, "end Join()");
            NN_AM_SERVICE_APPLET_LOG(seq, this, "process joined: id = %lld", processId);
        }
        NN_AM_CHECK_LIVING;
        // doReservation が true を返した場合には、プロセスを再起動する
        NN_AM_SERVICE_APPLET_LOG(seq, this, "reservation start");
        NN_AM_SERVICE_APPLET_LOG(call, this, "begin DoReservation()");
        doContinue = this->DoReservation();
        NN_AM_SERVICE_APPLET_LOG(call, this, "end DoReservation()");
        NN_AM_SERVICE_APPLET_LOG(seq, this, "reservation end -> repeat: %d", static_cast<int>(doContinue));
    }
    NN_AM_CHECK_LIVING;
    NN_AM_SERVICE_APPLET_LOG(call, this, "begin After()");
    NN_RESULT_DO(this->After());
    NN_AM_SERVICE_APPLET_LOG(call, this, "end After()");
    NN_RESULT_SUCCESS;
}

void WindableApplet::KillProcessImpl(std::unique_lock<decltype(m_Mutex)> lk) NN_NOEXCEPT
{
    NN_SDK_ASSERT(lk.owns_lock());
    NN_SDK_ASSERT(lk.mutex() == &m_Mutex);
    // ロック区間の最小化のために一時変数に退避し、ロック解除
    auto pCurrentProcess = std::move(m_CurrentProcess);
    lk.unlock();
    NN_AM_SERVICE_APPLET_LOG(call, this, "WindableApplet::KillProcessImpl() pCurrentProcess=0x%p", pCurrentProcess.get());
    if (pCurrentProcess)
    {
        pCurrentProcess->Terminate();
    }
}

void WindableApplet::TerminateOne() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_Living)
    {
        return;
    }
    this->m_Living = false;
    KillProcessImpl(std::move(lk));
}

Result WindableApplet::Before() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

Result WindableApplet::After() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

void WindableApplet::Cleanup() NN_NOEXCEPT
{
    // nop
}

void WindableApplet::OnProcessBegin(os::ProcessId) NN_NOEXCEPT
{
    // nop
}

void WindableApplet::OnProcessEnd(os::ProcessId) NN_NOEXCEPT
{
    // nop
}

void WindableApplet::RunBehindProcess(NsProcess* pProcess) NN_NOEXCEPT
{
    NN_UNUSED(pProcess);
}

bool WindableApplet::DoReservation() NN_NOEXCEPT
{
    return false;
}

Result WindableApplet::KillProcess() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    KillProcessImpl(std::move(lk));
    NN_RESULT_SUCCESS;
}

os::ProcessId WindableApplet::GetCurrentProcessId() const NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_CurrentProcess)
    {
        return os::ProcessId::GetInvalidId();
    }
    return m_CurrentProcess->GetProcessId();
}

}}}}
