﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_AppletPresentationManager.h>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_IntegratedDisplayManager.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>
#include <nn/am/service/display/am_ScreenShotOperation.h>
#include <nn/am/service/display/am_DisplayCaptureBufferOperation.h>

namespace nn{ namespace am{ namespace service{ namespace display{ namespace detail{


    void AppletPresentationManager::Initialize(const AppletPresentationResourceAccessor& resAccessor) NN_NOEXCEPT
    {
        NN_STATIC_ASSERT(sizeof(m_ResourceAccessorStorage) >= sizeof(AppletPresentationResourceAccessor));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_ResourceAccessorStorage) >= NN_ALIGNOF(AppletPresentationResourceAccessor));
        m_pResource = new(m_ResourceAccessorStorage) AppletPresentationResourceAccessor(resAccessor);
    }

    void AppletPresentationManager::Finalize() NN_NOEXCEPT
    {
        m_pResource->~AppletPresentationResourceAccessor();
        m_pResource = nullptr;
        std::memset(m_ResourceAccessorStorage, 0, sizeof(m_ResourceAccessorStorage));
    }

    void AppletPresentationManager::Suspend() NN_NOEXCEPT
    {
        if(m_State == State_Suspended)
        {
            NN_AM_DISPCTRL_LOG_TRACE("applet presentation is already suspended\n");
            return;
        }
        NN_AM_DISPCTRL_LOG_TRACE("suspending applet presentation...\n");

        // 現在のアクティブレイヤを退避
        m_SuspendState.m_SuspendedFullscreenLayerHandle = m_CurrentActiveFullscreenLayerHandle;
        m_SuspendState.m_SuspendedPartialLayerHandle    = m_CurrentActivePartialLayerHandle;
        m_SuspendState.m_SuspendedOverlayLayerHandle    = m_CurrentActiveOverlayLayerHandle;

        // 全レイヤを Deactivate
        DeactivateCurrentActiveFullscreenLayerImpl(false);
        DeactivateCurrentPartialLayerImpl();
        DeactivateCurrentOverlayLayerImpl();

        m_State = State_Suspended;
        NN_AM_DISPCTRL_LOG_TRACE("suspended applet presentation\n");
    }

    void AppletPresentationManager::Resume() NN_NOEXCEPT
    {
        if(m_State == State_Running)
        {
            NN_AM_DISPCTRL_LOG_TRACE("applet presentation is already running\n");
            return;
        }
        NN_AM_DISPCTRL_LOG_TRACE("resumeing applet presentation...\n");

        // TORIAEZU:
        //   正式には退避していたレイヤを戻す？
        m_SuspendState.m_SuspendedFullscreenLayerHandle = ActiveLayerNone;
        m_SuspendState.m_SuspendedPartialLayerHandle = ActiveLayerNone;
        m_SuspendState.m_SuspendedOverlayLayerHandle = ActiveLayerNone;

        // 最新の状態に更新
        UpdateActiveLayerImpl();

        m_State = State_Running;
        NN_AM_DISPCTRL_LOG_TRACE("resumed applet presentation\n");
    }

    void AppletPresentationManager::UpdateActiveLayer() NN_NOEXCEPT
    {
        if(m_State != State_Running)
        {
            NN_AM_DISPCTRL_LOG_TRACE("skipping updating active layer(suspended)\n");
            return;
        }

        UpdateActiveLayerImpl();
    }

    void AppletPresentationManager::UpdateActiveLayerImpl() NN_NOEXCEPT
    {
        // 最前面のレイヤを計算
        IntegratedDisplayLayerHandle hNewActiveFullscreen = ActiveLayerNone;
        IntegratedDisplayLayerHandle hNewActivePartial    = ActiveLayerNone;
        IntegratedDisplayLayerHandle hNewActiveOverlay    = ActiveLayerNone;
        CalculateNewActiveLayerImpl(
            &hNewActiveFullscreen,
            &hNewActivePartial,
            &hNewActiveOverlay,
            m_pResource->GetAppletLayerPool()
        );

        // 全画面のレイヤが変わっていれば変更
        if(hNewActiveFullscreen != m_CurrentActiveFullscreenLayerHandle)
        {
            NN_AM_DISPCTRL_LOG_TRACE("changing active fullscreen layer: h=%lld -> h=%lld\n", m_CurrentActiveFullscreenLayerHandle, hNewActiveFullscreen);
            DeactivateCurrentActiveFullscreenLayerImpl(true);
            ActivateFullscreenLayerImpl(hNewActiveFullscreen);
        }

        // 部分表示のレイヤが変わっていれば変更
        if(hNewActivePartial != m_CurrentActivePartialLayerHandle)
        {
            NN_AM_DISPCTRL_LOG_TRACE("changing active partial layer: h=%lld -> h=%lld\n", m_CurrentActivePartialLayerHandle, hNewActivePartial);
            DeactivateCurrentPartialLayerImpl();
            ActivatePartialLayerImpl(hNewActivePartial);
        }

        // オーバーレイが変わっていれば変更
        if(hNewActiveOverlay != m_CurrentActiveOverlayLayerHandle)
        {
            NN_AM_DISPCTRL_LOG_TRACE("changing active overlay layer: h=%lld -> h=%lld (aruid=%llu)\n", m_CurrentActiveOverlayLayerHandle, hNewActiveOverlay);
            DeactivateCurrentOverlayLayerImpl();
            ActivateOverlayLayerImpl(hNewActiveOverlay);
        }
    }

    //--------------------------------------------------------------

    void AppletPresentationManager::CalculateNewActiveLayerImpl(
        IntegratedDisplayLayerHandle* pOutNewActiveFullscreenLayerHandle,
        IntegratedDisplayLayerHandle* pOutNewActivePartialLayerHandle,
        IntegratedDisplayLayerHandle* pOutNewActiveOverlayLayerHandle,
        const IntegratedDisplayLayerPool* pLayerPool
    ) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("calculating new active layer\n");

        struct NewActiveLayer
        {
            IntegratedDisplayLayerHandle handle;
            int32_t zOrder;
            applet::AppletResourceUserId aruid;
            const char* name;
        };

        NewActiveLayer newActiveFullscreen = { ActiveLayerNone, -1, applet::AppletResourceUserId::GetInvalidId(), "ful" };
        NewActiveLayer newActivePartial    = { ActiveLayerNone, -1, applet::AppletResourceUserId::GetInvalidId(), "par" };
        NewActiveLayer newActiveOverlay    = { ActiveLayerNone, -1, applet::AppletResourceUserId::GetInvalidId(), "ovl" };

        for(int i = 0; i < IntegratedDisplayLayerPool::LayerCount; i++)
        {
            auto pLayer = pLayerPool->GetLayerAt(i);

            // 未使用・未確定のレイヤは無視
            if(pLayer == nullptr || pLayer->GetState() != detail::IntegratedDisplayLayerState_Determined)
            {
                //NN_AM_DISPCTRL_LOG_TRACE("  skip: %d (not-determined)\n", i);
                continue;
            }

            // ディスプレイなしは無視
            if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_NoDisplay)
            {
                NN_AM_DISPCTRL_LOG_TRACE("  skip: %d (no-display)\n", i);
                continue;
            }

            // 不可視レイヤは無視
            if(!pLayer->IsVisible())
            {
                NN_AM_DISPCTRL_LOG_TRACE("  skip: %d (invisible)\n", i);
                continue;
            }

            NewActiveLayer* pInfo = nullptr;
            if(pLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_IndirectLayer)
            {
                pInfo = &newActivePartial;
            }
            else if(pLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer)
            {
                if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application
                    || pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_FullscreenApplet)
                {
                    pInfo = &newActiveFullscreen;
                }
                else if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_PartialApplet)
                {
                    pInfo = &newActivePartial;
                }
                else if(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Overlay)
                {
                    pInfo = &newActiveOverlay;
                }
            }

            // 決定できなければ無視
            if(pInfo == nullptr)
            {
                NN_AM_DISPCTRL_LOG_TRACE("  skip: %d (invalid config)\n", i);
                continue;
            }

            // 手前になければ不採用
            if(pLayer->GetZOrder() <= pInfo->zOrder)
            {
                NN_AM_DISPCTRL_LOG_TRACE("  skip: %d (not-top)\n", i);
                continue;
            }

            NN_AM_DISPCTRL_LOG_TRACE("  candidate: %s -> %d\n", pInfo->name, i);
            pInfo->handle = i;
            pInfo->zOrder = pLayer->GetZOrder();
            pInfo->aruid  = pLayer->GetAruid();
        }

        *pOutNewActiveFullscreenLayerHandle = newActiveFullscreen.handle;
        *pOutNewActivePartialLayerHandle    = newActivePartial.handle;
        *pOutNewActiveOverlayLayerHandle    = newActiveOverlay.handle;
    }

    //--------------------------------------------------------------
    // Fullscreen/Deactivate
    //--------------------------------------------------------------

    void AppletPresentationManager::DeactivateCurrentActiveFullscreenLayerImpl(bool isUpdateCaptureBufferEnabled) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("deactivating fullscreen h=%lld\n", m_CurrentActiveFullscreenLayerHandle);

        // 未完了の画面写真撮影をキャンセル
        //   権利表記を隠す前にキャンセルすること
        ScreenShotOperation::CancelTakingScreenShotAsNoTarget(nn::os::GetSystemTick());

        // 遷移レイヤまたはバッファなしなら何もせず終わり
        if(m_CurrentActiveFullscreenLayerHandle < 0)
        {
            NN_AM_DISPCTRL_LOG_TRACE("  skip (no layer is active)\n");
            return;
        }

        IntegratedDisplayLayerHandle hOldLayer = m_CurrentActiveFullscreenLayerHandle;
        auto pLayer = m_pResource->GetAppletLayerPool()->GetLayer(hOldLayer);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        // 共通: Conductor 指定を解除
        if(auto layerId = GetFullscreenLayerIdImpl(hOldLayer))
        {
            DisplayPrimitiveOperation::SetIsConductorLayer(layerId, false);
        }

        // Deactivate を実行
        IntegratedDisplayLayerHandle hNewLayer = m_CurrentActiveFullscreenLayerHandle;
        switch(pLayer->GetBufferOwner())
        {
        case detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet:
            {
                hNewLayer= DeactivateCurrentActiveFullscreenDirectLayerImpl(hOldLayer, isUpdateCaptureBufferEnabled);
                break;
            }
        case detail::IntegratedDisplayLayerBufferOwner_SystemShared:
            {
                hNewLayer = DeactivateCurrentActiveFullscreenSharedLayerImpl(hOldLayer, isUpdateCaptureBufferEnabled);
                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }

        // 共通: キャプチャリクエストを解除
        if(isUpdateCaptureBufferEnabled)
        {
            pLayer->SetCallerAppletCaptureRequested(false, false);
        }

        // 共通: 権利表記を隠す
        //   画面写真撮影はキャンセルされているので権利表記なしでアプリの絵が保存されることはない
        HideApplicationCopyrightLayerImpl();

        m_CurrentActiveFullscreenLayerHandle = hNewLayer;
    }

    IntegratedDisplayLayerHandle AppletPresentationManager::DeactivateCurrentActiveFullscreenDirectLayerImpl(
        IntegratedDisplayLayerHandle hOldActiveLayer,
        bool isUpdateCaptureBufferEnabled
    ) NN_NOEXCEPT
    {
        auto pOldLayer  = m_pResource->GetAppletLayerPool()->GetLayer(hOldActiveLayer);
        auto pOldVisMgr = m_pResource->GetAppletLayerPool()->GetVisibilityManager(hOldActiveLayer);
        NN_SDK_REQUIRES_NOT_NULL(pOldLayer);
        NN_SDK_REQUIRES_NOT_NULL(pOldVisMgr);

        NN_SDK_REQUIRES(
            pOldLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application ||
            pOldLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_FullscreenApplet
        );
        NN_SDK_REQUIRES(
            pOldLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet
        );
        NN_SDK_REQUIRES(
            pOldLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer
        );

        auto layerId = pOldLayer->GetDirectLayerId();
        NN_SDK_ASSERT_NOT_EQUAL(layerId, 0u);

        // 撮影対象なら LastFg に反映
        bool isLastFgUpdated = false;
        if(isUpdateCaptureBufferEnabled && pOldLayer->IsCaptureTarget())
        {
            bool isScreenshotPermitted = pOldLayer->InvokeScreenShotPermissionGetterFunction();
            bool isCopyrightRequired
                = (pOldLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
                ? m_pResource->GetCurrentApplicationCopyrightVisibility()
                : false;

            DisplayCaptureBufferOperation::CaptureDisplay(
                CaptureBufferIndex_LastForeground, nn::vi::LayerStack_LastFrame, isScreenshotPermitted, isCopyrightRequired, m_pResource->GetParameter().captureBufferTimeout
            );
            isLastFgUpdated = true;

            // アプリなら必ず LastApp にコピーする
            if(pOldLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
            {
                DisplayCaptureBufferOperation::CopyBuffer(CaptureBufferIndex_LastApplication, CaptureBufferIndex_LastForeground, m_pResource->GetParameter().captureBufferTimeout);
            }

            // NOTE:
            //   自前バッファを使っている場合、 CallerApplet へのキャプチャは要求された時点で実行されるので
            //   Deactivate 時には何もしない。
        }

        // 遷移レイヤが必要なら出す
        IntegratedDisplayLayerHandle hNewLayer;
        if(pOldLayer->IsTransitionLayerRequested() && isLastFgUpdated)
        {
            PresentTransitionBufferImpl();
            hNewLayer = ActiveLayerTransition;
        }
        else
        {
            BlankSystemSharedLayerImpl();
            hNewLayer = ActiveLayerNone;
        }

        // アプレットのレイヤを非表示
        pOldVisMgr->SetForegroundVisibility(false, IntegratedDisplayZOrder_Default, false);

        // アプリの場合、非表示にする以前に提出されたバッファの GPU フェンスの完了を待つ
        if(pOldLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
        {
            NN_AM_DISPCTRL_LOG_TRACE("start waiting for all presented pre-acquire fence expired\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::StopDisplayLayerPresentationRecording(layerId));

            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::StartDisplayLayerPresentationFenceWait(layerId));
            NN_UTIL_SCOPE_EXIT{ DisplayPrimitiveOperation::StopDisplayLayerPresentationFenceWait(layerId); };

            nn::os::SystemEventType expiredEvent;
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::InitializeDisplayLayerPresentationAllFencesExpiredEvent(&expiredEvent, layerId));
            NN_UTIL_SCOPE_EXIT{ DisplayPrimitiveOperation::FinalizeDisplayLayerPresentationAllFencesExpiredEvent(&expiredEvent, layerId); };

            // TORIAEZU: 無限待ち
            // アプリの GPU が完全にはまった場合抜けなくなるが、その場合↑の遷移キャプチャも動かないので色々と対処が必要。
            nn::os::WaitSystemEvent(&expiredEvent);
            NN_AM_DISPCTRL_LOG_TRACE("complete waiting for all presented pre-acquire fence expired\n");
        }

        return hNewLayer;
    }

    IntegratedDisplayLayerHandle AppletPresentationManager::DeactivateCurrentActiveFullscreenSharedLayerImpl(
        IntegratedDisplayLayerHandle  hOldActiveLayer,
        bool isUpdateCaptueBufferEnabled
    ) NN_NOEXCEPT
    {
        auto pOldActiveLayer = m_pResource->GetAppletLayerPool()->GetLayer(hOldActiveLayer);
        NN_SDK_REQUIRES_NOT_NULL(pOldActiveLayer);

        NN_SDK_REQUIRES(
            pOldActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_FullscreenApplet
        );
        NN_SDK_REQUIRES(
            pOldActiveLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_SystemShared
        );
        NN_SDK_REQUIRES(
            pOldActiveLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer
        );

        auto param = m_pResource->GetParameter();
        auto pLowLayer = m_pResource->GetSystemSharedFullscreenLayer();
        auto hSharedLayer = pOldActiveLayer->GetSharedLayerHandle();
        NN_SDK_ASSERT(!hSharedLayer.IsInvalid());

        // Conductor 指定を強制解除
        DisplayPrimitiveOperation::SetIsConductorLayer(pLowLayer->GetLayerId(), false);

        // デタッチ
        NN_AM_DISPCTRL_LOG_TRACE("detaching buffers from shared-layer %lld\n", hSharedLayer);
        DisplayPrimitiveOperation::DetachSharedLayer(hSharedLayer, pOldActiveLayer->GetSharedLayerDetachReadyEvent(), param.sharedLayerDetachTimeout);

        int lastIndex = -1;
        DisplayPrimitiveOperation::SynchronizeSharedLowLevelLayer(&lastIndex, pLowLayer->GetLayerId(), pLowLayer->GetSynchronizedEvent());
        NN_AM_DISPCTRL_LOG_TRACE("synchronized display layer (buf=%d)\n", lastIndex);

        // 撮影対象なら LastFg に反映
        bool isLastFgUpdated = false;
        if(isUpdateCaptueBufferEnabled && pOldActiveLayer->IsCaptureTarget())
        {
            // スクリーンショット許可状態は現在の値とリクエストされた値の両方で制限
            bool isScreenshotPermitted= pOldActiveLayer->InvokeScreenShotPermissionGetterFunction();
            if(pOldActiveLayer->IsCallerAppletCaptureRequested())
            {
                isScreenshotPermitted &= pOldActiveLayer->IsCallerAppletCaptureScreenshotPermitted();
            }

            // LastFg にコピー
            if(lastIndex >= 0)
            {
                NN_AM_DISPCTRL_LOG_TRACE("  copying last frame(iBuf=%d) to last-foreground\n", lastIndex);
                NN_SDK_ASSERT_RANGE(lastIndex, 0, param.layout.count);
                DisplayCaptureBufferOperation::CaptureSharedTexture(CaptureBufferIndex_LastForeground, lastIndex, isScreenshotPermitted, false, param.captureBufferTimeout);
            }
            else
            {
                NN_AM_DISPCTRL_LOG_WARN("fill last-foreground buffer with black (lastIdx=%d)\n", lastIndex);
                DisplayCaptureBufferOperation::FillBuffer(CaptureBufferIndex_LastForeground, CaptureBufferColorBlack, isScreenshotPermitted, false);
            }
            isLastFgUpdated = true;

            // NOTE:
            //   アプリはシステム共有バッファを使わないので LastApplication へのコピーはない

            // 要求されていれば CallerApplet にコピーする
            if(pOldActiveLayer->IsCallerAppletCaptureRequested())
            {
                NN_AM_DISPCTRL_LOG_TRACE("  copying last-foreground to caller-applet\n");
                DisplayCaptureBufferOperation::CopyBuffer(CaptureBufferIndex_CallerApplet, CaptureBufferIndex_LastForeground, param.captureBufferTimeout);
            }
        }

        // 遷移レイヤが必要なら出す
        IntegratedDisplayLayerHandle newIndex;
        if(pOldActiveLayer->IsTransitionLayerRequested() && isLastFgUpdated)
        {
            PresentTransitionBufferImpl();
            newIndex = ActiveLayerTransition;
        }
        else
        {
            BlankSystemSharedLayerImpl();
            newIndex = ActiveLayerNone;
        }

        DisplayPrimitiveOperation::SetSharedLayerVisibility(hSharedLayer, false, detail::IntegratedDisplayZOrder_Default, false);

        return newIndex;
    }

    //--------------------------------------------------------------
    // Fullscreen/Activate
    //--------------------------------------------------------------

    void AppletPresentationManager::ActivateFullscreenLayerImpl(IntegratedDisplayLayerHandle hNewActiveLayer) NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("activating fullscreen h=%lld\n", hNewActiveLayer);

        // 遷移レイヤまたはバッファなしなら何もせず終わり
        if(hNewActiveLayer < 0)
        {
            NN_AM_DISPCTRL_LOG_TRACE("  skip (no layer is active)\n");
            return;
        }

        auto pLayer = m_pResource->GetAppletLayerPool()->GetLayer(hNewActiveLayer);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        // Activate を実行
        switch(pLayer->GetBufferOwner())
        {
        case detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet:
            ActivateFullscreenDirectLayerImpl(hNewActiveLayer);
            break;
        case detail::IntegratedDisplayLayerBufferOwner_SystemShared:
            ActivateFullscreenSharedLayerImpl(hNewActiveLayer);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        // Conductor 指定を更新
        if(auto fullLayerId = GetFullscreenLayerIdImpl(hNewActiveLayer))
        {
            DisplayPrimitiveOperation::SetIsConductorLayer(fullLayerId, pLayer->IsConductorIfForeground());
        }

        // 念の為 CallerAppletCapture のリクエストを false で上書き
        pLayer->SetCallerAppletCaptureRequested(false, false);

        m_CurrentActiveFullscreenLayerHandle = hNewActiveLayer;
    }

    void AppletPresentationManager::ActivateFullscreenDirectLayerImpl(IntegratedDisplayLayerHandle hNewActiveLayer) NN_NOEXCEPT
    {
        auto pNewActiveLayer = m_pResource->GetAppletLayerPool()->GetLayer(hNewActiveLayer);
        auto pNewVisMgr = m_pResource->GetAppletLayerPool()->GetVisibilityManager(hNewActiveLayer);
        NN_SDK_ASSERT_NOT_NULL(pNewActiveLayer);
        NN_SDK_ASSERT_NOT_NULL(pNewVisMgr);

        NN_SDK_REQUIRES(
            pNewActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application ||
            pNewActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_FullscreenApplet
        );
        NN_SDK_REQUIRES(
            pNewActiveLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet
        );
        NN_SDK_REQUIRES(
            pNewActiveLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer
        );

        auto layerId = pNewActiveLayer->GetDirectLayerId();
        NN_SDK_ASSERT_NOT_EQUAL(layerId, 0u);

        // Wait for next frame from applet:
        //   On swapping from null to physical display, the first buffer latched may be
        //   its last FG buffer.
        nn::os::SystemEventType presentingEvent;
        DisplayPrimitiveOperation::InitializeLayerTexturePresentingEvent(&presentingEvent, layerId);
        NN_UTIL_SCOPE_EXIT{ DisplayPrimitiveOperation::FinalizeLayerTexturePresentingEvent(&presentingEvent, layerId); };

        NN_AM_DISPCTRL_LOG_TRACE("Waiting for new active layer to flip\n");
        if( !nn::os::TimedWaitSystemEvent(&presentingEvent, nn::TimeSpan::FromMilliSeconds(64)) )
        {
            NN_AM_DISPCTRL_LOG_WARN("  Timeout waiting for new active layer flip!\n");
        }

        // 権利表記を有効化
        //   AM がレイヤを切り替えている間はスクリーンショット撮影要求はブロックされている
        if (pNewActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
        {
            ShowApplicationCopyrightLayerImpl();
        }

        // アプリの場合、提出したバッファを記録する
        if(pNewActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_Application)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::StartDisplayLayerPresentationRecording(pNewActiveLayer->GetDirectLayerId()));
        }

        // Visibility を更新
        pNewVisMgr->SetForegroundVisibility(true, pNewActiveLayer->GetZOrder(), pNewActiveLayer->IsCaptureTarget());
    }

    void AppletPresentationManager::ActivateFullscreenSharedLayerImpl(IntegratedDisplayLayerHandle hNewActiveLayer) NN_NOEXCEPT
    {
        auto pNewActiveLayer = m_pResource->GetAppletLayerPool()->GetLayer(hNewActiveLayer);
        NN_SDK_ASSERT_NOT_NULL(pNewActiveLayer);

        NN_SDK_REQUIRES(
            pNewActiveLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_FullscreenApplet
        );
        NN_SDK_REQUIRES(
            pNewActiveLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_SystemShared
        );
        NN_SDK_REQUIRES(
            pNewActiveLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer
        );

        auto param = m_pResource->GetParameter();
        auto pLowLayer = m_pResource->GetSystemSharedFullscreenLayer();
        auto hSharedLayer = pNewActiveLayer->GetSharedLayerHandle();

        DisplayPrimitiveOperation::SetSharedLayerVisibility(hSharedLayer, pNewActiveLayer->IsVisible(), pNewActiveLayer->GetZOrder(), pNewActiveLayer->IsCaptureTarget());

        NN_SDK_ASSERT(!hSharedLayer.IsInvalid());
        NN_AM_DISPCTRL_LOG_TRACE("attaching buffers to shared-layer %lld\n", hSharedLayer);
        DisplayPrimitiveOperation::AttachSharedLayer(hSharedLayer, pLowLayer->GetLayerId(), param.indexForFullscreen);
    }

    //--------------------------------------------------------------
    // Partial, Overlay
    //--------------------------------------------------------------

    void AppletPresentationManager::DeactivateCurrentPartialLayerImpl() NN_NOEXCEPT
    {
        auto param = m_pResource->GetParameter();

        // Detach
        if(m_CurrentActivePartialLayerHandle >= 0)
        {
            auto pOldLayer = m_pResource->GetAppletLayerPool()->GetLayer(m_CurrentActivePartialLayerHandle);
            NN_ABORT_UNLESS_NOT_NULL(pOldLayer);

            if(pOldLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet)
            {
                if(auto layerId = pOldLayer->GetDirectLayerId())
                {
                    auto pVisMgr = m_pResource->GetAppletLayerPool()->GetVisibilityManager(m_CurrentActivePartialLayerHandle);
                    NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
                    pVisMgr->SetForegroundVisibility(false, IntegratedDisplayZOrder_Default, false);
                }
                // Indirect の場合は何もしない
            }
            else if(pOldLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_SystemShared)
            {
                auto hBuffer = m_pResource->GetSystemSharedBufferHandle();
                auto pLowLayer = m_pResource->GetSystemSharedPartialLayer();
                auto hSharedLayer = pOldLayer->GetSharedLayerHandle();

                switch(pOldLayer->GetDestinationLayer())
                {
                case detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer:
                    {
                        DisplayPrimitiveOperation::SetSharedLayerVisibility(hSharedLayer, false, detail::IntegratedDisplayZOrder_Default, false);
                        DisplayPrimitiveOperation::DetachSharedLayer(hSharedLayer, pOldLayer->GetSharedLayerDetachReadyEvent(), param.sharedLayerDetachTimeout);
                        // SIGLO-68694: 必要なくなったら破棄
                        pLowLayer->Finalize();
                        break;
                    }
                case detail::IntegratedDisplayLayerDestinationLayer_IndirectLayer:
                    {
                        DisplayPrimitiveOperation::DetachSharedLayer(hSharedLayer, pOldLayer->GetSharedLayerDetachReadyEvent(), param.sharedLayerDetachTimeout);
                        DisplayPrimitiveOperation::BlankSharedLowLevelLayer(pOldLayer->GetIndirectProducerHandleForSharedLayer(), hBuffer);
                        break;
                    }
                default: NN_UNEXPECTED_DEFAULT;
                }
            }
            m_CurrentActivePartialLayerHandle = ActiveLayerNone;
        }
    }

    void AppletPresentationManager::ActivatePartialLayerImpl(IntegratedDisplayLayerHandle newActiveLayer) NN_NOEXCEPT
    {
        if(newActiveLayer < 0)
        {
            m_CurrentActivePartialLayerHandle = newActiveLayer;
            return;
        }

        auto param = m_pResource->GetParameter();

        // Attach
        auto pNewLayer = m_pResource->GetAppletLayerPool()->GetLayer(newActiveLayer);
        NN_ABORT_UNLESS_NOT_NULL(pNewLayer);

        if(pNewLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet)
        {
            if(auto layerId = pNewLayer->GetDirectLayerId())
            {
                auto pVisMgr = m_pResource->GetAppletLayerPool()->GetVisibilityManager(newActiveLayer);
                NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
                pVisMgr->SetForegroundVisibility(pNewLayer->IsVisible(), pNewLayer->GetZOrder(), pNewLayer->IsCaptureTarget());
            }
            // Indirect の場合は何もしない
        }
        else if(pNewLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_SystemShared)
        {
            auto hBuffer = m_pResource->GetSystemSharedBufferHandle();
            auto pLowLayer = m_pResource->GetSystemSharedPartialLayer();
            auto hSharedLayer = pNewLayer->GetSharedLayerHandle();

            switch(pNewLayer->GetDestinationLayer())
            {
            case detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer:
                {
                    // SIGLO-68694: 必要になったら初期化
                    NN_ABORT_UNLESS_RESULT_SUCCESS(pLowLayer->Initialize(hBuffer, false, detail::IntegratedDisplayZOrder_SystemSharedPartial));
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::AttachSharedLayer(hSharedLayer, pLowLayer->GetLayerId(), param.indexForPartial));
                    DisplayPrimitiveOperation::SetSharedLayerVisibility(hSharedLayer, pNewLayer->IsVisible(), pNewLayer->GetZOrder(), pNewLayer->IsCaptureTarget());
                    break;
                }
            case detail::IntegratedDisplayLayerDestinationLayer_IndirectLayer:
                DisplayPrimitiveOperation::AttachSharedLayer(hSharedLayer, pNewLayer->GetIndirectProducerHandleForSharedLayer(), param.indexForPartial);
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        m_CurrentActivePartialLayerHandle = newActiveLayer;
    }

    void AppletPresentationManager::DeactivateCurrentOverlayLayerImpl() NN_NOEXCEPT
    {
        // Overlay は自前バッファのみ
        if(m_CurrentActiveOverlayLayerHandle >= 0)
        {
            auto pVisMgr = m_pResource->GetAppletLayerPool()->GetVisibilityManager(m_CurrentActiveOverlayLayerHandle);
            NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
            pVisMgr->SetForegroundVisibility(false, IntegratedDisplayZOrder_Default, false);
            m_CurrentActiveOverlayLayerHandle = ActiveLayerNone;
        }
    }

    void AppletPresentationManager::ActivateOverlayLayerImpl(IntegratedDisplayLayerHandle newActiveLayer) NN_NOEXCEPT
    {
        // Overlay は自前バッファのみ
        if(newActiveLayer >= 0)
        {
            auto pNewLayer = m_pResource->GetAppletLayerPool()->GetLayer(newActiveLayer);
            auto pVisMgr = m_pResource->GetAppletLayerPool()->GetVisibilityManager(newActiveLayer);
            NN_ABORT_UNLESS_NOT_NULL(pNewLayer);
            NN_ABORT_UNLESS_NOT_NULL(pVisMgr);
            pVisMgr->SetForegroundVisibility(pNewLayer->IsVisible(), pNewLayer->GetZOrder(), pNewLayer->IsCaptureTarget());
            m_CurrentActiveOverlayLayerHandle = newActiveLayer;
        }
    }

    //--------------------------------------------------------------
    // Other
    //--------------------------------------------------------------

    void AppletPresentationManager::NotifyConductorChanged(IntegratedDisplayLayerHandle hLayer) NN_NOEXCEPT
    {
        auto fullLayerId = GetFullscreenLayerIdImpl(hLayer);
        if(fullLayerId == 0)
        {
            return;
        }

        auto pLayer = m_pResource->GetAppletLayerPool()->GetLayer(hLayer);
        NN_ABORT_UNLESS_NOT_NULL(pLayer);

        // Foreground なら設定値を使う。それ以外なら false 。
        bool value = false;
        if(m_CurrentActiveFullscreenLayerHandle == hLayer)
        {
            value = pLayer->IsConductorIfForeground();
        }

        DisplayPrimitiveOperation::SetIsConductorLayer(fullLayerId, value);
    }

    void AppletPresentationManager::ImmediateCaptureCurrentFullscreenLayer(CaptureBufferIndex index, bool isScreenShotPermitted) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(index, 0, CaptureBufferIndex_Max);
        if(index < 0 || index >= CaptureBufferIndex_Max)
        {
            return;
        }

        auto hLayer = m_CurrentActiveFullscreenLayerHandle;
        auto pLayer = m_pResource->GetAppletLayerPool()->GetLayer(hLayer);

        if(pLayer == nullptr)
        {
            // TORIAEZU: 何もしない。
            // 本当は黒塗り？
            NN_SDK_REQUIRES_NOT_NULL(pLayer);
            return;
        }

        auto param = m_pResource->GetParameter();
        auto pLowLayer = m_pResource->GetSystemSharedFullscreenLayer();

        NN_SDK_REQUIRES(pLayer->GetUserClass() == detail::IntegratedDisplayLayerUserClass_FullscreenApplet);
        NN_SDK_REQUIRES(pLayer->GetBufferOwner() == detail::IntegratedDisplayLayerBufferOwner_SystemShared);
        NN_SDK_REQUIRES(pLayer->GetDestinationLayer() == detail::IntegratedDisplayLayerDestinationLayer_DisplayLayer);

        auto hSharedLayer = pLayer->GetSharedLayerHandle();
        NN_SDK_ASSERT(!hSharedLayer.IsInvalid());

        // 一旦デタッチ＆同期してキャプチャした後アタッチしなおす。
        // 結果元に戻るので状態の書き換えは行わない。

        // デタッチ
        int lastIndex = -1;
        {
            NN_AM_DISPCTRL_LOG_TRACE("detaching buffers from shared-layer %lld\n", hSharedLayer);
            DisplayPrimitiveOperation::DetachSharedLayer(hSharedLayer, pLayer->GetSharedLayerDetachReadyEvent(), param.sharedLayerDetachTimeout);
            DisplayPrimitiveOperation::SynchronizeSharedLowLevelLayer(&lastIndex, pLowLayer->GetLayerId(), pLowLayer->GetSynchronizedEvent());
            NN_AM_DISPCTRL_LOG_TRACE("synchronized display layer (buf=%d)\n", lastIndex);
        }

        // 撮影
        if(lastIndex >= 0 && lastIndex < param.layout.count)
        {
            bool currentScreenShotPermission = pLayer->InvokeScreenShotPermissionGetterFunction();
            bool copyrightRequested = false; // アプリではないので不要
            DisplayCaptureBufferOperation::CaptureSharedTexture(
                index,
                lastIndex,
                isScreenShotPermitted & currentScreenShotPermission,
                copyrightRequested,
                param.captureBufferTimeout
            );
        }

        // 再アタッチ
        {
            NN_AM_DISPCTRL_LOG_TRACE("attaching buffers to shared-layer %lld\n", hSharedLayer);
            NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::AttachSharedLayer(hSharedLayer, pLowLayer->GetLayerId(), param.indexForFullscreen));
        }

        // キャプチャは実行されたのでキャプチャ要求を落とす。
        pLayer->SetCallerAppletCaptureRequested(false, false);
    }

    void AppletPresentationManager::ShowTransitionBuffer() NN_NOEXCEPT
    {
        if(m_CurrentActiveFullscreenLayerHandle == ActiveLayerNone)
        {
            PresentTransitionBufferImpl();
            m_CurrentActiveFullscreenLayerHandle = ActiveLayerTransition;
        }
    }

    void AppletPresentationManager::HideTransitionBuffer() NN_NOEXCEPT
    {
        if(m_CurrentActiveFullscreenLayerHandle == ActiveLayerTransition)
        {
            BlankSystemSharedLayerImpl();
            m_CurrentActiveFullscreenLayerHandle = ActiveLayerNone;
        }
    }


    //--------------------------------------------------------------
    // Utility
    //--------------------------------------------------------------

    nn::vi::LayerId AppletPresentationManager::GetFullscreenLayerIdImpl(IntegratedDisplayLayerHandle handle) const NN_NOEXCEPT
    {
        auto pLayer = m_pResource->GetAppletLayerPool()->GetLayer(handle);

        if(pLayer == nullptr)
        {
            return 0;
        }

        if(pLayer->GetState() != detail::IntegratedDisplayLayerState_Determined)
        {
            return 0;
        }

        if(pLayer->GetUserClass() != detail::IntegratedDisplayLayerUserClass_Application &&
            pLayer->GetUserClass() != detail::IntegratedDisplayLayerUserClass_FullscreenApplet)
        {
            return 0;
        }

        switch(pLayer->GetBufferOwner())
        {
        case detail::IntegratedDisplayLayerBufferOwner_OwnedByApplet:
            return pLayer->GetDirectLayerId();
        case detail::IntegratedDisplayLayerBufferOwner_SystemShared:
            return m_pResource->GetSystemSharedFullscreenLayer()->GetLayerId();
        default:
            return 0;
        }
    }

    void AppletPresentationManager::PresentTransitionBufferImpl() NN_NOEXCEPT
    {
        auto param = m_pResource->GetParameter();
        NN_AM_DISPCTRL_LOG_TRACE("presenting transition buffer(%d)\n", param.transitionBufferIndex);

        auto hBuffer = m_pResource->GetSystemSharedBufferHandle();
        auto pLowLayer = m_pResource->GetSystemSharedFullscreenLayer();

        int presentedIndex = -1;
        DisplayPrimitiveOperation::SynchronizeSharedLowLevelLayer(&presentedIndex, pLowLayer->GetLayerId(), pLowLayer->GetSynchronizedEvent());
        if(presentedIndex != param.transitionBufferIndex)
        {
            auto presentResult = DisplayPrimitiveOperation::PresentDetachedSharedFrameBufferToLowLevelLayer(hBuffer, pLowLayer->GetLayerId(), param.transitionBufferIndex);
            NN_AM_DISPCTRL_LOG_TRACE("  presenting transition buffer -> %d-%d\n", presentResult.GetModule(), presentResult.GetDescription());
            NN_UNUSED(presentResult);
        }
        else
        {
            NN_AM_DISPCTRL_LOG_TRACE("  no-op (transition buffer is already presented)\n");
        }
    }

    void AppletPresentationManager::BlankSystemSharedLayerImpl() NN_NOEXCEPT
    {
        NN_AM_DISPCTRL_LOG_TRACE("blanking system layer\n");
        auto hBuffer = m_pResource->GetSystemSharedBufferHandle();
        auto pLowLayer = m_pResource->GetSystemSharedFullscreenLayer();

        DisplayPrimitiveOperation::BlankSharedLowLevelLayer(pLowLayer->GetLayerId(), hBuffer);
    }

    void AppletPresentationManager::ShowApplicationCopyrightLayerImpl() NN_NOEXCEPT
    {
        m_pResource->ActivateApplicationCopyrightLayer();
    }

    void AppletPresentationManager::HideApplicationCopyrightLayerImpl() NN_NOEXCEPT
    {
        m_pResource->DeactivateApplicationCopyrightLayer();
    }

}}}}}

