﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_GpuResourceControl.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/os/os_Mutex.h>
#include <mutex>

#include <nv_Scheduler.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <algorithm>

namespace nn { namespace am { namespace service {

namespace {

bool g_Initialized = false;
nv::sched::Client g_Client;

class RunList
{
private:

    nv::sched::Client::runlist_id_t m_Id;
    nv::sched::Client::priority_t m_Priority;
    TimeSpan m_CurrentTimeSlice;
    TimeSpan m_NextTimeSlice;
    const char* m_NameForDebug = "";

    void UpdateImpl() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(seq, "GPU:SetRunlistParameters([%d]%s, priority = %d, %lld us) (was %lld us)\n"
            , static_cast<int>(m_Id)
            , m_NameForDebug
            , static_cast<int>(m_Priority)
            , static_cast<int>(m_NextTimeSlice.GetMicroSeconds())
            , static_cast<int>(m_CurrentTimeSlice.GetMicroSeconds())
        );
        this->m_CurrentTimeSlice = m_NextTimeSlice;
        // ペナルティ時間が異常に増加してしまう不具合に備えてペナルティ時間の上限を設定
        nn::TimeSpan maxDebt = std::min(nn::TimeSpan::FromNanoSeconds(16666666), m_CurrentTimeSlice + m_CurrentTimeSlice);
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.SetRunlistParameters(m_Id, m_Priority, m_CurrentTimeSlice, maxDebt));
    }

public:

    void Initialize(nv::sched::Client::priority_t priority) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.CreateRunlist(m_Id));
        this->m_Priority = priority;
        this->m_CurrentTimeSlice = 0;
        this->m_NextTimeSlice = 0;
    }

    void SetNameForDebug(const char* p) NN_NOEXCEPT
    {
        this->m_NameForDebug = p;
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.DeleteRunlist(m_Id));
    }

    void Add(applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.RegisterRunlist(aruid.lower, m_Id));
    }

    void Remove(applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.DetachApplication(aruid.lower));
    }

    void SetNext(TimeSpan timeSlice) NN_NOEXCEPT
    {
        this->m_NextTimeSlice = timeSlice;
    }

    TimeSpan GetNext() const NN_NOEXCEPT
    {
        return m_NextTimeSlice;
    }

    void UpdateIfNegative() NN_NOEXCEPT
    {
        if (m_CurrentTimeSlice > m_NextTimeSlice)
        {
            UpdateImpl();
        }
    }

    void UpdateIfPositive() NN_NOEXCEPT
    {
        if (m_CurrentTimeSlice < m_NextTimeSlice)
        {
            UpdateImpl();
        }
    }

};

class GpuResourceManager
{
private:

    RunList m_RunLists[GpuResourceGroupId_Count];

public:

    void Initialize() NN_NOEXCEPT
    {
        m_RunLists[GpuResourceGroupId_Overlay].Initialize(50);
        m_RunLists[GpuResourceGroupId_System].Initialize(60);
        m_RunLists[GpuResourceGroupId_Application].Initialize(70);
        m_RunLists[GpuResourceGroupId_Overlay].SetNameForDebug("Ovl");
        m_RunLists[GpuResourceGroupId_System].SetNameForDebug("Sys");
        m_RunLists[GpuResourceGroupId_Application].SetNameForDebug("App");
    }

    void Finalize() NN_NOEXCEPT
    {
        for (auto&& e: m_RunLists)
        {
            e.Finalize();
        }
    }

    RunList& operator[](GpuResourceGroupId groupId) NN_NOEXCEPT
    {
        return m_RunLists[groupId];
    }

    void SetNext(GpuResourceGroupId groupId, TimeSpan timeSlice) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(0 <= groupId && groupId < GpuResourceGroupId_Count);
        m_RunLists[groupId].SetNext(timeSlice);
    }

    void Update() NN_NOEXCEPT
    {
        auto rest = GetGpuScheduleTimeSpan();
        for (auto&& e: m_RunLists)
        {
            rest -= e.GetNext();
        }
        if (rest > 0)
        {
            auto&& runlist = m_RunLists[GpuResourceGroupId_Application];
            runlist.SetNext(runlist.GetNext() + rest);
        }
        for (auto&& e: m_RunLists)
        {
            e.UpdateIfNegative();
        }
        for (auto&& e: m_RunLists)
        {
            e.UpdateIfPositive();
        }
    }

};

GpuResourceManager g_GpuResourceManager;
os::Mutex g_GpuResourceManagerUpdateMutex{false};

class Configuration
{
private:

    bool m_Enabled;
    TimeSpan m_GpuScheduleTimeSpan;
    TimeSpan m_AppletTimeSlices[GpuTimeKind_Count];

    template <typename T, typename U>
    static T ReadFirmwareDebugSettings(const char* name, const U& defaultValue) NN_NOEXCEPT
    {
        T ret;
        auto size = nn::settings::fwdbg::GetSettingsItemValue(&ret, sizeof(ret), "am.gpu", name);
        if (!(sizeof(ret) == size))
        {
            NN_AM_SERVICE_LOG(error, "failed: ReadFirmwareDebugSettings(%s) size = %d(expected:%d)\n", name, static_cast<int>(size), static_cast<int>(sizeof(ret)));
            return defaultValue;
        }
        return ret;
    }

public:

    Configuration() NN_NOEXCEPT
    {
        this->m_Enabled = ReadFirmwareDebugSettings<bool>("gpu_scheduling_enabled", false);
        NN_AM_SERVICE_LOG(seq, "gpu_scheduling_enabled: %s\n", m_Enabled ? "true" : "false");

        this->m_GpuScheduleTimeSpan = TimeSpan::FromMicroSeconds(ReadFirmwareDebugSettings<int32_t>("gpu_scheduling_frame_time_us", 16666));

        #define READ_TIME_SLICE(kind, name, v) \
            m_AppletTimeSlices[GpuTimeKind_##kind] = TimeSpan::FromMicroSeconds(ReadFirmwareDebugSettings<int32_t>("gpu_scheduling_" name "_us", v)); \
            NN_AM_SERVICE_LOG(call, "gpu_scheduling_%s_us: %d\n", name, static_cast<int>(m_AppletTimeSlices[GpuTimeKind_##kind].GetMicroSeconds()))

            READ_TIME_SLICE(FgApp, "fg_app", 16166);
            READ_TIME_SLICE(BgApp, "bg_app", 4499);
            READ_TIME_SLICE(Oa, "oa", 500);
            READ_TIME_SLICE(FgSa, "fg_sa", 11666);
            READ_TIME_SLICE(BgSa, "bg_sa", 0);
            READ_TIME_SLICE(FgLa, "fg_la", 11666);
            READ_TIME_SLICE(PartialFgLa, "partial_fg_la", 2000);
            READ_TIME_SLICE(BgLa, "bg_la", 0);

        #undef READ_TIME_SLICE
    }

    bool IsEnabled() const NN_NOEXCEPT
    {
        return m_Enabled;
    }

    TimeSpan GetGpuScheduleTimeSpan() const NN_NOEXCEPT
    {
        return m_GpuScheduleTimeSpan;
    }

    TimeSpan GetGpuTimeSliceOf(GpuTimeKind kind) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(kind <= GpuTimeKind_Count);
        return m_AppletTimeSlices[kind];
    }

};

Configuration& GetConfiguration() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(Configuration, g_Configuration);
    return g_Configuration;
}

}

bool IsGpuSchedulingEnabled() NN_NOEXCEPT
{
    return GetConfiguration().IsEnabled();
}

TimeSpan GetGpuScheduleTimeSpan() NN_NOEXCEPT
{
    return GetConfiguration().GetGpuScheduleTimeSpan();
}

void StartGpuResourceControl() NN_NOEXCEPT
{
    if (GetConfiguration().IsEnabled())
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.Open());
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.Enable());
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.SetFrameInterval(GetGpuScheduleTimeSpan()));
        g_GpuResourceManager.Initialize();
    }
    g_Initialized = true;
}

TimeSpan GetGpuTimeSliceOf(GpuTimeKind kind) NN_NOEXCEPT
{
    return GetConfiguration().GetGpuTimeSliceOf(kind);
}

void StopGpuResourceControl() NN_NOEXCEPT
{
    if (GetConfiguration().IsEnabled())
    {
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(gpu_scheduler, 60);
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.Disable());
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_Client.Close());
    }
}

void RegisterToGpuResourceControl(GpuResourceGroupId groupId, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (GetConfiguration().IsEnabled())
    {
        g_GpuResourceManager[groupId].Add(aruid);
    }
}

void UnregisterFromGpuResourceControl(GpuResourceGroupId groupId, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    if (GetConfiguration().IsEnabled())
    {
        if (!(groupId == GpuResourceGroupId_Application))
        {
            g_GpuResourceManager[GpuResourceGroupId_Application].Add(aruid);
        }
        g_GpuResourceManager[groupId].Remove(aruid);
    }
}

void SetGpuResourceControl(const GpuResourceControlInfo& info) NN_NOEXCEPT
{
    if (GetConfiguration().IsEnabled())
    {
        std::lock_guard<decltype(g_GpuResourceManagerUpdateMutex)> lk(g_GpuResourceManagerUpdateMutex);
        if (!info.IsSetAny())
        {
            g_GpuResourceManager.SetNext(GpuResourceGroupId_Application, GetGpuTimeSliceOf(GpuTimeKind_BgApp));
            g_GpuResourceManager.SetNext(GpuResourceGroupId_System, GetGpuTimeSliceOf(GpuTimeKind_FgSa));
            g_GpuResourceManager.SetNext(GpuResourceGroupId_Overlay, GetGpuTimeSliceOf(GpuTimeKind_Oa));
        }
        else
        {
            NN_SDK_ASSERT(info.IsSetAny());
            for (int i = 0; i < GpuResourceGroupId_Count; ++i)
            {
                auto groupId = static_cast<GpuResourceGroupId>(i);
                g_GpuResourceManager.SetNext(groupId, info[groupId]);
            }
        }
        g_GpuResourceManager.Update();
    }
}

}}}
