﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/am/service/am_AppletSystem.h>
#include <nn/am/service/am_ServiceConfig.h>

namespace nn { namespace am { namespace service {

//-----------------------------------------------------------------------------
//  CaptureButton 監視用クラス
//  TORIAEZU:
//      単一スレッドでの利用を前提としているため、
//      m_State 変数の排他などは行なわない。
//
class CaptureButton
{
public:
    static TimeSpan GetTimeSpanShortPressing() NN_NOEXCEPT
    {
        return TimeSpan::FromMilliSeconds(30);
    }
    static TimeSpan GetTimeSpanLongPressing() NN_NOEXCEPT
    {
        return service::GetCaptureLongPressTime();
    }

public:
    CaptureButton() NN_NOEXCEPT;
    ~CaptureButton() NN_NOEXCEPT;

    void SetWindowManager(window::WindowManager* p) NN_NOEXCEPT
    {
        m_pWindowManager = p;
    }

    void LinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        os::LinkMultiWaitHolder(p, &m_CaptureButtonEventHolder);
        os::LinkMultiWaitHolder(p, &m_TimerEventHolder);
    }
    void UnlinkMultiWaitHolders(os::MultiWaitType* p) NN_NOEXCEPT
    {
        NN_UNUSED(p);
        os::UnlinkMultiWaitHolder(&m_CaptureButtonEventHolder);
        os::UnlinkMultiWaitHolder(&m_TimerEventHolder);
    }
    bool HandleEventIfHolderOwner(os::MultiWaitHolderType* holder)
    {
        if (holder == &m_TimerEventHolder)
        {
            TimerEventHandler();
            return true;
        }
        else if (holder == &m_CaptureButtonEventHolder)
        {
            CaptureButtonEventHandler();
            return true;
        }
        return false;
    }
    void InvalidateCurrentButtonPressing() NN_NOEXCEPT;

private:
    void TimerEventHandler() NN_NOEXCEPT;
    void CaptureButtonEventHandler() NN_NOEXCEPT;

    void RequestToTakeScreenShot() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(event, "capture button short pressed\n");
        m_pWindowManager->NotifyCaptureButtonShortPressed();
    }

    void RequestToStartMovieRecording() NN_NOEXCEPT
    {
        NN_AM_SERVICE_LOG(event, "capture button long pressed\n");
        m_pWindowManager->NotifyCaptureButtonLongPressed();
    }

private:
    window::WindowManager* m_pWindowManager;

    os::TimerEvent              m_TimerEvent;
    os::SystemEvent             m_CaptureButtonEvent;

    os::MultiWaitHolderType     m_TimerEventHolder;
    os::MultiWaitHolderType     m_CaptureButtonEventHolder;

    os::Tick    m_TickShortPressing;
    os::Tick    m_TickLongPressing;
    os::Tick    m_StartTick;

    os::Mutex   m_Mutex{false};
    bool        m_State;
};

CaptureButton& GetCaptureButton() NN_NOEXCEPT;

//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

