﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/hid/system/hid_CaptureButton.h>
#include <nn/am/service/am_ServiceConfig.h>

#include "am_CaptureButton.h"
#include <mutex>

namespace nn { namespace am { namespace service {

CaptureButton& GetCaptureButton() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(CaptureButton, g_CaptureButton);
    return g_CaptureButton;
}

//-----------------------------------------------------------------------------
//  コンストラクタ
//  TORIAEZU:
//      初期化時は Capture ボタンが押されていないと仮定する。
//      仮に押されていたとしても長押しとして検知しない。
//
CaptureButton::CaptureButton() NN_NOEXCEPT
      : m_TimerEvent(os::EventClearMode_AutoClear),
        m_CaptureButtonEvent(os::EventClearMode_AutoClear, true),
        m_State(false)
{
    hid::system::BindCaptureButtonEvent(m_CaptureButtonEvent.GetBase(), os::EventClearMode_AutoClear);
    m_CaptureButtonEvent.Clear();

    m_TickShortPressing  = os::ConvertToTick(GetTimeSpanShortPressing());
    m_TickLongPressing   = os::ConvertToTick(GetTimeSpanLongPressing());

    os::InitializeMultiWaitHolder(&m_TimerEventHolder,         m_TimerEvent.GetBase());
    os::InitializeMultiWaitHolder(&m_CaptureButtonEventHolder, m_CaptureButtonEvent.GetBase());

    m_TimerEventHolder.userData         = reinterpret_cast<uintptr_t>(this);
    m_CaptureButtonEventHolder.userData = reinterpret_cast<uintptr_t>(this);
}


//-----------------------------------------------------------------------------
//  デストラクタ
//
CaptureButton::~CaptureButton() NN_NOEXCEPT
{
    os::FinalizeMultiWaitHolder(&m_TimerEventHolder);
    os::FinalizeMultiWaitHolder(&m_CaptureButtonEventHolder);
}


//-----------------------------------------------------------------------------
//  現在押下中のキャプチャボタンを一度だけ無効化
//
void CaptureButton::InvalidateCurrentButtonPressing() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);
    this->m_State = false;
}


//-----------------------------------------------------------------------------
//  TimerEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void CaptureButton::TimerEventHandler() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    if (!m_TimerEvent.TryWait())
    {
        return;
    }

    if (m_State)
    {
        auto interval = os::GetSystemTick() - m_StartTick;

        // ホールド時間によってメッセージを変える
        if (interval >= m_TickLongPressing)
        {
            m_TimerEvent.Stop();

            // 長押しを検知したら動画撮影開始を依頼
            RequestToStartMovieRecording();
        }
    }
}


//-----------------------------------------------------------------------------
//  CaptureButtonEvent ハンドラ関数（多重待ち解除時に呼ばれる）
//
void CaptureButton::CaptureButtonEventHandler() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lk(m_Mutex);

    if (!m_CaptureButtonEvent.TryWait())
    {
        return;
    }

    // 試遊台の場合はキャプチャボタンを常に無効化しておく。
    // この実装では、長押しも短押しも全て無効化される。
    //
    // FIXME:
    //  将来的には、capsrv 等の IPC 発行レイヤーを am::service 以下に用意し、
    //  ウィンドウ処理の結果として呼ばれる OnCaptureButtonShortPressed() 等から
    //  そのレイヤーに用意した関数を経由して他モジュール機能を呼ぶようにする。
    //
    //  これにより、下記の IsQuestMode() のように、制御したい機能の単位で
    //  塞ぐことができるし、メンテナンス性も向上する。
    //
    if ( IsQuestMode() )
    {
        return;
    }

    hid::system::CaptureButtonState     captureButtonState;
    int count = hid::system::GetCaptureButtonStates(&captureButtonState, 1);
    NN_SDK_ASSERT(count == 1);
    NN_UNUSED(count);

    if ( captureButtonState.buttons.Test<hid::system::CaptureButton::Active>() )
    {
        // 押下エッジ検出なら時間計測開始
        m_StartTick = os::GetSystemTick();
        m_State     = true;
        m_TimerEvent.Clear();
        m_TimerEvent.StartOneShot( GetTimeSpanLongPressing() );
    }
    else
    {
        // 開放エッジ検出なら時間計測停止
        m_TimerEvent.Stop();

        // ボタンが押されていたら、その時間によってメッセージを通知
        if (m_State)
        {
            auto interval = os::GetSystemTick() - m_StartTick;

#if 0
            // ローンチ時点では 30msec 以上の押下継続＋キャプチャボタンリリース
            // で常にキャプチャ撮影を発動させる。
            if (m_TickShortPressing <= interval)
#else
            // 30msec～500msec の短押し検知でキャプチャ撮影を発動させる
            if (m_TickShortPressing <= interval &&
                                       interval < m_TickLongPressing)
#endif
            {
                RequestToTakeScreenShot();
            }
        }
        m_State = false;
    }
}


//-----------------------------------------------------------------------------

}}} // namespace nn::am::service

