﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_ApplicationFunctions.h>

#include <mutex>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/account/account_Types.h>
#include <nn/am/service/am_ErrorReport.h>
#include <nn/arp/arp_Api.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_TerminateResultApi.h>
#include <nn/ns/ns_PseudoDeviceIdApi.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/util/util_BitPack.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_UuidApi.private.h>
#include <nn/oe/oe_ApplicationControlTypes.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/pdm/pdm_QueryApiForSystem.h>

namespace nn { namespace am { namespace service {

namespace {
    os::Mutex g_ControlPropertyMutex(false);
}

Result EnsureSaveData(int64_t* pOut, ncm::ApplicationId applicationId, const account::Uid& uid, const ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_EnsureSaveData, 60);
    int64_t requiredSize = 0;
    // TODO: BGDL 等をサスペンド・レジューム
    {
        std::lock_guard<os::Mutex> scopedLock(g_ControlPropertyMutex);
        NN_RESULT_TRY(fs::EnsureApplicationSaveData(&requiredSize, applicationId, controlProperty, uid))
            NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
            {
                // 不足時も pOut を返すため Success を返す
                *pOut = requiredSize;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
    }

    *pOut = 0;
    NN_RESULT_SUCCESS;
}

Result GetDesiredLanguage(settings::LanguageCode* pOut, Bit32 supportedLanguageFlag) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_GetDesiredLanguage, 60);
    {
        std::lock_guard<os::Mutex> scopedLock(g_ControlPropertyMutex);
        *pOut = ns::GetApplicationSupportedLanguage(supportedLanguageFlag);
    }

    NN_RESULT_SUCCESS;
}

Result SetTerminateResult(ncm::ApplicationId id, Result result, const ApplicationErrorReportInfo& info) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_SetTerminateResult, 60);
    static os::Mutex s_Mutex(false);

    std::lock_guard<os::Mutex> scopedLock(s_Mutex);

    NN_RESULT_DO(ns::SetApplicationTerminateResult(id, result));

    {
        std::lock_guard<os::Mutex> controlPropertyLock(g_ControlPropertyMutex);
        NN_RESULT_DO(MakeApplicationAbortErrorReport(result, info));
    }

    NN_RESULT_SUCCESS;
}

Result GetDisplayVersion(oe::DisplayVersion* pOut, std::add_lvalue_reference<const char[16]>::type displayVersion) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_GetDisplayVersion, 60);
    {
        std::lock_guard<os::Mutex> scopedLock(g_ControlPropertyMutex);
        util::Strlcpy(pOut->value, displayVersion, sizeof(displayVersion));
    }

    NN_RESULT_SUCCESS;
}

Result GetPseudoDeviceId(util::Uuid* pOutId, Bit64 seedForPseudoDeviceId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_GetPseudoDeviceId, 60);
    ns::SystemSeedForPseudoDeviceId systemSeed;
    ns::GetSystemSeedForPseudoDeviceId( &systemSeed );
    Bit64 applicationSeed;

    {
        std::lock_guard<os::Mutex> scopedLock(g_ControlPropertyMutex);
        applicationSeed = seedForPseudoDeviceId;
    }

    Bit8 hash[crypto::Sha1Generator::HashSize];
    crypto::Sha1Generator sha1Generator;
    sha1Generator.Initialize();
    sha1Generator.Update( &systemSeed, ns::SystemSeedSize );
    sha1Generator.Update( &applicationSeed, sizeof(Bit64) );
    sha1Generator.GetHash( hash, crypto::Sha1Generator::HashSize );

    *pOutId = util::GenerateUuidVersion5( hash );

    NN_RESULT_SUCCESS;
}

Result ExtendSaveData(int64_t* pOut, ncm::ApplicationId applicationId, uint8_t saveDataType, const account::Uid& uid, int64_t saveDataSize, int64_t saveDataJournalSize, const ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_ExtendSaveData, 60);
    int64_t requiredSize = 0;
    // TODO: BGDL 等をサスペンド・レジューム
    {
        std::lock_guard<os::Mutex> scopedLock(g_ControlPropertyMutex);

        NN_RESULT_TRY(fs::ExtendApplicationSaveData(&requiredSize, applicationId, controlProperty, static_cast<fs::SaveDataType>(saveDataType), uid, saveDataSize, saveDataJournalSize))
            NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
            {
                // 不足時も pOut を返すため Success を返す
                *pOut = requiredSize;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
    }

    *pOut = 0;

    NN_RESULT_SUCCESS;
}

Result GetSaveDataSize(int64_t* pOutSize, int64_t* pOutJournalSize, ncm::ApplicationId applicationId, uint8_t saveDataType, const account::Uid& uid) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_GetSaveDataSize, 60);
    return fs::GetApplicationSaveDataSize(pOutSize, pOutJournalSize, applicationId, static_cast<fs::SaveDataType>(saveDataType), uid);
}

Result CreateCacheStorage(int64_t* pOut, int32_t* pOutMedia, ncm::ApplicationId applicationId, uint16_t index, int64_t cacheSotargeSize, int64_t cacheStorageJournalSize, const ns::ApplicationControlProperty& controlProperty) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_EnsureCacheStorage, 60);
    fs::CacheStorageTargetMedia media = fs::CacheStorageTargetMedia_Nand;
    int64_t requiredSize = 0;
    // TODO: BGDL 等をサスペンド・レジューム
    {
        std::lock_guard<os::Mutex> scopedLock(g_ControlPropertyMutex);
        NN_RESULT_TRY(fs::CreateApplicationCacheStorage(&requiredSize, &media, applicationId, controlProperty, index, cacheSotargeSize, cacheStorageJournalSize))
            NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
            {
                // 不足時も pOut, pOutMedia を返すため Success を返す
                *pOut = requiredSize;
                *pOutMedia = static_cast<int32_t>(media);
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
    }

    *pOut = 0;
    *pOutMedia = static_cast<int32_t>(media);
    NN_RESULT_SUCCESS;
}

Result QueryApplicationPlayStatistics(
    int* pOutCount, pdm::ApplicationPlayStatistics outValues[], const nn::ncm::ApplicationId applicationIds[], int count,
    const PlayLogQueryPolicy& playLogQueryPolicy, ncm::ApplicationId myApplicationId) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(application_QueryApplicationPlayStatistics, 60);
    // .nmeta に記載された権限のチェック
    // 自分自身のプレイ情報取得については .nmeta への記載は不要
    switch( playLogQueryPolicy.capability )
    {
    case ns::PlayLogQueryCapability::None:
        {
            for( int i = 0; i < count; i++ )
            {
                NN_RESULT_THROW_UNLESS(applicationIds[i] == myApplicationId, ResultInvalidCall());
            }
        }
        break;
    case ns::PlayLogQueryCapability::WhiteList:
        {
            for( int i = 0; i < count; i++ )
            {
                bool isListed = false;
                for( int j = 0; j < ns::PlayLogQueryableApplicationCountMax; j++ )
                {
                    if( applicationIds[i] == playLogQueryPolicy.queryableApplicationId[j] ||
                        applicationIds[i] == myApplicationId)
                    {
                        isListed = true;
                        break;
                    }
                }
                NN_RESULT_THROW_UNLESS(isListed, ResultInvalidCall());
            }
        }
        break;
    case ns::PlayLogQueryCapability::All:
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    pdm::InitializeForQuery();
    NN_UTIL_SCOPE_EXIT{ pdm::FinalizeForQuery(); };
    *pOutCount = pdm::QueryApplicationPlayStatisticsForSystem(outValues, applicationIds, count);
    NN_RESULT_SUCCESS;
}

}}}
