﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/sf/sf_ShimLibraryUtility.h>
//#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>
#include <nn/am/service/am_ServiceName.h>
#include <nn/am/service/am_AppletProxyService.sfdl.h>
#include <nn/am/service/am_SystemAppletProxyService.sfdl.h>


namespace nn { namespace am {

//-----------------------------------------------------------------------------

namespace {

#if 0
    sf::SimpleAllInOneHipcClientManager<100> g_HipcManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER;
#else   // サブドメイン化
    sf::SimpleAllInOneHipcSubDomainClientManager<100> g_HipcManager = NN_SF_SIMPLE_ALL_IN_ONE_HIPC_SUB_DOMAIN_CLIENT_MANAGER_INITIALIZER;
    sf::ShimLibraryObjectHolder<service::IAllSystemAppletProxiesService> g_AllSystemAppletProxyService = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
    sf::ShimLibraryObjectHolder<service::IApplicationProxyService> g_ApplicationProxyService = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
#endif

    #define NN_AM_RETURN_UNLESS_RESULT_NO_APPLET_PROXY(func) \
                for (;;)                                                            \
                {                                                                   \
                    auto _nn_am_temp_result = (func);                               \
                    if (!(_nn_am_temp_result <= ::nn::am::ResultNoAppletProxy()))   \
                    {                                                               \
                        return _nn_am_temp_result;                                  \
                    }                                                               \
                    os::SleepThread( TimeSpan::FromMilliSeconds(10) );              \
                }

}   // namespace anonymous

Interfaces g_Interfaces =
{
    {   // ApplicationServiceInterfaces
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
    },
    {   // SystemServiceInterfaces
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,

        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,

        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
    },
    {   // CommonInterfaces
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
        NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER,
    }
};

applet::AppletResourceUserId g_AppletResourceUserId = {};

//-----------------------------------------------------------------------------

service::IApplicationFunctions* GetApplicationFunctions() NN_NOEXCEPT
{
    auto p = g_Interfaces.application.applicationFunctions.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

//-----------------------------------------------------------------------------

service::ISystemAppletProxy* GetSystemAppletProxy() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.systemAppletProxy.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IHomeMenuFunctions* GetHomeMenuFunctions() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.homeMenuFunctions.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IGlobalStateController* GetGlobalStateController() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.globalStateController.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IApplicationCreator* GetApplicationCreator() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.applicationCreator.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::ILibraryAppletProxy* GetLibraryAppletProxy() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.libraryAppletProxy.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::ILibraryAppletSelfAccessor* GetLibraryAppletSelfAccessor() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.libraryAppletSelfAccessor.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IOverlayAppletProxy* GetOverlayAppletProxy() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.overlayAppletProxy.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IOverlayFunctions* GetOverlayFunctions() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.overlayFunctions.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IApplicationProxy* GetApplicationProxy() NN_NOEXCEPT
{
    auto p = g_Interfaces.application.applicationProxy.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

//-----------------------------------------------------------------------------

service::ILibraryAppletCreator* GetLibraryAppletCreator() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.libraryAppletCreator.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IProcessWindingController* GetProcessWindingController() NN_NOEXCEPT
{
    auto p = g_Interfaces.system.processWindingController.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::ICommonStateGetter* GetCommonStateGetter() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.commonStateGetter.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::ISelfController* GetSelfController() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.selfController.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IAudioController* GetAudioController() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.audioController.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IDisplayController* GetDisplayController() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.displayController.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IWindowController* GetWindowController() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.windowController.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

service::IDebugFunctions* GetDebugFunctions() NN_NOEXCEPT
{
    auto p = g_Interfaces.common.debugFunctions.GetObject().Get();
    NN_SDK_REQUIRES( p );
    return p;
}

//-----------------------------------------------------------------------------

bool IsInitializedAsApplication() NN_NOEXCEPT
{
    return NULL != g_Interfaces.application.applicationProxy.GetObject().Get();
}

bool IsInitializedAsSystemApplet() NN_NOEXCEPT
{
    return NULL != g_Interfaces.system.systemAppletProxy.GetObject().Get();
}

bool IsInitializedAsLibraryApplet() NN_NOEXCEPT
{
    return NULL != g_Interfaces.system.libraryAppletProxy.GetObject().Get();
}

bool IsInitializedAsOverlayApplet() NN_NOEXCEPT
{
    return NULL != g_Interfaces.system.overlayAppletProxy.GetObject().Get();
}

//-----------------------------------------------------------------------------

#define NN_AM_GET_INTERFACE_LOCAL(Interface, Member, Provider, Getter)         \
            NN_ABORT_UNLESS_RESULT_SUCCESS( Member.InitializeHolder(           \
                [Provider](sf::SharedPointer<Interface>* pProxy) -> nn::Result \
                {                                                              \
                    return Provider->Getter(pProxy);                           \
                }                                                              \
            ));

#define NN_AM_GET_INTERFACE(Interface, Member, Provider, Getter)               \
            NN_ABORT_UNLESS_RESULT_SUCCESS( Member.InitializeHolder(           \
                [](sf::SharedPointer<Interface>* pProxy) -> nn::Result         \
                {                                                              \
                    return Provider->Getter(pProxy);                           \
                }                                                              \
            ));

//-----

void InitializeCommonInterfaces(service::IStackableAppletProxy* proxy) NN_NOEXCEPT
{
    NN_AM_GET_INTERFACE_LOCAL(service::ILibraryAppletCreator,
                              g_Interfaces.common.libraryAppletCreator,
                              proxy,
                              GetLibraryAppletCreator);

    NN_AM_GET_INTERFACE_LOCAL(service::ICommonStateGetter,
                              g_Interfaces.common.commonStateGetter,
                              proxy,
                              GetCommonStateGetter);

    NN_AM_GET_INTERFACE_LOCAL(service::ISelfController,
                              g_Interfaces.common.selfController,
                              proxy,
                              GetSelfController);

    NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interfaces.common.windowController.InitializeHolder(
        [proxy](sf::SharedPointer<service::IWindowController>* pOut) -> nn::Result
        {
            sf::SharedPointer<service::IWindowController> ret;
            NN_RESULT_DO(proxy->GetWindowController(&ret));
            NN_ABORT_UNLESS_RESULT_SUCCESS(ret->GetAppletResourceUserId(&g_AppletResourceUserId));
            *pOut = std::move(ret);
            NN_RESULT_SUCCESS;
        },
        [](void*)
        {
            g_AppletResourceUserId = applet::AppletResourceUserId::GetInvalidId();
        }
    ));

    NN_AM_GET_INTERFACE_LOCAL(service::IAudioController,
                              g_Interfaces.common.audioController,
                              proxy,
                              GetAudioController);

    NN_AM_GET_INTERFACE_LOCAL(service::IDisplayController,
                              g_Interfaces.common.displayController,
                              proxy,
                              GetDisplayController);

    NN_AM_GET_INTERFACE_LOCAL(service::IDebugFunctions,
                              g_Interfaces.common.debugFunctions,
                              proxy,
                              GetDebugFunctions);
}

void FinalizeCommonInterfaces() NN_NOEXCEPT
{
    g_Interfaces.common.debugFunctions.FinalizeHolder();
    g_Interfaces.common.windowController.FinalizeHolder();
    g_Interfaces.common.displayController.FinalizeHolder();
    g_Interfaces.common.audioController.FinalizeHolder();
    g_Interfaces.common.selfController.FinalizeHolder();
    g_Interfaces.common.commonStateGetter.FinalizeHolder();
    g_Interfaces.common.libraryAppletCreator.FinalizeHolder();
}

//-----

void InitializeSystemAppletInterfaces() NN_NOEXCEPT
{
    // AppletServiceNameForSystem の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_HipcManager.InitializeShimLibraryHolder(&g_AllSystemAppletProxyService, service::AppletServiceNameForSystem) );

    // ISystemAppletProxy の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interfaces.system.systemAppletProxy.InitializeHolder(
        [](sf::SharedPointer<service::ISystemAppletProxy>* pProxy) -> nn::Result
        {
            NN_AM_RETURN_UNLESS_RESULT_NO_APPLET_PROXY( g_AllSystemAppletProxyService->OpenSystemAppletProxy(pProxy, 0, sf::NativeHandle(os::GetCurrentProcessHandle(), false)) );
        }
    ));

    // その他のインターフェイスの取得
    NN_AM_GET_INTERFACE(service::IHomeMenuFunctions,
                        g_Interfaces.system.homeMenuFunctions,
                        g_Interfaces.system.systemAppletProxy,
                        GetHomeMenuFunctions);

    NN_AM_GET_INTERFACE(service::IGlobalStateController,
                        g_Interfaces.system.globalStateController,
                        g_Interfaces.system.systemAppletProxy,
                        GetGlobalStateController);

    NN_AM_GET_INTERFACE(service::IApplicationCreator,
                        g_Interfaces.system.applicationCreator,
                        g_Interfaces.system.systemAppletProxy,
                        GetApplicationCreator);

    InitializeCommonInterfaces( GetSystemAppletProxy() );
}

void FinalizeSystemAppletInterfaces() NN_NOEXCEPT
{
    FinalizeCommonInterfaces();

    g_Interfaces.system.applicationCreator.FinalizeHolder();
    g_Interfaces.system.globalStateController.FinalizeHolder();
    g_Interfaces.system.homeMenuFunctions.FinalizeHolder();
    g_Interfaces.system.systemAppletProxy.FinalizeHolder();
    g_AllSystemAppletProxyService.FinalizeHolder();
}

//----------

namespace {

    void (*g_LibraryAppletResolver)();

}

void SetLibraryAppletResolver(void (*resolver)()) NN_NOEXCEPT
{
    g_LibraryAppletResolver = resolver;
}

void InvokeWithSelfLibraryCreatorEnabled(void (*f)()) NN_NOEXCEPT
{
    sf::SharedPointer<service::ILibraryAppletCreator> pSelfLibraryAppletCreator;
    auto result =g_AllSystemAppletProxyService->CreateSelfLibraryAppletCreatorForDevelop(&pSelfLibraryAppletCreator, 0);
    if (!result.IsSuccess())
    {
        if (result <= ResultRefuseToCreateSelfLibraryAppletProxy())
        {
            // アプレットプロキシの作成を拒絶された場合は、
            // 引き続きタイムアウト時間までポーリングを継続する
            return;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    g_Interfaces.common.libraryAppletCreator.InitializeHolderDirectly(pSelfLibraryAppletCreator);
    NN_UTIL_SCOPE_EXIT
    {
        g_Interfaces.common.libraryAppletCreator.FinalizeHolder();
    };
    f();
}

void InitializeLibraryAppletInterfaces(const AppletAttribute* pAttribute) NN_NOEXCEPT
{
    // AppletServiceNameForSystem の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_HipcManager.InitializeShimLibraryHolder(&g_AllSystemAppletProxyService, service::AppletServiceNameForSystem) );

    // ILibraryAppletProxy の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interfaces.system.libraryAppletProxy.InitializeHolder(
        [pAttribute](sf::SharedPointer<service::ILibraryAppletProxy>* pProxy) -> nn::Result
        {
            auto timeToGiveUp = os::GetSystemTick() + os::Tick(TimeSpan::FromMilliSeconds(10000));
            auto timeToTryResolving = os::GetSystemTick() + os::Tick(TimeSpan::FromMilliSeconds(1000));
            bool resolving = false;
            for (;;)
            {
                NN_RESULT_TRY(g_AllSystemAppletProxyService->OpenLibraryAppletProxy(pProxy, 0, sf::NativeHandle(os::GetCurrentProcessHandle(), false), *pAttribute))
                    NN_RESULT_CATCH(nn::am::ResultNoAppletProxy)
                    {
                        auto currentTick = os::GetSystemTick();
                        // 対応するプロキシが見つからなかったとき
                        NN_ABORT_UNLESS(currentTick < timeToGiveUp, "No library applet is not found and give up");
                        if (resolving || currentTick < timeToTryResolving)
                        {
                            // すでに解決中なら、しばらく待ってリトライ
                            os::SleepThread(TimeSpan::FromMilliSeconds(100));
                            continue;
                        }
                        else
                        {
                            // まだ解決中でなかったら、resolver を使用して解決を試みる
                            NN_ABORT_UNLESS(g_LibraryAppletResolver, "No library applet is not found and library applet resolver is not set");
                            g_LibraryAppletResolver();
                            resolving = true;
                            continue;
                        }
                    }
                NN_RESULT_END_TRY
                NN_RESULT_SUCCESS;
            }
        }
    ));

    // その他のインターフェイスの取得
    NN_AM_GET_INTERFACE(service::ILibraryAppletSelfAccessor,
                        g_Interfaces.system.libraryAppletSelfAccessor,
                        g_Interfaces.system.libraryAppletProxy,
                        OpenLibraryAppletSelfAccessor);

    NN_AM_GET_INTERFACE(service::IProcessWindingController,
                        g_Interfaces.system.processWindingController,
                        g_Interfaces.system.libraryAppletProxy,
                        GetProcessWindingController);

    InitializeCommonInterfaces( GetLibraryAppletProxy() );
}

void FinalizeLibraryAppletInterfaces() NN_NOEXCEPT
{
    FinalizeCommonInterfaces();

    g_Interfaces.system.processWindingController.FinalizeHolder();
    g_Interfaces.system.libraryAppletSelfAccessor.FinalizeHolder();
    g_Interfaces.system.libraryAppletProxy.FinalizeHolder();
    g_AllSystemAppletProxyService.FinalizeHolder();
}

//----------

void InitializeOverlayAppletInterfaces() NN_NOEXCEPT
{
    // AppletServiceNameForSystem の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_HipcManager.InitializeShimLibraryHolder(&g_AllSystemAppletProxyService, service::AppletServiceNameForSystem) );

    // IOverlayAppletProxy の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interfaces.system.overlayAppletProxy.InitializeHolder(
        [](sf::SharedPointer<service::IOverlayAppletProxy>* pProxy) -> nn::Result
        {
            NN_AM_RETURN_UNLESS_RESULT_NO_APPLET_PROXY( g_AllSystemAppletProxyService->OpenOverlayAppletProxy(pProxy, 0, sf::NativeHandle(os::GetCurrentProcessHandle(), false)) );
        }
    ));

    // その他のインターフェイスの取得
    NN_AM_GET_INTERFACE(service::IOverlayFunctions,
                        g_Interfaces.system.overlayFunctions,
                        g_Interfaces.system.overlayAppletProxy,
                        GetOverlayFunctions);

    InitializeCommonInterfaces( GetOverlayAppletProxy() );
}

void FinalizeOverlayAppletInterfaces() NN_NOEXCEPT
{
    FinalizeCommonInterfaces();

    g_Interfaces.system.overlayFunctions.FinalizeHolder();
    g_Interfaces.system.overlayAppletProxy.FinalizeHolder();
    g_AllSystemAppletProxyService.FinalizeHolder();
}

//----------

bool InitializeApplicationInterfaces() NN_NOEXCEPT
{
    // AppletServiceNameForApplication の取得
    if (!g_HipcManager.InitializeShimLibraryHolder(&g_ApplicationProxyService, service::AppletServiceNameForApplication).IsSuccess())
    {
        return false;
    }

    // IApplicationProxy の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interfaces.application.applicationProxy.InitializeHolder(
        [](sf::SharedPointer<service::IApplicationProxy>* pProxy) -> nn::Result
        {
            NN_AM_RETURN_UNLESS_RESULT_NO_APPLET_PROXY( g_ApplicationProxyService->OpenApplicationProxy(pProxy, 0, sf::NativeHandle(os::GetCurrentProcessHandle(), false)) );
        }
    ));

    // その他のインターフェイスの取得
    NN_AM_GET_INTERFACE(service::IApplicationFunctions,
                        g_Interfaces.application.applicationFunctions,
                        g_Interfaces.application.applicationProxy,
                        GetApplicationFunctions);

    InitializeCommonInterfaces( GetApplicationProxy() );

    return true;
}

void FinalizeApplicationInterfaces() NN_NOEXCEPT
{
    FinalizeCommonInterfaces();

    g_Interfaces.application.applicationProxy.FinalizeHolder();
    g_ApplicationProxyService.FinalizeHolder();
}

//-----------------------------------------------------------------------------

void InitializeSystemApplicationInterfaces() NN_NOEXCEPT
{
    // AppletServiceNameForApplication の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_HipcManager.InitializeShimLibraryHolder(&g_AllSystemAppletProxyService, service::AppletServiceNameForSystem) );

    // IApplicationProxy の取得
    NN_ABORT_UNLESS_RESULT_SUCCESS( g_Interfaces.application.applicationProxy.InitializeHolder(
        [](sf::SharedPointer<service::IApplicationProxy>* pProxy) -> nn::Result
        {
            NN_AM_RETURN_UNLESS_RESULT_NO_APPLET_PROXY( g_AllSystemAppletProxyService->OpenSystemApplicationProxy(pProxy, 0, sf::NativeHandle(os::GetCurrentProcessHandle(), false)) );
        }
    ));

    // その他のインターフェイスの取得
    NN_AM_GET_INTERFACE(service::IApplicationFunctions,
                        g_Interfaces.application.applicationFunctions,
                        g_Interfaces.application.applicationProxy,
                        GetApplicationFunctions);

    InitializeCommonInterfaces( GetApplicationProxy() );
}

void FinalizeSystemApplicationInterfaces() NN_NOEXCEPT
{
    FinalizeCommonInterfaces();

    g_Interfaces.application.applicationProxy.FinalizeHolder();
    g_AllSystemAppletProxyService.FinalizeHolder();
}

//-----------------------------------------------------------------------------

applet::AppletResourceUserId GetAppletResourceUserId() NN_NOEXCEPT
{
    return g_AppletResourceUserId;
}

//-----------------------------------------------------------------------------

sf::SharedPointer<service::ISystemAppletControllerForDebug> CreateSystemAppletControllerForDebug() NN_NOEXCEPT
{
    class Allocator
    {
    private:
        sf::ProxyObjectAllocator<4> m_Impl = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
    public:
        Allocator() NN_NOEXCEPT
        {
            m_Impl.Initialize();
        }
        MemoryResource* GetMemoryResource() NN_NOEXCEPT
        {
            return m_Impl.GetMemoryResource();
        }
    };
    NN_FUNCTION_LOCAL_STATIC(Allocator, g_Allocator);
retry:
    auto ppService = nn::sf::CreateHipcProxyByName<service::IAllSystemAppletProxiesService>(g_Allocator.GetMemoryResource(), service::AppletServiceNameForSystem);
    NN_ABORT_UNLESS_RESULT_SUCCESS(ppService);
    auto pService = *ppService;
    sf::SharedPointer<service::ISystemAppletControllerForDebug> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(pService->GetSystemAppletControllerForDebug(&p));
    if (!p)
    {
        os::SleepThread(TimeSpan::FromMilliSeconds(100));
        goto retry;
    }
    return p;
}

}}  // namespace nn::ae

