﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/sf/sf_HipcServer.h>
#include <type_traits>
#include <new>
#include <nn/nn_SdkAssert.h>

#include "ahid_Hdr.h"
#include "ahid_HdrServiceName.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace ahid {
namespace hdr {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result HdrServer::Initialize() NN_NOEXCEPT
{
    m_HdrMutex.Initialize();
    m_HdrDevice.Initialize();
    m_HdrHandle.Initialize();

    m_pDeviceFilter         = 0;
    m_DeviceFilterEntries   = 0;

    Result result = InitializePort(0, HDR_SERVER_MAX_SESSIONS, HdrServiceName);

    if (result.IsSuccess())
    {
        m_HeapHandle = nn::lmem::CreateExpHeap(
                                                &m_SfHeapBuffer,
                                                sizeof(m_SfHeapBuffer),
                                                nn::lmem::CreationOption_NoOption
                                                );

        m_SfAllocator.Attach(m_HeapHandle);

        nn::os::CreateThread(
            &m_Thread,
            SessionThread,
            this,
            m_ThreadStack,
            HDR_SERVER_STACK_SIZE,
            NN_SYSTEM_THREAD_PRIORITY(hdr,HdrIpcServer)
            );

        nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(hdr,HdrIpcServer));

        nn::os::StartThread(&m_Thread);
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result HdrServer::Finalize() NN_NOEXCEPT
{
    RequestStop();

    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);

    m_HdrHandle.Finalize();
    m_HdrDevice.Finalize();
    m_HdrMutex.Finalize();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
nn::sf::SharedPointer<nn::ahid::hdr::ISession> HdrServer::GetDfcSession() NN_NOEXCEPT
{
    return CreateSession(true);
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
nn::sf::SharedPointer<nn::ahid::hdr::ISession> HdrServer::CreateSession(bool isDfcSession) NN_NOEXCEPT
{
    typedef nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy> Factory;
    return Factory::CreateSharedEmplaced<nn::ahid::hdr::ISession, HdrSession>(&m_SfAllocator, this, isDfcSession);
}


//////////////////////////////////////////////////////////////////////////////
Result HdrServer::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
{
    NN_UNUSED(portIndex);
    return AcceptImpl(pPort, CreateSession(false));
}


//////////////////////////////////////////////////////////////////////////////
void HdrServer::SessionThread(void *arg) NN_NOEXCEPT
{
    HdrServer *pThis = reinterpret_cast<HdrServer*>(arg);

    pThis->Start();
    pThis->LoopAuto();
}


}   // hdr
}   // ahid
}   // nn
