﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ahid_Hdr.h"

namespace nn {
namespace ahid {
namespace hdr {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result HdrHandle::Initialize()
{
    for (int i = 0; i < HdrHandleCountMax; i++)
    {
        m_Instance[i]   = 0;
        m_IsAttached[i] = false;
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result HdrHandle::Finalize()
{
    for (int i = 0; i < HdrHandleCountMax; i++)
    {
        DeviceHandle handle;
        Result result = GetHandleForIndex(i, &handle);

        if (result.IsSuccess())
        {
            DetachHandle(handle);
        }
    }

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result HdrHandle::AttachHandle(DeviceHandle* pHandle)
{
    for (int i = 0; i < HdrHandleCountMax; i++)
    {
        if (m_IsAttached[i] == false)
        {
            m_IsAttached[i] = true;

            // never generate a valid handle of 0
            if (m_Instance[i] == 0)
            {
                m_Instance[i]++;
            }

            *pHandle = EncodeHandle(i);

            return ResultSuccess();
        }
    }

    return ResultMaxEntries();
}


//////////////////////////////////////////////////////////////////////////////
Result HdrHandle::DetachHandle(DeviceHandle handle)
{
    int index = HandleIndex(handle);

    if (index < HdrHandleCountMax)
    {
        if (HandleIsValid(handle))
        {
            m_Instance[index]++;
            m_IsAttached[index] = false;

            return ResultSuccess();
        }
    }

    return ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
Result HdrHandle::GetIndexForHandle(DeviceHandle handle, int* pIndex)
{
    int index = HandleIndex(handle);

    if (index < HdrHandleCountMax)
    {
        if (HandleIsValid(handle))
        {
            *pIndex = index;

            return ResultSuccess();
        }
    }

    return ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
Result HdrHandle::GetHandleForIndex(int index, DeviceHandle* pHandle)
{
    if (index < HdrHandleCountMax)
    {
        if (m_IsAttached[index] == true)
        {
            *pHandle = EncodeHandle(index);

            return ResultSuccess();
        }
    }

    return ResultMaxEntries();
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
bool HdrHandle::HandleIsValid(DeviceHandle handle)
{
    int index  = static_cast<int>(handle >> HdrHandleIndexShift);

    if ((m_Instance[index] & HdrHandleInstanceMask) == (handle & HdrHandleInstanceMask))
    {
        return true;
    }

    return false;
}


//////////////////////////////////////////////////////////////////////////////
int HdrHandle::HandleIndex(DeviceHandle handle)
{
    return static_cast<int>(handle >> HdrHandleIndexShift);
}


//////////////////////////////////////////////////////////////////////////////
DeviceHandle HdrHandle::EncodeHandle(int index)
{
    return static_cast<DeviceHandle>(index << HdrHandleIndexShift) | (m_Instance[index] & HdrHandleInstanceMask);
}


}   // namespace hdr
}   // namespace ahid
}   // namespace nn
