﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasUserResourceCache.h>

#include "account_NasUserAdaptor.h"
#include "../detail/account_CacheUtil.h"
#include "../detail/account_PathUtil.h"
#include <nn/account/json/account_RapidJsonApi.h>
#include <nn/account/json/account_RapidJsonStream.h>

namespace nn {
namespace account {
namespace nas {

/* --------------------------------------------------------------------------------------------
    NasUserResourceCacheOperator
**/
bool NasUserResourceCacheOperator::IsExpired(int64_t) NN_NOEXCEPT
{
    // User リソースは失効しない
    return false;
}
void NasUserResourceCacheOperator::DeleteCacheData(const std::pair<detail::Uuid, TimeSpan>& data, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    detail::CacheUtil::DeleteCacheFile(data.first, storage);
}

/* --------------------------------------------------------------------------------------------
    NasUserResourceCache
**/

bool NasUserResourceCache::IsAvailable(const NintendoAccountId& tag) const NN_NOEXCEPT
{
    std::lock_guard<const Base> lock(*static_cast<const Base*>(this));
    if (Base::IsAvailableUnsafe(tag))
    {
        return true;
    }
    auto& storage = GetStorageRef();
    char path[128];
    detail::PathUtil::GetNasUserResourceCacheFilePath(path, sizeof(path), tag, storage.GetRootPath());
    size_t size;
    auto r = storage.GetFileSystem().GetSize(&size, path);
    NN_UNUSED(size);
    if (fs::ResultPathNotFound::Includes(r))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    return true;
}
Result NasUserResourceCache::Perpetuate(const NintendoAccountId& tag) NN_NOEXCEPT
{
    std::lock_guard<Base> lock(*static_cast<Base*>(this));

    auto& storage = GetStorageRef();
    char path[128];
    char from[128];

    std::pair<detail::Uuid, TimeSpan> cache;
    NN_RESULT_DO(Base::FindUnsafe(&cache, tag));
    if (!cache.first)
    {
#if !defined(NN_SDK_BUILD_RELEASE)
        detail::PathUtil::GetNasUserResourceCacheFilePath(path, sizeof(path), tag, storage.GetRootPath());
        size_t size;
        NN_SDK_ASSERT(storage.GetFileSystem().GetSize(&size, path).IsSuccess());
#endif
        NN_RESULT_SUCCESS;
    }

    // 永続化されていない場合のみ処理を行う
    NN_RESULT_DO(storage.GetFileSystem().Move(
        detail::PathUtil::GetNasUserResourceCacheFilePath(path, sizeof(path), tag, storage.GetRootPath()),
        detail::PathUtil::GetCachePath(from, sizeof(from), cache.first, storage.GetRootPath())));

    // キャッシュ側を無効化
    cache.first = detail::InvalidUuid;
    Base::StoreUnsafe(tag, 0, cache);
    NN_RESULT_SUCCESS;
}
Result NasUserResourceCache::Store(const NintendoAccountId& tag, detail::Uuid&& cacheId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(cacheId);
    auto movedCacheId = std::move(cacheId);
    cacheId = detail::InvalidUuid;
    NN_UTIL_SCOPE_EXIT
    {
        if (movedCacheId)
        {
            detail::CacheUtil::DeleteCacheFile(movedCacheId, GetStorageRef());
        }
    };

    std::lock_guard<Base> lock(*static_cast<Base*>(this));

    auto& storage = GetStorageRef();
    char path[128];

    detail::PathUtil::GetNasUserResourceCacheFilePath(path, sizeof(path), tag, storage.GetRootPath());
    size_t size;
    auto r = storage.GetFileSystem().GetSize(&size, path);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultPathNotFound::Includes(r), r);
    bool isPerpetuated = r.IsSuccess();

    std::pair<detail::Uuid, TimeSpan> cache(movedCacheId, os::GetSystemTick().ToTimeSpan());
    if (isPerpetuated)
    {
        char from[128];
        auto wlock = storage.AcquireWriterLock();
        NN_RESULT_DO(storage.GetFileSystem().Move(
            path,
            detail::PathUtil::GetCachePath(from, sizeof(from), movedCacheId, storage.GetRootPath())));
        cache.first = detail::InvalidUuid;
    }
    Base::StoreUnsafe(tag, 0, cache);
    movedCacheId = detail::InvalidUuid;
    NN_RESULT_SUCCESS;
}
Result NasUserResourceCache::GetTimeStamp(TimeSpan* pOut, const NintendoAccountId& tag) const NN_NOEXCEPT
{
    std::lock_guard<const Base> lock(*static_cast<const Base*>(this));

    std::pair<detail::Uuid, TimeSpan> cache;
    NN_RESULT_DO(Base::FindUnsafe(&cache, tag));
    *pOut = cache.second;
    NN_RESULT_SUCCESS;
}
Result NasUserResourceCache::Load(NasUserBase* pOut, const NintendoAccountId& tag, void* rawBuffer, size_t bufferSize) const NN_NOEXCEPT
{
    struct Buffer
    {
        char input[512];
        char string[512];
    };
    NN_RESULT_THROW_UNLESS(bufferSize >= sizeof(Buffer), ResultInsufficientBuffer());

    std::lock_guard<const Base> lock(*static_cast<const Base*>(this));

    auto& storage = GetStorageRef();
    char path[128];
    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    std::pair<detail::Uuid, TimeSpan> cache;
    auto r = Base::FindUnsafe(&cache, tag);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || detail::DefaultCacheResult::ResultNotExist::Includes(r), r);
    if (r.IsSuccess() && cache.first)
    {
        detail::PathUtil::GetCachePath(path, sizeof(path), cache.first, storage.GetRootPath());
    }
    else
    {
        detail::PathUtil::GetNasUserResourceCacheFilePath(path, sizeof(path), tag, storage.GetRootPath());
    }

    NasUserCacheAdaptor adaptor(pOut);
    json::BufferedFileInputStreamForRapidJson stream(path, storage, nullptr);
    stream.SetInputBuffer(buffer->input, sizeof(buffer->input));
    stream.SetStringBuffer(buffer->string, sizeof(buffer->string));
    NN_RESULT_TRY(json::ImportJsonByRapidJson(adaptor, stream, nullptr))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(detail::DefaultCacheResult::ResultNotExist());
        }
    NN_RESULT_END_TRY
    NN_RESULT_DO(adaptor.Adapt());
    NN_RESULT_SUCCESS;
}
void NasUserResourceCache::Invalidate(const NintendoAccountId& tag) NN_NOEXCEPT
{
    std::lock_guard<Base> lock(*static_cast<Base*>(this));
    InvalidateVolatileCacheUnsafe(tag);
    char path[128];
    auto r = GetStorageRef().GetFileSystem().Delete(
        detail::PathUtil::GetNasUserResourceCacheFilePath(path, sizeof(path), tag, GetStorageRef().GetRootPath()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
}

}
}
} // ~namespace nn::account::nas
