﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/nas/account_NasOperator.h>

#include "account_NasLoginAdaptor.h"
#include "account_NasCredentialHolder.h"
#include "account_NasResourceResolver.h"
#include "../baas/account_BaasProfileSynchronizer.h"
#include "../detail/account_CacheUtil.h"
#include "../http/account_OAuthUtil.h"
#include <nn/account/baas/account_BaasLoginCache.h>
#include <nn/account/nas/account_ResultForNas.h>
#include <nn/account/http/account_SimpleDownloader.h>
#include <nn/account/account_CachedNintendoAccountInfoTypes.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_RuntimeResource.h>

#include <algorithm>

#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace nn {
namespace account {
namespace nas {

namespace
{

Result HandleInteractiveAuthorizationResult(NasOperator& nasOp, const Uid& user, Result&& r) NN_NOEXCEPT
{
    NN_RESULT_TRY(r)
    NN_RESULT_CATCH(ResultNasAuthorizationStatusAccessDenied)
    {
        // 一般的なキャンセル
        NN_RESULT_THROW(ResultCancelledByUser());
    }
    NN_RESULT_CATCH(ResultNasAuthorizationStatusAccessDeniedIdTokenHintInvalid)
    {
        // 不正な id_token_hint を送った場合 = 認証情報が破損している場合
        // → SIGLO-83428 で何もしないように変更
        NN_DETAIL_ACCOUNT_WARN("Invalid ID token hint is specified for Nintendo Account login.\n");
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH(ResultNasAuthorizationStatusAccessDeniedUserDeleted)
    {
        // NA が削除されている
        if (user)
        {
            auto availability = nasOp.CheckAvailability(user);
            if (availability.IsSuccess() || ResultNintendoAccountStateInteractionRequired::Includes(availability))
            {
                // ユーザーの状態を「削除済み」に更新
                NN_RESULT_DO(nasOp.DegradeLinkedNintendoAccountUserState(user, baas::NintendoAccountUserState_Deleted));
                NN_RESULT_DO(nasOp.CheckAvailability(user));
            }
        }
        NN_RESULT_RETHROW;
    }
    NN_RESULT_CATCH(ResultNasAuthorizationStatusLoginRequiredUserDifferentFromIdTokenHint)
    {
        // 予期しない NA でログイン
        NN_RESULT_THROW(ResultUserInputDenied());
    }
    NN_RESULT_CATCH(ResultNasAuthorizationStatusConsentRequired)
    {
        // 認可拒否
        NN_RESULT_THROW(ResultCancelledByUser());
    }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

Result HandleImplicitAuthorizationResult(Result&& r) NN_NOEXCEPT
{
    NN_RESULT_TRY(r)
    // 次のケースはアクセストークン取得時に検出されるはずなので、エラーハンドリングしない。
    // - ResultNasAuthorizationStatusLoginRequiredUserNotLoggedIn
    // - ResultNasAuthorizationStatusInteractionRequiredUserBanned
    // - ResultNasAuthorizationStatusInteractionRequiredUserSuspended
    // - ResultNasAuthorizationStatusInteractionRequiredUserTermsAgreementRequired
    NN_RESULT_CATCH(ResultNasAuthorizationStatusInteractionRequired)
    {
        // prompt=consent での対話型ログインが必要
        NN_RESULT_THROW(ResultNintendoAccountAuthorizationInteractionRequired());
    }
    NN_RESULT_CATCH(ResultNasAuthorizationStatusConsentRequired)
    {
        // prompt=consent での対話型ログインが必要
        NN_RESULT_THROW(ResultNintendoAccountAuthorizationInteractionRequired());
    }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::nas::<anonymous>

NasOperator::NasOperator(
    nas::NasUserResourceCache& nasUserResourceCache,
    baas::BaasOperator& baasOperator,
    ndas::NdasOperator& ndasOperator,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_NasLoginDriver(ndasOperator, storage)
    , m_NasUserDriver(storage)
    , m_NasOp2Driver(storage)
    , m_NdasOperator(ndasOperator)
    , m_BaasOp(baasOperator)
    , m_NasUserResourceCache(nasUserResourceCache)
    , m_Storage(storage)
    , m_PkceSessionCache(storage)
{
}

/* --------------------------------------------------------------------------------------------
    非通信系
*/

Result NasOperator::CheckAvailability(const Uid& user) const NN_NOEXCEPT
{
    NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));
    NN_RESULT_THROW_UNLESS(NasCredentialHolder::HasCredential(naId, m_Storage), ResultNintendoAccountLinkageRequired());
    NN_RESULT_SUCCESS;
}
Result NasOperator::GetLinkedNintendoAccountId(NintendoAccountId* pOutId, const Uid& user) const NN_NOEXCEPT
{
    return m_BaasOp.GetLinkedNintendoAccountId(pOutId, user);
}
Result NasOperator::DegradeLinkedNintendoAccountUserState(const Uid& user, baas::NintendoAccountUserState state) NN_NOEXCEPT
{
    return m_BaasOp.DegradeLinkedNintendoAccountUserState(user, state);
}

Result NasOperator::GetRequestForLinkage(
    RequestUrl* pOutRequest, const http::CodeVerifier& codeVerifier, const State& state,
    NintendoAccountAuthorizationPageTheme theme) const NN_NOEXCEPT
{
    auto clientInfo = NasResourceResolver::GetClientIdForAccountSystem();
    NN_RESULT_DO(m_NasLoginDriver.GetUrlForAuthorizationRequest(pOutRequest->url, sizeof(pOutRequest->url), clientInfo, codeVerifier, state, theme));
    NN_RESULT_SUCCESS;
}
Result NasOperator::SuspendPkceSessionForLinkage(detail::Uuid* pOutSessionId, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
{
    return m_PkceSessionCache.StoreForLinkage(pOutSessionId, user, prop);
}
Result NasOperator::ResumePkceSessionForLinkage(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    return m_PkceSessionCache.ExtractForLinkage(pOutProp, user, sessionId);
}
Result NasOperator::GetRequestForCredentialUpdate(
    RequestUrl* pOutRequest,
    const Uid& user, const http::CodeVerifier& codeVerifier, const State& state) const NN_NOEXCEPT
{
    NintendoAccountId id;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&id, user));
    auto clientInfo = NasResourceResolver::GetClientIdForAccountSystem();
    NN_RESULT_DO(m_NasLoginDriver.GetUrlForAuthorizationRequestWithIdTokenHint(pOutRequest->url, sizeof(pOutRequest->url), id, clientInfo, codeVerifier, state));
    NN_RESULT_SUCCESS;
}
Result NasOperator::SuspendPkceSessionForCredentialUpdate(detail::Uuid* pOutSessionId, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
{
    return m_PkceSessionCache.StoreForCredentialUpdate(pOutSessionId, user, prop);
}
Result NasOperator::ResumePkceSessionForCredentialUpdate(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    return m_PkceSessionCache.ExtractForCredentialUpdate(pOutProp, user, sessionId);
}
Result NasOperator::GetRequestForNnidLinkage(RequestUrl* pOutRequest, const Uid& user, const State& state) NN_NOEXCEPT
{
    NintendoAccountId id;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&id, user));
    auto clientInfo = NasResourceResolver::GetClientIdForAccountSystem();
    NN_RESULT_DO(m_NasLoginDriver.GetUrlForNnidLinkageRequest(pOutRequest->url, sizeof(pOutRequest->url), id, clientInfo, state));
    NN_RESULT_SUCCESS;
}
Result NasOperator::SuspendPkceSessionForNnidLinkage(detail::Uuid* pOutSessionId, const Uid& user, const NasPkceProperty& prop) NN_NOEXCEPT
{
    return m_PkceSessionCache.StoreForNnidLinkage(pOutSessionId, user, prop);
}
Result NasOperator::ResumePkceSessionForNnidLinkage(NasPkceProperty* pOutProp, const Uid& user, const detail::Uuid& sessionId) NN_NOEXCEPT
{
    return m_PkceSessionCache.ExtractForNnidLinkage(pOutProp, user, sessionId);
}
Result NasOperator::GetRequestForApplicationAuthorizationRequest(
    RequestUrl* pOutRequest,
    const Uid& user, const NasClientInfo& clientInfo, const Scope& scope, const State& state, const Nonce& nonce) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(clientInfo);
    NN_SDK_ASSERT(scope.IsValid());
    NN_SDK_ASSERT(state.IsValid());
    NN_SDK_ASSERT(nonce.IsValid());

    NintendoAccountId id;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&id, user));
    NN_RESULT_DO(m_NasLoginDriver.GetUrlForApplicationAuthorizationRequest(pOutRequest->url, sizeof(pOutRequest->url), id, clientInfo, scope, state, nonce));
    NN_RESULT_SUCCESS;
}
Result NasOperator::GetRequestForGuestLogin(RequestUrl* pOutRequest, const State& state) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(state.IsValid());

    auto clientInfo = NasResourceResolver::GetClientIdForAccountSystem();
    NN_RESULT_DO(m_NasLoginDriver.GetUrlForGuestLoginRequest(pOutRequest->url, sizeof(pOutRequest->url), clientInfo, state));
    NN_RESULT_SUCCESS;
}
Result NasOperator::GetTimeSpanSinceLastUpdateOfUserResourceCache(TimeSpan* pOut, const Uid& user) NN_NOEXCEPT
{
    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));
    TimeSpan timeStamp;
    NN_RESULT_DO(m_NasUserResourceCache.GetTimeStamp(&timeStamp, naId));
    *pOut = (os::GetSystemTick().ToTimeSpan() - timeStamp);
    NN_RESULT_SUCCESS;
}
Result NasOperator::LoadUserResourceCache(NintendoAccountId* pOutId, NasUserBase* pOutBase, const Uid& user, void* rawBuffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(bufferSize >= RequiredBufferSizeForCachedNintendoAccountInfo, ResultInsufficientBuffer());

    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));
    NN_RESULT_DO(m_NasUserResourceCache.Load(pOutBase, naId, rawBuffer, bufferSize));
    *pOutId = naId;
    NN_RESULT_SUCCESS;
}
Result NasOperator::LoadUserResourceCacheForApplication(NintendoAccountId* pOutId, NasUserBaseForApplication* pOutBase, const Uid& user, void* rawBuffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(bufferSize >= RequiredBufferSizeForCachedNintendoAccountInfo, ResultInsufficientBuffer());
    NN_UNUSED(bufferSize);

    struct Buffer
    {
        NasUserBase userBase;
        char ioBuffer[1024];
    };
    NN_STATIC_ASSERT(NN_ALIGNOF(Buffer) == 1);
    NN_STATIC_ASSERT(sizeof(Buffer) <= RequiredBufferSizeForCachedNintendoAccountInfo);
    Buffer* buffer = reinterpret_cast<Buffer*>(rawBuffer);

    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));
    NN_RESULT_DO(m_NasUserResourceCache.Load(&buffer->userBase, naId, buffer->ioBuffer, bufferSize));

    *pOutId = naId;
    strncpy(pOutBase->birthday, buffer->userBase.birthday, sizeof(pOutBase->birthday));
    strncpy(pOutBase->gender, buffer->userBase.gender, sizeof(pOutBase->gender));
    strncpy(pOutBase->language, buffer->userBase.language, sizeof(pOutBase->language));
    strncpy(pOutBase->country, buffer->userBase.country, sizeof(pOutBase->country));
    strncpy(pOutBase->region, buffer->userBase.region, sizeof(pOutBase->region));
    strncpy(pOutBase->timezone, buffer->userBase.timezone, sizeof(pOutBase->timezone));
    NN_RESULT_SUCCESS;
}
void NasOperator::InvalidateUserResourceCacheTimestamp(const Uid& user) NN_NOEXCEPT
{
    NintendoAccountId naId;
    auto r = m_BaasOp.GetLinkedNintendoAccountId(&naId, user);
    if (r.IsSuccess())
    {
        m_NasUserResourceCache.InvalidateVolatileCache(naId);
    }
}

Result NasOperator::GetTimeSpanSinceLastUpdateOfNetworkServiceLicenseCache(TimeSpan* pOut, const Uid& user) NN_NOEXCEPT
{
    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));
    TimeSpan timeStamp;
    NN_RESULT_DO(m_NasOp2MembershipCache.GetTimeStamp(&timeStamp, naId));
    *pOut = (os::GetSystemTick().ToTimeSpan() - timeStamp);
    NN_RESULT_SUCCESS;
}

Result NasOperator::GetNetworkServiceLicenseCache(NetworkServiceLicense* pOutLicense, time::PosixTime* pOutExpiration, const Uid& user) const NN_NOEXCEPT
{
    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));
    NN_RESULT_DO(m_NasOp2MembershipCache.Get(pOutLicense, pOutExpiration, naId));
    NN_RESULT_SUCCESS;
}
void NasOperator::InvalidateNetworkServiceLicenseCache(const Uid& user) NN_NOEXCEPT
{
    NintendoAccountId naId;
    auto r = m_BaasOp.GetLinkedNintendoAccountId(&naId, user);
    if (r.IsSuccess())
    {
        m_NasOp2MembershipCache.Invalidate(naId);
    }
}

/* --------------------------------------------------------------------------------------------
    通信系
*/
Result NasOperator::GetCredentialFromAuthorizationCode(
    NintendoAccountId* pOutNaId, NasCredentialCache *pOutCredentialCache,
    const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForNintendoAccount(resource, pCancellable));

    auto clientInfo = NasResourceResolver::GetClientIdForAccountSystem();
    NasCredentialCache credentialCache;
    NasAccessTokenCache accessTokenCache;
    NN_RESULT_DO(m_NasLoginDriver.AcquirePrimaryTokens(
        &credentialCache, &accessTokenCache,
        clientInfo, codeCacheId, codeVerifier,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));
    bool success = false;
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            detail::CacheUtil::DeleteCacheFile(credentialCache.refreshTokenCacheId, m_Storage);
            detail::CacheUtil::DeleteCacheFile(credentialCache.idTokenCacheId, m_Storage);
        }
        detail::CacheUtil::DeleteCacheFile(accessTokenCache.cacheId, m_Storage);
    };

    NintendoAccountId naId;
    NN_RESULT_DO(m_NasUserDriver.AcquireUserResourceCache(
        &naId, m_NasUserResourceCache,
        accessTokenCache.cacheId,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    *pOutNaId = naId;
    *pOutCredentialCache = credentialCache;
    success = true;
    NN_RESULT_SUCCESS;
}
Result NasOperator::Link(
    bool* pOutIsNetworkServiceAccountReplaced,
    const Uid& user, const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NintendoAccountId naId;
    NasCredentialCache credentialCache;
    NN_RESULT_DO(GetCredentialFromAuthorizationCode(&naId, &credentialCache, codeVerifier, codeCacheId, resource, pCancellable));
    NN_UTIL_SCOPE_EXIT
    {
        detail::CacheUtil::DeleteCacheFile(credentialCache.refreshTokenCacheId, m_Storage);
        detail::CacheUtil::DeleteCacheFile(credentialCache.idTokenCacheId, m_Storage);
    };

    NN_RESULT_DO(m_BaasOp.LinkWithNintendoAccount(
        pOutIsNetworkServiceAccountReplaced,
        user, naId, credentialCache, codeVerifier, resource, pCancellable));
    NN_RESULT_SUCCESS;
}

Result NasOperator::UpdateCredential(
    const Uid& user, const http::CodeVerifier& codeVerifier, const detail::Uuid& codeCacheId,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NintendoAccountId naId;
    NasCredentialCache credentialCache;
    NN_RESULT_DO(GetCredentialFromAuthorizationCode(&naId, &credentialCache, codeVerifier, codeCacheId, resource, pCancellable));
    NN_UTIL_SCOPE_EXIT
    {
        detail::CacheUtil::DeleteCacheFile(credentialCache.refreshTokenCacheId, m_Storage);
        detail::CacheUtil::DeleteCacheFile(credentialCache.idTokenCacheId, m_Storage);
    };

    NN_RESULT_DO(m_BaasOp.UpdateLinkageWithNintendoAccount(
        user, naId, credentialCache, codeVerifier, resource, pCancellable));
    NN_RESULT_SUCCESS;
}

Result NasOperator::GetClientInfo(
    NasClientInfo* pOut, const detail::ApplicationInfo& appInfo,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(appInfo);
    ndas::ApplicationMeta meta;
    NN_RESULT_DO(m_NdasOperator.AuthenticateApplication(appInfo, resource, pCancellable));
    NN_RESULT_DO(m_NdasOperator.GetApplicationMetaCache(&meta, appInfo));
    *pOut = meta.nasClientInfo;
    NN_RESULT_SUCCESS;
}

Result NasOperator::AcrequireAccessToken(
    NasAccessTokenCache *pOut, const Uid& user, const Scope& scope,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NintendoAccountId id;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&id, user));

    NN_RESULT_DO(m_NdasOperator.AuthenticateServiceForNintendoAccount(resource, pCancellable));

    NN_RESULT_TRY(m_NasLoginDriver.RefreshAccessToken(
        pOut, id, NasResourceResolver::GetClientIdForAccountSystem(), scope,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable))
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidGrant)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_OtherButInteractionRequired));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidGrantUserDeleted)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_Deleted));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidGrantUserBanned)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_Banned));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidGrantUserSuspended)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_Suspended));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidGrantUserWithdrawn)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_Withdrawn));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidGrantUserTermsAgreementRequired)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_TermsAgreementRequired));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_CATCH(ResultNasTokenRequestStatusInvalidScopeScopeTokenNotAuthorized)
    {
        NN_RESULT_DO(m_BaasOp.DegradeLinkedNintendoAccountUserState(
            user, baas::NintendoAccountUserState_ReauthorizationRequired));
        NN_RESULT_DO(m_BaasOp.CheckAvailability(user));
    }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

Result NasOperator::AcquireApplicationAuthorizationWithoutPrompt(
    NasApplicationAuthorization* pOut,
    const Uid& user,
    const NasClientInfo& clientInfo, const Scope& scope, const State& state, const Nonce& nonce,
    const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // アクセストークンの取得
    NasAccessTokenCache accessTokenCache;
    NN_RESULT_DO(AcrequireAccessToken(&accessTokenCache, user, NasResourceResolver::GetScopeForAccountSystem(), resource, pCancellable));
    NN_UTIL_SCOPE_EXIT
    {
        detail::CacheUtil::DeleteCacheFile(accessTokenCache.cacheId, m_Storage);
    };

    // 認可コードとIDトークンの取得
    NN_RESULT_DO(HandleImplicitAuthorizationResult(
        m_NasLoginDriver.AcquireApplicationAuthorizationWithoutPrompt(
            pOut, accessTokenCache.cacheId, clientInfo, scope, state, nonce,
            resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable)));
    NN_RESULT_SUCCESS;
}

Result NasOperator::UpdateUserResourceCache(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // アクセストークンの取得
    NasAccessTokenCache accessTokenCache;
    NN_RESULT_DO(AcrequireAccessToken(&accessTokenCache, user, NasResourceResolver::GetScopeForAccountSystem(), resource, pCancellable));
    NN_UTIL_SCOPE_EXIT
    {
        detail::CacheUtil::DeleteCacheFile(accessTokenCache.cacheId, m_Storage);
    };

    // ユーザーリソースの取得
    NintendoAccountId naId;
    NN_RESULT_DO(m_NasUserDriver.AcquireUserResourceCache(
        &naId, m_NasUserResourceCache,
        accessTokenCache.cacheId,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    // ユーザーリソースの更新の確定
    auto fsLock = m_Storage.AcquireWriterLock();
    NN_RESULT_DO(m_Storage.Commit());
    NN_RESULT_SUCCESS;
}

Result NasOperator::UpdateNetworkServiceLicenseCache(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    // アクセストークンの取得
    NasAccessTokenCache accessTokenCache;
    NN_RESULT_DO(AcrequireAccessToken(&accessTokenCache, user, NasResourceResolver::GetScopeForOp2Membership(), resource, pCancellable));
    NN_UTIL_SCOPE_EXIT
    {
        detail::CacheUtil::DeleteCacheFile(accessTokenCache.cacheId, m_Storage);
    };

    // Op2Membership の取得
    NintendoAccountId naId;
    NN_RESULT_DO(m_BaasOp.GetLinkedNintendoAccountId(&naId, user));

    NetworkServiceLicense license;
    time::PosixTime expiration;
    NN_RESULT_DO(m_NasOp2Driver.GetMembership(
        &license, &expiration,
        naId, accessTokenCache.cacheId,
        resource.curlHandle, resource.buffer.address, resource.buffer.size, pCancellable));

    m_NasOp2MembershipCache.Store(naId, license, expiration);
    NN_RESULT_SUCCESS;
}

Result NasOperator::TryRecoverNintendoAccountUserStateImplicitly(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    if (CheckAvailability(user).IsSuccess())
    {
        NN_DETAIL_ACCOUNT_INFO("NA of Specified user is already available\n");
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(UpdateUserResourceCache(user, resource, pCancellable));

    NN_RESULT_DO(m_BaasOp.RecoveryLinkedNintendoAccountUserState(user));

    auto r = UpdateNetworkServiceLicenseCache(user, resource, pCancellable);
    // OP2 の可用性次第で失敗する可能性があるので、通信エラーは許容する。
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkCommunicationError::Includes(r), r);

    NN_SDK_ASSERT(CheckAvailability(user).IsSuccess());
    NN_RESULT_SUCCESS;
}

/* ----------------------------------------------------------------------------------------------------------------------------
    NasLinkageProcedure
*/
NasLinkageProcedure::NasLinkageProcedure(const Uid& user, NasOperator& nasOp) NN_NOEXCEPT
    : m_NasOp(nasOp)
    , m_User(user)
    , m_ProcState(http::ProcedureState_Initialized)
    , m_CodeCacheId(detail::InvalidUuid)
{
}
NasLinkageProcedure::~NasLinkageProcedure() NN_NOEXCEPT
{
    if (m_CodeCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CodeCacheId, m_NasOp.GetStorageRef());
    }
}
Result NasLinkageProcedure::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri, NintendoAccountAuthorizationPageTheme theme) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    http::GenerateCodeVerifier(&m_Property.codeVerifier);
    http::GenerateStateString(m_Property.state.data, sizeof(m_Property.state.data));
    m_IsNetworkServiceAccountReplaced = false;

    // URL 生成
    NN_RESULT_DO(m_NasOp.GetRequestForLinkage(pOutRequestUrl, m_Property.codeVerifier, m_Property.state, theme));
    std::strncpy(m_Property.callback.uri, NasResourceResolver::GetClientIdForAccountSystem().redirectUri, sizeof(m_Property.callback.uri));
    *pOutCallbackUri = m_Property.callback;
    NN_RESULT_SUCCESS;
}
Result NasLinkageProcedure::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NasServiceAuthorizationAdaptor adaptor(m_Property.callback.uri, m_Property.state, m_NasOp.GetStorageRef());
    NN_RESULT_THROW_UNLESS(adaptor.Parse(response, responseBufferSize), ResultNasDataBroken());
    NN_RESULT_DO(HandleInteractiveAuthorizationResult(m_NasOp, m_User, adaptor.Adapt()));

    NN_RESULT_DO(adaptor.PullAuthorizationCode(&m_CodeCacheId));
    NN_RESULT_SUCCESS;
}
Result NasLinkageProcedure::ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CodeCacheId);
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.Link(&m_IsNetworkServiceAccountReplaced, m_User, m_Property.codeVerifier, m_CodeCacheId, resource, pCancellable));

    auto r = m_NasOp.UpdateNetworkServiceLicenseCache(m_User, resource, pCancellable);
    // OP2 の可用性次第で失敗する可能性があるので、通信エラーは許容する。
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkCommunicationError::Includes(r), r);

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}
Result NasLinkageProcedure::Suspend(detail::Uuid* pOutSessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.SuspendPkceSessionForLinkage(pOutSessionId, m_User, m_Property));
    m_ProcState = http::ProcedureState_Suspended;
    NN_RESULT_SUCCESS;
}
Result NasLinkageProcedure::Resume(const detail::Uuid& sessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.ResumePkceSessionForLinkage(&m_Property, m_User, sessionId));
    m_ProcState = http::ProcedureState_Resumed;
    NN_RESULT_SUCCESS;
}

/* ----------------------------------------------------------------------------------------------------------------------------
    NasCredentialUpdateProcedure
*/
NasCredentialUpdateProcedure::NasCredentialUpdateProcedure(const Uid& user, NasOperator& nasOp) NN_NOEXCEPT
    : m_NasOp(nasOp)
    , m_User(user)
    , m_ProcState(http::ProcedureState_Initialized)
    , m_CodeCacheId(detail::InvalidUuid)
{
}
NasCredentialUpdateProcedure::~NasCredentialUpdateProcedure() NN_NOEXCEPT
{
    if (m_CodeCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CodeCacheId, m_NasOp.GetStorageRef());
    }
}
Result NasCredentialUpdateProcedure::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    http::GenerateCodeVerifier(&m_Property.codeVerifier);
    http::GenerateStateString(m_Property.state.data, sizeof(m_Property.state.data));

    // URL 生成
    NN_RESULT_DO(m_NasOp.GetRequestForCredentialUpdate(pOutRequestUrl, m_User, m_Property.codeVerifier, m_Property.state));
    std::strncpy(m_Property.callback.uri, NasResourceResolver::GetClientIdForAccountSystem().redirectUri, sizeof(m_Property.callback.uri));
    *pOutCallbackUri = m_Property.callback;
    NN_RESULT_SUCCESS;
}
Result NasCredentialUpdateProcedure::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NasServiceAuthorizationAdaptor adaptor(m_Property.callback.uri, m_Property.state, m_NasOp.GetStorageRef());
    NN_RESULT_THROW_UNLESS(adaptor.Parse(response, responseBufferSize), ResultNasDataBroken());
    NN_RESULT_DO(HandleInteractiveAuthorizationResult(m_NasOp, m_User, adaptor.Adapt()));

    NN_RESULT_DO(adaptor.PullAuthorizationCode(&m_CodeCacheId));
    NN_RESULT_SUCCESS;
}
Result NasCredentialUpdateProcedure::ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_CodeCacheId);
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.UpdateCredential(m_User, m_Property.codeVerifier, m_CodeCacheId, resource, pCancellable));

    auto r = m_NasOp.UpdateNetworkServiceLicenseCache(m_User, resource, pCancellable);
    // OP2 の可用性次第で失敗する可能性があるので、通信エラーは許容する。
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkCommunicationError::Includes(r), r);

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}
Result NasCredentialUpdateProcedure::Suspend(detail::Uuid* pOutSessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.SuspendPkceSessionForCredentialUpdate(pOutSessionId, m_User, m_Property));
    m_ProcState = http::ProcedureState_Suspended;
    NN_RESULT_SUCCESS;
}
Result NasCredentialUpdateProcedure::Resume(const detail::Uuid& sessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.ResumePkceSessionForCredentialUpdate(&m_Property, m_User, sessionId));
    m_ProcState = http::ProcedureState_Resumed;
    NN_RESULT_SUCCESS;
}

/* ----------------------------------------------------------------------------------------------------------------------------
    NasNnidLinkageProcedure
*/
NasNnidLinkageProcedure::NasNnidLinkageProcedure(const Uid& user, NasOperator& nasOp) NN_NOEXCEPT
    : m_NasOp(nasOp)
    , m_User(user)
    , m_ProcState(http::ProcedureState_Initialized)
{
}
Result NasNnidLinkageProcedure::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    std::memset(m_Property.codeVerifier.data, 0, sizeof(m_Property.codeVerifier.data));
    http::GenerateStateString(m_Property.state.data, sizeof(m_Property.state.data));

    // URL 生成
    NN_RESULT_DO(m_NasOp.GetRequestForNnidLinkage(pOutRequestUrl, m_User, m_Property.state));
    std::strncpy(m_Property.callback.uri, NasResourceResolver::GetClientIdForAccountSystem().redirectUri, sizeof(m_Property.callback.uri));
    *pOutCallbackUri = m_Property.callback;
    NN_RESULT_SUCCESS;
}
Result NasNnidLinkageProcedure::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NasServiceAuthorizationAdaptor adaptor(m_Property.callback.uri, m_Property.state, m_NasOp.GetStorageRef());
    NN_RESULT_THROW_UNLESS(adaptor.Parse(response, responseBufferSize), ResultNasDataBroken());
    NN_RESULT_DO(HandleInteractiveAuthorizationResult(m_NasOp, m_User, adaptor.Adapt()));
    NN_RESULT_SUCCESS;
}
Result NasNnidLinkageProcedure::ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.UpdateUserResourceCache(m_User, resource, pCancellable));

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}
Result NasNnidLinkageProcedure::Suspend(detail::Uuid* pOutSessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.SuspendPkceSessionForNnidLinkage(pOutSessionId, m_User, m_Property));
    m_ProcState = http::ProcedureState_Suspended;
    NN_RESULT_SUCCESS;
}
Result NasNnidLinkageProcedure::Resume(const detail::Uuid& sessionId) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.ResumePkceSessionForNnidLinkage(&m_Property, m_User, sessionId));
    m_ProcState = http::ProcedureState_Resumed;
    NN_RESULT_SUCCESS;
}

/* ----------------------------------------------------------------------------------------------------------------------------
    NasApplicationAuthorizationTaskHolder
*/
NasApplicationAuthorizationProcedure::NasApplicationAuthorizationProcedure(
    const Uid& user, const detail::ApplicationInfo& appInfo, NasOperator& nasOp) NN_NOEXCEPT
    : m_NasOp(nasOp)
    , m_User(user)
    , m_ProcState(http::ProcedureState_Initialized)
    , m_AppInfo(appInfo)
    , m_NasClientInfo(nas::InvalidNasClientInfo)
    , m_Mapped(false)
    , m_pResource(nullptr)
{
}
NasApplicationAuthorizationProcedure::NasApplicationAuthorizationProcedure(
    const Uid& user, const NasClientInfo& clientInfo, NasOperator& nasOp) NN_NOEXCEPT
    : m_NasOp(nasOp)
    , m_User(user)
    , m_ProcState(http::ProcedureState_Initialized)
    , m_AppInfo(detail::InvalidApplicationInfo)
    , m_NasClientInfo(clientInfo)
    , m_Mapped(false)
    , m_pResource(nullptr)
{
}
NasApplicationAuthorizationProcedure::~NasApplicationAuthorizationProcedure() NN_NOEXCEPT
{
    if (m_Mapped)
    {
        std::memset(m_pResource, 0x3A, sizeof(*m_pResource));
        m_Memory.Unmap();
    }
}

void NasApplicationAuthorizationProcedure::AttachTransferMemory(os::NativeHandle transferMemoryHandle, size_t size, bool managed) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size >= RequiredMemorySize);
    m_Memory.Attach(size, transferMemoryHandle, managed);
}
Result NasApplicationAuthorizationProcedure::Initialize(const NintendoAccountAuthorizationRequestParameters& params) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Mapped);
    NN_SDK_ASSERT(m_pResource == nullptr);

    void* address;
    NN_RESULT_DO(m_Memory.Map(&address, os::MemoryPermission_ReadWrite));
    m_Mapped = true;

    m_pResource = new(address) Resource;

    std::strncpy(m_pResource->scope.data, params.scope, sizeof(m_pResource->scope.data));
    NN_RESULT_THROW_UNLESS(m_pResource->scope.IsValid(), ResultInvalidStringFormat());

    std::strncpy(m_pResource->state.data, params.state, sizeof(m_pResource->state.data));
    NN_RESULT_THROW_UNLESS(m_pResource->state.IsValid(), ResultInvalidStringFormat());

    std::strncpy(m_pResource->nonce.data, params.nonce, sizeof(m_pResource->nonce.data));
    NN_RESULT_THROW_UNLESS(m_pResource->nonce.IsValid(), ResultInvalidStringFormat());

    std::memset(&m_pResource->authorization.code, 0xDC, sizeof(m_pResource->authorization.code));
    std::memset(&m_pResource->authorization.idToken, 0x2F, sizeof(m_pResource->authorization.idToken));
    NN_RESULT_SUCCESS;
}
Result NasApplicationAuthorizationProcedure::ExecuteImplicitly(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_DO(ExecutePreprocess(resource, pCancellable));

    NN_RESULT_DO(m_NasOp.AcquireApplicationAuthorizationWithoutPrompt(
        &m_pResource->authorization,
        m_User, m_NasClientInfo, m_pResource->scope, m_pResource->state, m_pResource->nonce,
        resource, pCancellable));

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}
Result NasApplicationAuthorizationProcedure::ExecutePreprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());
    if (!m_NasClientInfo)
    {
        NN_RESULT_THROW_UNLESS(m_AppInfo, ResultNotSupported());
        NasClientInfo clientInfo;
        NN_RESULT_DO(m_NasOp.GetClientInfo(&clientInfo, m_AppInfo, resource, pCancellable));
        NN_RESULT_THROW_UNLESS(clientInfo, ResultInvalidProtocolAccess());
        m_NasClientInfo = clientInfo;
    }
    NN_RESULT_SUCCESS;
}
Result NasApplicationAuthorizationProcedure::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_NasClientInfo, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    // URL 生成
    NN_RESULT_DO(m_NasOp.GetRequestForApplicationAuthorizationRequest(
        pOutRequestUrl,
        m_User, m_NasClientInfo, m_pResource->scope, m_pResource->state, m_pResource->nonce));
    std::strncpy(pOutCallbackUri->uri, m_NasClientInfo.redirectUri, sizeof(pOutCallbackUri->uri));
    NN_SDK_ASSERT(strnlen(pOutCallbackUri->uri, sizeof(pOutCallbackUri->uri)) < sizeof(pOutCallbackUri->uri));
    NN_RESULT_SUCCESS;
}
Result NasApplicationAuthorizationProcedure::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_NasClientInfo, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NasApplicationAuthorizationAdaptor adaptor(&m_pResource->authorization, m_NasClientInfo.redirectUri, m_pResource->state);
    NN_RESULT_THROW_UNLESS(adaptor.Parse(response, responseBufferSize), ResultNasDataBroken());
    NN_RESULT_DO(HandleInteractiveAuthorizationResult(m_NasOp, m_User, adaptor.Adapt()));

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}
bool NasApplicationAuthorizationProcedure::IsAuthorized() const NN_NOEXCEPT
{
    return m_ProcState == http::ProcedureState_Completed;
}
Result NasApplicationAuthorizationProcedure::GetAuthorizationCode(size_t* pOutActualSize, char* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOutActualSize = strnlen(m_pResource->authorization.code, sizeof(m_pResource->authorization.code));
    std::strncpy(buffer, m_pResource->authorization.code, std::min(size, *pOutActualSize));
    if (*pOutActualSize < size)
    {
        buffer[*pOutActualSize] = '\0';
    }
    NN_RESULT_SUCCESS;
}
Result NasApplicationAuthorizationProcedure::GetIdToken(size_t* pOutActualSize, char* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOutActualSize = strnlen(m_pResource->authorization.idToken, sizeof(m_pResource->authorization.idToken));
    std::strncpy(buffer, m_pResource->authorization.idToken, std::min(size, *pOutActualSize));
    if (*pOutActualSize < size)
    {
        buffer[*pOutActualSize] = '\0';
    }
    NN_RESULT_SUCCESS;
}
Result NasApplicationAuthorizationProcedure::GetState(State* pOut) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOut = m_pResource->state;
    NN_RESULT_SUCCESS;
}

/* ----------------------------------------------------------------------------------------------------------------------------
    NasGuestLoginProcedure
*/

NasGuestLoginProcedure::NasGuestLoginProcedure(
    const detail::ApplicationInfo& appInfo,
    ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT
    : m_NdasOp(ndasOp)
    , m_BaasOp(baasOp)
    , m_NasOp(nasOp)
    , m_ProcState(http::ProcedureState_Initialized)
    , m_AppInfo(appInfo)
    , m_Mapped(false)
    , m_pResource(nullptr)
{
}
NasGuestLoginProcedure::~NasGuestLoginProcedure() NN_NOEXCEPT
{
    if (m_Mapped)
    {
        std::memset(m_pResource, 0xDC, sizeof(*m_pResource));
        m_Memory.Unmap();
    }
}

void NasGuestLoginProcedure::AttachTransferMemory(os::NativeHandle transferMemoryHandle, size_t size, bool managed) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size >= RequiredMemorySize);
    m_Memory.Attach(size, transferMemoryHandle, managed);
}
Result NasGuestLoginProcedure::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Mapped);
    NN_SDK_ASSERT(m_pResource == nullptr);

    void* address;
    NN_RESULT_DO(m_Memory.Map(&address, os::MemoryPermission_None));
    m_Mapped = true;

    m_pResource = new(address) Resource;
    std::memset(&m_pResource->callbackUri, 0xDC, sizeof(m_pResource->callbackUri));
    std::memset(&m_pResource->state, 0x3A, sizeof(m_pResource->state));
    std::memset(&m_pResource->nasIdToken, 0xC1, sizeof(m_pResource->nasIdToken));
    std::memset(&m_pResource->userInfo, 0x54, sizeof(m_pResource->userInfo));
    m_pResource->imageSize = 0u;
    NN_RESULT_SUCCESS;
}

Result NasGuestLoginProcedure::ExecutePreprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultNotSupported());

    NN_RESULT_DO(m_NdasOp.AuthenticateApplication(m_AppInfo, resource, pCancellable));
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    http::GenerateStateString(m_pResource->state.data, sizeof(m_pResource->state.data));
    NN_RESULT_DO(m_NasOp.GetRequestForGuestLogin(pOutRequestUrl, m_pResource->state));

    std::strncpy(m_pResource->callbackUri.uri, NasResourceResolver::GetClientIdForAccountSystem().redirectUri, sizeof(m_pResource->callbackUri.uri));
    *pOutCallbackUri = m_pResource->callbackUri;
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    NasIdTokenAdaptor adaptor(m_pResource->nasIdToken, m_pResource->callbackUri.uri, m_pResource->state, m_NasOp.GetStorageRef());
    NN_RESULT_THROW_UNLESS(adaptor.Parse(response, responseBufferSize), ResultNasDataBroken());
    NN_RESULT_DO(HandleInteractiveAuthorizationResult(m_NasOp, InvalidUid, adaptor.Adapt()));
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_AppInfo, ResultNotSupported());

    // NAS ID トークンによるログイン
    NN_RESULT_DO(m_BaasOp.EnsureIdTokenCacheForApplicationForGuest(
        &m_pResource->userInfo,
        m_AppInfo, m_pResource->nasIdToken,
        resource, pCancellable));
    // すでに本体に存在する NA が指定されていないかの検査
    NN_RESULT_THROW_UNLESS(
        !nas::NasCredentialHolder::HasCredential(m_pResource->userInfo.linkedNaId, m_NasOp.GetStorageRef()),
        ResultDuplicativeExternalNetworkServiceAccount());

    // プロフィール画像のダウンロード
    if (m_pResource->userInfo.imageUrl[0] != '\0')
    {
        NN_RESULT_DO(baas::DownloadProfileImage(
            &m_pResource->imageSize, m_pResource->image, sizeof(m_pResource->image),
            m_pResource->userInfo.imageUrl, resource, pCancellable));
    }

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}

Result NasGuestLoginProcedure::GetNetworkServiceAccountId(NetworkServiceAccountId* pOut) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOut = m_pResource->userInfo.id;
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::GetLinkedNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOut = m_pResource->userInfo.linkedNaId;
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::GetNickname(char* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(size >= sizeof(m_pResource->userInfo.nickname), ResultInsufficientBuffer());
    auto l = strnlen(m_pResource->userInfo.nickname, sizeof(m_pResource->userInfo.nickname));
    std::strncpy(buffer, m_pResource->userInfo.nickname, l);
    if (l < size)
    {
        buffer[l] = '\0';
    }
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::GetProfileImage(size_t* pOutActualSize, void* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    if (buffer != nullptr)
    {
        std::memcpy(buffer, m_pResource->image, std::min(size, m_pResource->imageSize));
    }
    *pOutActualSize = m_pResource->imageSize;
    NN_RESULT_SUCCESS;
}
Result NasGuestLoginProcedure::LoadBaasIdTokenCache(size_t* pOutActualSize, char* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    return m_BaasOp.LoadIdTokenCache(pOutActualSize, buffer, size, m_pResource->userInfo.id, m_AppInfo);
}

/* ----------------------------------------------------------------------------------------------------------------------------
    NasGuestLoginProcedure
*/
NasFloatingRegistrationProcedure::NasFloatingRegistrationProcedure(const ndas::NdasOperator& ndasOp, baas::BaasOperator& baasOp, NasOperator& nasOp) NN_NOEXCEPT
    : m_NdasOp(ndasOp)
    , m_BaasOp(baasOp)
    , m_NasOp(nasOp)
    , m_ProcState(http::ProcedureState_Initialized)
    , m_CodeCacheId(detail::InvalidUuid)
    , m_Mapped(false)
    , m_pResource(nullptr)
{
    m_CredentialCache.refreshTokenCacheId = detail::InvalidUuid;
    m_CredentialCache.idTokenCacheId = detail::InvalidUuid;
}
NasFloatingRegistrationProcedure::~NasFloatingRegistrationProcedure() NN_NOEXCEPT
{
    if (m_CodeCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CodeCacheId, m_NasOp.GetStorageRef());
    }
    if (m_CredentialCache.refreshTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CredentialCache.refreshTokenCacheId, m_NasOp.GetStorageRef());
    }
    if (m_CredentialCache.idTokenCacheId)
    {
        detail::CacheUtil::DeleteCacheFile(m_CredentialCache.idTokenCacheId, m_NasOp.GetStorageRef());
    }
    if (m_Mapped)
    {
        std::memset(m_pResource, 0xDC, sizeof(*m_pResource));
        m_Memory.Unmap();
    }
}

void NasFloatingRegistrationProcedure::AttachTransferMemory(os::NativeHandle transferMemoryHandle, size_t size, bool managed) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size >= RequiredMemorySize);
    m_Memory.Attach(size, transferMemoryHandle, managed);
}
Result NasFloatingRegistrationProcedure::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_Mapped);
    NN_SDK_ASSERT(m_pResource == nullptr);

    void* address;
    NN_RESULT_DO(m_Memory.Map(&address, os::MemoryPermission_None));
    m_Mapped = true;

    m_pResource = new(address) Resource;
    std::memset(&m_pResource->userInfo, 0xD5, sizeof(m_pResource->userInfo));
    m_pResource->imageSize = 0u;
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::RegisterNetworkServiceAccount(const Uid& user, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_CredentialCache.idTokenCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_CredentialCache.refreshTokenCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_BaasOp.Import(
        user,
        m_NsaId, m_BaasCredential, m_NaId, m_CredentialCache, m_Property.codeVerifier,
        resource, pCancellable));

    detail::CacheUtil::DeleteCacheFile(m_CredentialCache.refreshTokenCacheId, m_NasOp.GetStorageRef());
    m_CredentialCache.refreshTokenCacheId = detail::InvalidUuid;

    detail::CacheUtil::DeleteCacheFile(m_CredentialCache.idTokenCacheId, m_NasOp.GetStorageRef());
    m_CredentialCache.idTokenCacheId = detail::InvalidUuid;

    m_BaasCredential.loginId = 0x00ull;
    std::memset(m_BaasCredential.loginPassword, 0x00, sizeof(m_BaasCredential.loginPassword));
    std::memset(m_Property.codeVerifier.data, 0x00, sizeof(m_Property.codeVerifier.data));

    NN_RESULT_DO(m_NasOp.UpdateUserResourceCache(user, resource, pCancellable));

    auto r = m_NasOp.UpdateNetworkServiceLicenseCache(user, resource, pCancellable);
    // OP2 の可用性次第で失敗する可能性があるので、通信エラーは許容する。
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkCommunicationError::Includes(r), r);
    NN_RESULT_SUCCESS;
}
const baas::UserProfile& NasFloatingRegistrationProcedure::GetUserProfileRef() const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_ProcState == http::ProcedureState_Completed);
    return m_pResource->userInfo;
}
const void* NasFloatingRegistrationProcedure::GetProfileImagePtr(size_t* pOut) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_ProcState == http::ProcedureState_Completed);
    *pOut = m_pResource->imageSize;
    return (m_pResource->imageSize > 0 ? m_pResource->image : nullptr);
}

Result NasFloatingRegistrationProcedure::ExecutePreprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());
    // 今は特にすることがない。
    NN_UNUSED(resource);
    NN_UNUSED(pCancellable);
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::GetRequest(RequestUrl* pOutRequestUrl, CallbackUri* pOutCallbackUri) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    http::GenerateCodeVerifier(&m_Property.codeVerifier);
    http::GenerateStateString(m_Property.state.data, sizeof(m_Property.state.data));

    // URL 生成
    NN_RESULT_DO(m_NasOp.GetRequestForLinkage(
        pOutRequestUrl, m_Property.codeVerifier, m_Property.state, NintendoAccountAuthorizationPageTheme_SimpleAuthentication));
    std::strncpy(m_Property.callback.uri, NasResourceResolver::GetClientIdForAccountSystem().redirectUri, sizeof(m_Property.callback.uri));
    *pOutCallbackUri = m_Property.callback;
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::ApplyResponse(const char* response, size_t responseBufferSize) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized || m_ProcState == http::ProcedureState_Resumed, ResultInvalidProtocolAccess());

    NasServiceAuthorizationAdaptor adaptor(m_Property.callback.uri, m_Property.state, m_NasOp.GetStorageRef());
    NN_RESULT_THROW_UNLESS(adaptor.Parse(response, responseBufferSize), ResultNasDataBroken());
    NN_RESULT_DO(HandleInteractiveAuthorizationResult(m_NasOp, InvalidUid, adaptor.Adapt()));

    NN_RESULT_DO(adaptor.PullAuthorizationCode(&m_CodeCacheId));
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::ExecutePostprocess(const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_CodeCacheId, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Initialized, ResultInvalidProtocolAccess());

    NN_RESULT_DO(m_NasOp.GetCredentialFromAuthorizationCode(
        &m_NaId, &m_CredentialCache,
        m_Property.codeVerifier, m_CodeCacheId,
        resource, pCancellable));
    // すでに本体に存在する NA が指定されていないかの検査
    NN_RESULT_THROW_UNLESS(
        !nas::NasCredentialHolder::HasCredential(m_NaId, m_NasOp.GetStorageRef()),
        ResultDuplicativeExternalNetworkServiceAccount());

    // DA の作成
    NN_RESULT_DO(m_BaasOp.AcquireCredentialFromNintendoAccountIdToken(
        &m_NsaId, &m_BaasCredential, &m_pResource->userInfo, m_NaId, m_CredentialCache.idTokenCacheId,
        resource, pCancellable));

    // プロフィール画像のダウンロード
    if (m_pResource->userInfo.imageUrl[0] != '\0')
    {
        NN_RESULT_DO(baas::DownloadProfileImage(
            &m_pResource->imageSize, m_pResource->image, sizeof(m_pResource->image),
            m_pResource->userInfo.imageUrl, resource, pCancellable));
    }

    m_ProcState = http::ProcedureState_Completed;
    NN_RESULT_SUCCESS;
}

Result NasFloatingRegistrationProcedure::GetNetworkServiceAccountId(NetworkServiceAccountId* pOut) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOut = m_NsaId;
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::GetLinkedNintendoAccountId(NintendoAccountId* pOut) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    *pOut = m_NaId;
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::GetNickname(char* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    NN_RESULT_THROW_UNLESS(size >= sizeof(m_pResource->userInfo.base.nickname), ResultInsufficientBuffer());
    auto l = strnlen(m_pResource->userInfo.base.nickname, sizeof(m_pResource->userInfo.base.nickname));
    std::strncpy(buffer, m_pResource->userInfo.base.nickname, l);
    if (l < size)
    {
        buffer[l] = '\0';
    }
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::GetProfileImage(size_t* pOutActualSize, void* buffer, size_t size) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    if (buffer != nullptr)
    {
        std::memcpy(buffer, m_pResource->image, std::min(size, m_pResource->imageSize));
    }
    *pOutActualSize = m_pResource->imageSize;
    NN_RESULT_SUCCESS;
}
Result NasFloatingRegistrationProcedure::EnsureNetworkServiceAccountIdToken(const detail::ApplicationInfo& appInfo, const ExecutionResource& resource, detail::Cancellable* pCancellable) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    return m_BaasOp.EnsureIdTokenCacheForApplication(m_NsaId, m_BaasCredential, m_NaId, appInfo, resource, pCancellable);
}
Result NasFloatingRegistrationProcedure::LoadBaasIdTokenCache(size_t* pOutActualSize, char* buffer, size_t size, const detail::ApplicationInfo& appInfo) const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(appInfo, ResultInvalidApplication());
    NN_RESULT_THROW_UNLESS(m_ProcState == http::ProcedureState_Completed, ResultInvalidProtocolAccess());
    return m_BaasOp.LoadIdTokenCache(pOutActualSize, buffer, size, m_NsaId, appInfo);
}

} // ~namespace nn::account::nas
}
}
