﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_NasCredentialHolder.h"

#include "../detail/account_PathUtil.h"
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/detail/account_InternalConfig.h>

#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace nas {

namespace
{
struct Info
{
    http::CodeVerifier codeVerifier;
};
} // ~namespace nn::account::nas

bool NasCredentialHolder::HasCredential(const NintendoAccountId& naId, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    size_t sizeActual;
    char path[128];
    auto r = storage.GetFileSystem().GetSize(
        &sizeActual, detail::PathUtil::GetNasUserInfoFilePath(path, sizeof(path), naId, storage.GetRootPath()));
    if (fs::ResultPathNotFound::Includes(r))
    {
        return false;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    if (sizeActual != sizeof(Info))
    {
        Delete(naId, storage);
        return false;
    }
    return true;
}

Result NasCredentialHolder::LoadCodeVerifier(
    http::CodeVerifier* pOut,
    const NintendoAccountId& naId, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOut != nullptr);
    size_t sizeActual;
    char path[128];
    Info info;
    auto r = storage.GetFileSystem().Read(
        &sizeActual, &info, sizeof(info),
        detail::PathUtil::GetNasUserInfoFilePath(path, sizeof(path), naId, storage.GetRootPath()));
    if (fs::ResultPathNotFound::Includes(r))
    {
        NN_RESULT_THROW(ResultNintendoAccountLinkageRequired());
    }
    NN_RESULT_DO(r);
    if (sizeActual != sizeof(info))
    {
        Delete(naId, storage);
        NN_RESULT_THROW(ResultNintendoAccountLinkageRequired());
    }
    *pOut = info.codeVerifier;
    NN_RESULT_SUCCESS;
}

Result NasCredentialHolder::LoadIdToken(
    size_t* pOutSizeActual, char* buffer, size_t bufferSize,
    const NintendoAccountId& naId, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(bufferSize >= detail::NasIdTokenSizeMax, ResultInsufficientBuffer());
    char path[128];
    size_t sizeActual;
    auto r = storage.GetFileSystem().Read(
        &sizeActual, buffer, bufferSize,
        detail::PathUtil::GetNasUserIdTokenFilePath(path, sizeof(path), naId, storage.GetRootPath()));
    if (fs::ResultPathNotFound::Includes(r))
    {
        NN_RESULT_THROW(ResultNintendoAccountLinkageRequired());
    }
    NN_RESULT_DO(r);
    if (sizeActual > detail::NasIdTokenSizeMax)
    {
        Delete(naId, storage);
        NN_RESULT_THROW(ResultNintendoAccountLinkageRequired());
    }
    *pOutSizeActual = sizeActual;
    NN_RESULT_SUCCESS;
}

Result NasCredentialHolder::LoadRefreshToken(
    size_t* pOutSizeActual, char* buffer, size_t bufferSize,
    const NintendoAccountId& naId, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(bufferSize >= detail::NasRefreshTokenSizeMax, ResultInsufficientBuffer());
    char path[128];
    size_t sizeActual;
    auto r = storage.GetFileSystem().Read(
        &sizeActual, buffer, bufferSize,
        detail::PathUtil::GetNasUserRefreshTokenFilePath(path, sizeof(path), naId, storage.GetRootPath()));
    if (fs::ResultPathNotFound::Includes(r))
    {
        NN_RESULT_THROW(ResultNintendoAccountLinkageRequired());
    }
    NN_RESULT_DO(r);
    if (sizeActual > detail::NasRefreshTokenSizeMax)
    {
        Delete(naId, storage);
        NN_RESULT_THROW(ResultNintendoAccountLinkageRequired());
    }
    *pOutSizeActual = sizeActual;
    NN_RESULT_SUCCESS;
}

Result NasCredentialHolder::Store(
    const NintendoAccountId& naId, const NasCredentialCache& cache, const http::CodeVerifier& codeVerifier,
    const detail::AbstractLocalStorage& storage,
    void *workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    auto& fs = storage.GetFileSystem();
    char path[128];
    char from[128];

    Info info = {codeVerifier};
    detail::PathUtil::GetNasUserInfoFilePath(path, sizeof(path), naId, storage.GetRootPath());
    NN_RESULT_DO(fs.Write(path, &info, sizeof(info)));

    detail::PathUtil::GetNasUserIdTokenFilePath(path, sizeof(path), naId, storage.GetRootPath());
    detail::PathUtil::GetCachePath(from, sizeof(from), cache.idTokenCacheId, storage.GetRootPath());
    NN_RESULT_DO(fs.Copy(path, from, workBuffer, workBufferSize));

    detail::PathUtil::GetNasUserRefreshTokenFilePath(path, sizeof(path), naId, storage.GetRootPath());
    detail::PathUtil::GetCachePath(from, sizeof(from), cache.refreshTokenCacheId, storage.GetRootPath());
    NN_RESULT_DO(fs.Copy(path, from, workBuffer, workBufferSize));

    NN_RESULT_SUCCESS;
}

void NasCredentialHolder::Delete(const NintendoAccountId& naId, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    auto& fs = storage.GetFileSystem();
    char path[128];

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs.Delete(
        detail::PathUtil::GetNasUserRefreshTokenFilePath(path, sizeof(path), naId, storage.GetRootPath())));

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs.Delete(
        detail::PathUtil::GetNasUserIdTokenFilePath(path, sizeof(path), naId, storage.GetRootPath())));

    NN_ABORT_UNLESS_RESULT_SUCCESS(fs.Delete(
        detail::PathUtil::GetNasUserInfoFilePath(path, sizeof(path), naId, storage.GetRootPath())));
}

}}} // ~namespace nn::account::baas
