﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/json/account_RapidJsonStream.h>

#include <nn/account/detail/account_LocalStorage.h>

namespace nn { namespace account { namespace json {

BufferedFileInputStreamForRapidJson::BufferedFileInputStreamForRapidJson(const char* path, const detail::AbstractLocalStorage& storage, detail::Cancellable* pCancellable) NN_NOEXCEPT
    : CancelInjectable(pCancellable)
    , m_Storage(storage)
    , m_Path(path)
{
}
Result BufferedFileInputStreamForRapidJson::GetResult() const NN_NOEXCEPT
{
    return m_IoResult;
}
size_t BufferedFileInputStreamForRapidJson::FillBufferImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    if (!(!IsCancelled() && m_IoResult.IsSuccess()))
    {
        return 0u;
    }
    // Read From File
    size_t actualRead;
    m_IoResult = m_Storage.GetFileSystem().Read(&actualRead, buffer, bufferSize, m_Path, m_Offset);
    if (m_IoResult.IsFailure())
    {
        return 0u;
    }
    m_Offset += actualRead;
    return actualRead;
}

BufferedFileOutputStreamForRapidJson::BufferedFileOutputStreamForRapidJson(const char* path, Ch* buffer, size_t bufferSize, const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
    : m_Storage(storage)
    , m_Path(path)
    , m_Buffer(buffer)
    , m_BufferSize(bufferSize)
{
    NN_SDK_ASSERT(buffer != nullptr);
    NN_SDK_ASSERT(reinterpret_cast<uintptr_t>(buffer) % NN_ALIGNOF(Ch) == 0);
    NN_SDK_ASSERT(bufferSize > 0);
    NN_SDK_ASSERT(bufferSize % sizeof(Ch) == 0);
}
void BufferedFileOutputStreamForRapidJson::FlushImpl() NN_NOEXCEPT
{
    if (m_IoResult.IsSuccess())
    {
        auto lock = m_Storage.AcquireWriterLock();
        m_IoResult = m_Storage.GetFileSystem().Append(m_Path, m_FlushSize, m_Buffer, m_FilledSize);
        m_FlushSize += m_FilledSize;
    }
    m_FilledSize = 0u;
}
Result BufferedFileOutputStreamForRapidJson::Finalize(size_t *pOutSize) NN_NOEXCEPT
{
    Flush();
    *pOutSize = m_FlushSize;
    NN_RESULT_THROW(m_IoResult);
}
void BufferedFileOutputStreamForRapidJson::Put(Ch c) NN_NOEXCEPT
{
    if (m_FilledSize >= m_BufferSize)
    {
        NN_SDK_ASSERT(m_FilledSize == m_BufferSize);
        FlushImpl();
    }
    m_Buffer[m_FilledSize ++] = c;
}
void BufferedFileOutputStreamForRapidJson::Flush() NN_NOEXCEPT
{
    if (m_FilledSize > 0)
    {
        FlushImpl();
    }
}

}}} // ~namespace nn::account::json
