﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_UrlEncoder.h"

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace account { namespace http {

int UrlEncode(char *dst, size_t dstSize, const char* src, size_t srcSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(strnlen(src, srcSize) < srcSize); // 終端
    NN_ABORT_UNLESS(dstSize >= 3 * (srcSize - 1) + 1); // 上限
    size_t offset = 0;
    for (size_t i = 0u; i < srcSize; ++ i)
    {
        auto c = src[i];
        if (c == '\0')
        {
            *(dst + offset) = '\0';
            return static_cast<int>(offset);
        }

        auto l = PutCharacterWithUrlEncoding(dst + offset, dstSize - offset, c);
        NN_SDK_ASSERT(l < dstSize - offset);
        offset += l;
    }
    NN_ABORT("[nn::account] ABORT: Unreachable\n");
}

bool CompareEncodedStringPartially(const char* expect, size_t expectSize, const char* codedTest, size_t codedTestSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(strnlen(expect, expectSize) < expectSize); // 終端
    NN_SDK_ASSERT(strnlen(codedTest, codedTestSize) <= codedTestSize); // 終端は見ない

    size_t offset = 0;
    for (size_t i = 0u; i < expectSize; ++ i)
    {
        auto c = expect[i];
        if (c == '\0')
        {
            return true;
        }

        char codedExpect[3];
        auto l = PutCharacterWithUrlEncoding(codedExpect, sizeof(codedExpect), c);
        NN_SDK_ASSERT(1 <= l && l <= 3);
        if (!(l <= codedTestSize - offset))
        {
            // 比較できるほど残文字がない
            return false;
        }

        for (size_t j = 0u; j < l; ++ j)
        {
            if (codedTest[offset + j] != codedExpect[j])
            {
                // 不一致
                return false;
            }
        }
        offset += l;
    }
    NN_ABORT("[nn::account] ABORT: Unreachable\n");
}

bool IsReservedCharacterForUri(char c) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc3986#section-2.2
    static const char Reserved[] = {
        0x20, // スペース : 未定義
        0x21, // !
        0x22, // " : 未定義
        0x23, // #
        0x24, // $
        0x25, // % : 未定義
        0x26, // &
        0x27, // '
        0x28, // (
        0x29, // )
        0x2A, // *
        0x2B, // +
        0x2C, // ,
        // 0x2D, // -
        // 0x2E, // .
        0x2F, // /
        0x3A, // :
        0x3B, // ;
        0x3C, // < : 未定義
        0x3D, // =
        0x3E, // > : 未定義
        0x3F, // ?
        0x40, // @
        0x5B, // [
        0x5C, // \ : 未定義
        0x5D, // ]
        0x5E, // ^ : 未定義
        // 0x5F, // _
        0x60, // ` : 未定義
        0x7B, // { : 未定義
        0x7C, // | : 未定義
        0x7D, // } : 未定義
        // 0x7E, // ~
    };
    size_t i = 0u;
    while (i < sizeof(Reserved) && Reserved[i] != c)
    {
        ++ i;
    }
    return i != sizeof(Reserved);
}

bool IsUnreservedCharacterForUri(char c) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc3986#section-2.2
    return false
        || ('A' <= c && c <= 'Z')
        || ('a' <= c && c <= 'z')
        || ('0' <= c && c <= '9')
        || c == '-'
        || c == '_'
        || c == '.'
        || c == '~';
}

size_t PutCharacterWithUrlEncoding(char* str, size_t strSize, char c) NN_NOEXCEPT
{
    // https://tools.ietf.org/html/rfc3986#section-2.2
    NN_SDK_ASSERT(strSize >= 3);
    NN_UNUSED(strSize);
    auto p = str;
    if (c == ' ')
    {
        // +
        *(p ++) = '+';
    }
    else if (IsUnreservedCharacterForUri(c))
    {
        // c
        *(p ++) = c;
    }
    else
    {
        NN_SDK_ASSERT(IsReservedCharacterForUri(c));
        auto u = (c >> 4);
        NN_ABORT_UNLESS(u <= 7);
        auto l = (c & 0x0F);
        NN_SDK_ASSERT(l <= 15);

        // %xx
        *(p ++) = '%';
        *(p ++) = static_cast<char>('0' + u);
        *(p ++) = static_cast<char>((l >= 10 ? 'A' + l - 10 : '0' + l));
    }
    return static_cast<int>(p - str);
}

}}} // ~namespace nn::account::http
