﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_UuidUtil.h"

#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/detail/account_InternalConfig.h>

#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Random.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/nifm/nifm_ApiNetworkProfile.h>
#include <nn/settings/system/settings_SerialNumber.h>
#endif

#include <nn/nn_SdkLog.h>

namespace nn {
namespace account {
namespace detail {

namespace {
static const uint32_t SecondIn100NanoSeconds = static_cast<uint32_t>(TimeSpan::FromSeconds(1).GetNanoSeconds() / 100);

const uint64_t MaxTimeValue = 0x0FFFFFFFFFFFFFFFull;
const uint64_t MaxNodeValue = 0x0000FFFFFFFFFFFFull;
const uint16_t MaxClockSequenceValue = 0x3FFF;

struct Context
{
    uint64_t span;
    uint16_t clockSeq;
    uint64_t node;
};

struct ContextStore
{
    time::SteadyClockTimePoint origin;
    int64_t span;
    uint32_t fraction;
    uint16_t clockSeq;
    uint64_t node;
};

void GenerateNodeAndClockSeq(uint64_t* pOutNode, uint16_t* pOutClockSeq) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // エントロピーの収集
    // - ランダムなバイト列 (16 bytes)
    uint8_t randomBytes[16];
    os::GenerateRandomBytes(randomBytes, sizeof(randomBytes));
    // - シリアル番号 (16 文字くらい)
    settings::system::SerialNumber serialNumber;
    settings::system::GetSerialNumber(&serialNumber);
    // - ネットワークインターフェースの MAC アドレス (6 bytes)
    int ifCount;
    nifm::NetworkInterfaceInfo info[1];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::EnumerateNetworkInterfaces(
        info, &ifCount, sizeof(info) / sizeof(info[0]),
        nifm::EnumerateNetworkInterfacesFilter_Ieee80211));
    NN_ABORT_UNLESS(ifCount >= 1, "[nn::account] ERROR: No wireless network interface found.\n");

    // ハッシュの生成
    crypto::Sha1Generator sha1;
    sha1.Initialize();
    sha1.Update(randomBytes, sizeof(randomBytes));
    sha1.Update(&serialNumber.string, strnlen(serialNumber.string, sizeof(serialNumber.string)));
    for (auto& f : info)
    {
        auto& ba = f.macAddress.data;
        sha1.Update(&ba, sizeof(ba));
    }

    // ノード値の生成
    char hash[crypto::Sha1Generator::HashSize]; // 20 bytes
    sha1.GetHash(hash, sizeof(hash));

    uint64_t seed = 0x00ull;
    for (int i = 0; i < sizeof(hash); ++ i)
    {
        seed = (seed ^ (static_cast<uint64_t>(hash[i]) << (8 * (i % sizeof(uint64_t))))); // [0, 64) ビットまで 8 bit 単位で埋める
    }
    *pOutNode = (seed & MaxNodeValue) | 0x01; // multicast-bit
    *pOutClockSeq = (seed >> 48) & MaxClockSequenceValue;
    // 最上位 2bit は捨ててしまうが...。

#elif defined(NN_BUILD_CONFIG_OS_WIN)
    *pOutNode = 0x00ull | 0x01; // multicast-bit
    *pOutClockSeq = 0;

#else
#error "Unsupported os specified"
#endif
}

Result TryLoadContextStore(bool* pOutIsLoaded, ContextStore* pOutStore, const AbstractFileSystem& fs) NN_NOEXCEPT
{
    // 前回のコンテキストの取得
    size_t size;
    auto r = fs.GetSize(&size, IdGenerationContextPath);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultPathNotFound::Includes(r), r);
    if (!r.IsSuccess())
    {
        *pOutIsLoaded = false;
        NN_RESULT_SUCCESS;
    }
    NN_ABORT_UNLESS(size == sizeof(ContextStore));

    NN_RESULT_DO(fs.Read(&size, pOutStore, sizeof(*pOutStore), IdGenerationContextPath));
    NN_ABORT_UNLESS(size == sizeof(ContextStore));
    *pOutIsLoaded = true;
    NN_RESULT_SUCCESS;
}

Result StoreContextStore(const ContextStore& store, const AbstractFileSystem& fs) NN_NOEXCEPT
{
    return fs.Write(IdGenerationContextPath, &store, sizeof(store));
}

inline uint32_t GetCurrent100NanoSeconds() NN_NOEXCEPT
{
    return static_cast<uint32_t>((os::GetSystemTick().ToTimeSpan().GetNanoSeconds() / 100) % SecondIn100NanoSeconds);
}

Result AcquireContext(Context* pOutContext, const AbstractFileSystem& fs) NN_NOEXCEPT
{
    // 現在時刻の取得
    time::SteadyClockTimePoint timePoint;
    NN_RESULT_DO(time::StandardSteadyClock::GetCurrentTimePoint(&timePoint));

    // 前回のコンテキストの取得
    bool loaded = false;
    ContextStore store;
    NN_RESULT_DO(TryLoadContextStore(&loaded, &store, fs));

    // コンテキストの作成, 更新
    if (loaded)
    {
        bool needToIncrementClockSeq = false;

        // 初期化時点からの経過秒数
        int64_t spanSinceOrigin;
        auto r = time::GetSpanBetween(&spanSinceOrigin, store.origin, timePoint);
        needToIncrementClockSeq = (needToIncrementClockSeq
            || !r.IsSuccess());
        needToIncrementClockSeq = (needToIncrementClockSeq
            || !(spanSinceOrigin >= 0 && static_cast<uint64_t>(spanSinceOrigin) <= (MaxTimeValue / SecondIn100NanoSeconds)));

        // 前回値との比較
        const auto FractionMax = SecondIn100NanoSeconds - 1;
        needToIncrementClockSeq = (needToIncrementClockSeq
            || !(spanSinceOrigin > store.span || (spanSinceOrigin == store.span && store.fraction < FractionMax)));

        // コンテキストの更新
        if (needToIncrementClockSeq)
        {
            // クロックシーケンスの更新が必要
            store.origin = timePoint;
            store.span = 0;
            store.fraction = GetCurrent100NanoSeconds();
            store.clockSeq = (store.clockSeq + 1) % MaxClockSequenceValue;
        }
        else
        {
            // Micro Seconds はチックベースで生成するか、前回からインクリメントする
            auto fraction = (spanSinceOrigin > store.span
                ? GetCurrent100NanoSeconds()
                : ((store.fraction + 1) % SecondIn100NanoSeconds));

            store.span = spanSinceOrigin;
            store.fraction = fraction;
        }
    }
    else
    {
        // コンテキストの新規作成
        uint64_t node;
        uint16_t clockSeq;
        GenerateNodeAndClockSeq(&node, &clockSeq);
        ContextStore context = {timePoint, 0, GetCurrent100NanoSeconds(), clockSeq, node};
        store = context;
    }

    NN_SDK_ASSERT(store.fraction < SecondIn100NanoSeconds);
    NN_SDK_ASSERT(store.span * SecondIn100NanoSeconds + store.fraction >= 0);
    NN_SDK_ASSERT(store.span * SecondIn100NanoSeconds + store.fraction <= MaxTimeValue);
    // origin を 1582-10-15 00:00:00::00 とみなす。
    Context ctx = {
        static_cast<uint64_t>(store.span * SecondIn100NanoSeconds + store.fraction),
        store.clockSeq,
        store.node
    };

    // コンテキストの保存
    NN_RESULT_DO(StoreContextStore(store, fs));
    *pOutContext = ctx;
    NN_RESULT_SUCCESS;
}
} // ~namespace nn::account::detail::<anonymous>

Uuid GenerateUuid(const AbstractFileSystem& fs) NN_NOEXCEPT
{
    // Generates UUID v1 based on RFC 4122
    // https://www.ietf.org/rfc/rfc4122.txt

    const uint16_t Version = 0x1000;
    const uint16_t VersionMask = 0xF000;
    const uint16_t Reserved = 0x80;
    const uint16_t ReservedMask = 0xC0;

    // コンテキスト取得
    Context context;
    NN_ABORT_UNLESS_RESULT_SUCCESS(AcquireContext(&context, fs));
    NN_SDK_ASSERT(context.span <= MaxTimeValue);
    NN_SDK_ASSERT(context.clockSeq <= MaxClockSequenceValue);
    NN_SDK_ASSERT(context.node <= MaxNodeValue);

    const auto timeLo = static_cast<uint32_t>(context.span & 0xFFFFFFFFull);
    const auto timeMid = static_cast<uint16_t>((context.span >> 32) & 0xFFFFull);
    const auto timeHi = static_cast<uint16_t>(context.span >> 48);
    const auto clockSeqLow = static_cast<uint8_t>(context.clockSeq & 0xFF);
    const auto clockSeqHi = static_cast<uint8_t>(context.clockSeq >> 8);
    const auto nodeLo = static_cast<uint16_t>(context.node & 0xFFFFull);
    const auto nodeHi = static_cast<uint32_t>((context.node >> 16) & 0xFFFFFFFFull);

    util::BitPack32 data[4] = {{0}, {0}, {0}, {0}};
    data[0].Set<UuidTraits::TimeLow>(timeLo);
    data[1].Set<UuidTraits::TimeMid>(timeMid);
    data[1].Set<UuidTraits::TimeHiAndVersion>((timeHi & ~VersionMask) | (Version));
    data[2].Set<UuidTraits::ClkSeqHiAndRes>(static_cast<uint8_t>((clockSeqHi & ~ReservedMask) | Reserved));
    data[2].Set<UuidTraits::ClkSeqLow>(static_cast<uint8_t>(clockSeqLow));
    data[2].Set<UuidTraits::NodeLo>(nodeLo);
    data[3].Set<UuidTraits::NodeHi>(nodeHi);

    Uuid uuid;
    uuid._data[0] = data[0].storage;
    uuid._data[1] = data[1].storage;
    uuid._data[2] = data[2].storage;
    uuid._data[3] = data[3].storage;
    return uuid;
}

} // ~namespace nn::account::detail
}
}
