﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "account_ShimLibraryUtility.h"
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_TypesForSystemServices.h>

#include <nn/nn_Abort.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
#include <nn/applet/applet.h>
#include <nn/applet/applet_ApplicationSelf.h>
#include <nn/applet/applet_Storage.h>
#include <nn/la/la_Api.h>
#endif

namespace nn {
namespace account {
namespace detail {

/* ---------------------------------------------------------------------------------------------
    nn::account のユーティリティ
*/

// NOTE: TrySelectUserWithoutInteraction() の実装は account_Api.cpp にある

// アプリケーション起動時に指定されたユーザーアカウントの取得
Uid TryPopPreselectedUser() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    applet::StorageHandle storageHandle;
    if (!applet::TryPopFromApplicationParameterChannel(&storageHandle, applet::LaunchParameterKind_Account))
    {
        return InvalidUid;
    }
    NN_UTIL_SCOPE_EXIT
    {
        applet::ReleaseStorage(storageHandle);
    };
    NN_ABORT_UNLESS(applet::GetStorageSize(storageHandle) >= sizeof(PreselectionInfo));

    PreselectionInfo preselection;
    NN_ABORT_UNLESS_RESULT_SUCCESS(applet::ReadFromStorage(storageHandle, 0, &preselection, sizeof(preselection)));
    NN_ABORT_UNLESS(preselection._magic == PreselectionInfoMagicNumber);
    NN_ABORT_UNLESS(preselection._uidCount == 1);
    return preselection._uids[0];
#else
    return InvalidUid;
#endif
}

Result CheckInternetAvailability() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        nifm::IsAnyInternetRequestAccepted(nifm::GetClientId()),
        ResultInternetConnectionRequestNotAccepted());
    NN_RESULT_SUCCESS;
}

/* ---------------------------------------------------------------------------------------------
    アプレット関係の内部利用 API
*/

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
namespace {
// CommonArgumentのバージョン指定
const int MajorVersion = 2;
const int MinorVersion = 0;

Result StartPselApplet(UiReturnArg* pOut, size_t outSize, const UiSettings& settings, const void* inData, size_t inDataBytes) NN_NOEXCEPT
{
    nn::applet::LibraryAppletHandle handle;
    NN_RESULT_DO(nn::applet::CreateLibraryApplet(
        &handle,
        nn::applet::AppletId_LibraryAppletPlayerSelect,
        nn::applet::LibraryAppletMode_AllForeground));
    NN_UTIL_SCOPE_EXIT
    {
        nn::applet::CloseLibraryApplet(handle);
    };

    // LA共通パラメータをストレージにプッシュする
    nn::la::CommonArgumentsWriter commonArg(MajorVersion, MinorVersion);
    commonArg.PushToInChannel(handle);

    // PSEL引数をプッシュする
    nn::la::PushToInChannel(handle, &settings, sizeof(settings));

    // 有効な inData が与えられているなら LargeStroge に画像バッファをプッシュする
    if (inData)
    {
        void* ptr = const_cast<void*>(inData); // NOTE: 渡した先で変更されないことは別途保証される。
        nn::applet::StorageHandle largeStrageHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::CreateLargeStorage(
            &largeStrageHandle, ptr, inDataBytes, false/*writable*/));
        nn::applet::PushToInChannel(handle, largeStrageHandle);
    }

    // LA起動
    auto result = nn::la::StartLibraryApplet(handle, false/*isExtermity*/);
    // nn::la での失敗は UIキャンセル扱いにして返す
    NN_RESULT_THROW_UNLESS(result.IsSuccess(), nn::account::ResultCancelledByUser());

    // LA戻り値取得
    nn::la::PopFromOutChannel(handle, pOut, outSize);

    return pOut->result;
}
} // ~namespace nn::account::detail::<anonymous>
#endif

Result StartPselApplet(const UiSettings& settings) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    UiReturnArg returnArg;
    return StartPselApplet(&returnArg, sizeof(returnArg), settings, nullptr, 0);
#else
    NN_UNUSED(settings);
    NN_RESULT_THROW(ResultNotSupported());
#endif
}

Result StartPselApplet(const UiSettings& settings, const void* inData, size_t inDataBytes) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    UiReturnArg returnArg;
    return StartPselApplet(&returnArg, sizeof(returnArg), settings, inData, inDataBytes);
#else
    NN_UNUSED(settings);
    NN_UNUSED(inData);
    NN_UNUSED(inDataBytes);
    NN_RESULT_THROW(ResultNotSupported());
#endif
}

Result ShowUserSelectorCommon(Uid* pOut, const UiSettings& settings) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOut != nullptr);
    NN_SDK_REQUIRES(settings.mode == UiMode_SelectUser);

    NN_ABORT_UNLESS(IsInitialized());
    if (settings.select.isSkipEnabled)
    {
        // Invalid 指定なし、かつ、追加選択でない
        NN_ABORT_UNLESS(
            !settings.select.invalidUidList[0],
            "[nn::account] With UserSelectionSettings.isSkipEnabled == true, any UIDs must not be specified to invalidUidList.\n");
        NN_ABORT_UNLESS(
            !settings.select.additionalSelect,
            "[nn::account] With UserSelectionSettings.isSkipEnabled == true, UserSelectionSettings.additionalSelect must not be enabled\n");

        Uid uid;
        NN_RESULT_DO(TrySelectUserWithoutInteraction(&uid, settings.select.isNetworkServiceAccountRequired));
        if (uid)
        {
            *pOut = uid;
            NN_RESULT_SUCCESS;
        }
    }

#if defined(NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON)
    UiReturnArg returnArg;
    NN_RESULT_DO(StartPselApplet(&returnArg, sizeof(returnArg), settings, nullptr, 0));
    *pOut = returnArg.select.uid;
    NN_RESULT_SUCCESS;
#else
    NN_RESULT_THROW(ResultNotSupported());
#endif
}

} // ~namespace nn::account::detail
}
}
