﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/account/account_IAccountService.sfdl.h>
#include <nn/account/account_ServiceNames.h>

#include <type_traits>

#include <nn/nn_Abort.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_HipcSimpleClientSessionManager.h>
#include <nn/sf/sf_IServiceObject.h>

namespace nn { namespace account { namespace detail {

const Bit32 InitializationManagerMagicNumber = 0x13724394;
#define NN_ACCOUNT_INITIALIZATION_MANAGER_INITIALIZER { \
    ::nn::account::detail::InitializationManagerMagicNumber, \
    0u, \
    NN_OS_MUTEX_INITIALIZER(false)}

/**
    @brief Shim ライブラリの初期化状態を管理する
 */
struct InitializationManager
{
    Bit32 _magicNumber;
    uint32_t _counter;
    os::MutexType _mutex;

    NN_EXPLICIT_OPERATOR bool() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        std::lock_guard<InitializationManager> lock(*this);
        return _counter > 0;
    }
    template <typename Initializer>
    void Initialize(const Initializer& initializer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        std::lock_guard<decltype(*this)> lock(*this);
        NN_ABORT_UNLESS(
            _counter < std::numeric_limits<uint32_t>::max(),
            "[nn::account] -----------------------------------------------\n"
            "  ABORT: Too much call for client initialization\n");
        if (_counter == 0)
        {
            initializer();
        }
        ++ _counter;
    }
    template <typename Finalizer>
    void Finalize(const Finalizer& finalizer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        std::lock_guard<decltype(*this)> lock(*this);
        NN_ABORT_UNLESS(
            _counter > 0,
            "[nn::account] -----------------------------------------------\n"
            "  ABORT: Too much call for client finalization\n");
        if (_counter == 1)
        {
            finalizer();
        }
        -- _counter;
    }
    void lock() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        os::LockMutex(&_mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == InitializationManagerMagicNumber);
        os::UnlockMutex(&_mutex);
    }
};

/**
    @brief Shim ライブラリに必要なリソースを管理する
 */
class ObjectHolder
{
private:
    struct ObjectHolderAllocatorTag {};
    static const size_t SizeOfHeapToAcquireObject = 1024 * 2;
    typedef sf::ExpHeapStaticAllocator<SizeOfHeapToAcquireObject, ObjectHolderAllocatorTag> AllocatorType;

    IAccountServiceBase* m_pBase;
    IAccountServiceForApplication* m_pForApplication;
    IAccountServiceForSystemService* m_pForSystemService;
    IAccountServiceForAdministrator* m_pForAdministrator;

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    static const int ClientSessionConcurrency = 1;
    sf::HipcSimpleClientSessionManager m_HipcDomain;
#endif

    template <typename Interface>
    Result AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT;

public:
    ObjectHolder() NN_NOEXCEPT;
    ~ObjectHolder() NN_NOEXCEPT;

    // 一般アプリ開発者向けに、最低限の権限を伴う初期化
    Result AcquireForApplicatoion() NN_NOEXCEPT;
    // 一般システム開発者向けに、状態の監視の権限を付加した初期化
    Result AcquireForSystemService() NN_NOEXCEPT;
    // アカウント管理の開発者向けに、全権限を付加した初期化
    Result AcquireForAdministrator() NN_NOEXCEPT;

    // サーバプロセス向けの、オブジェクト実体を使用した初期化
    Result InitializeWith(
        sf::SharedPointer<IAccountServiceForAdministrator>&& servicePtr) NN_NOEXCEPT;
    // オブジェクトの解放
    void Release() NN_NOEXCEPT;

    // 共通機能へのアクセッサ
    IAccountServiceBase& GetBase() const NN_NOEXCEPT;
    // 一般アプリ権限で使用可能な機能へのアクセッサ
    IAccountServiceForApplication& GetForApplication() const NN_NOEXCEPT;
    // 一般システム権限で使用可能な機能へのアクセッサ
    IAccountServiceForSystemService& GetForSystemService() const NN_NOEXCEPT;
    // アカウント管理者権限で使用可能な機能へのアクセッサ
    IAccountServiceForAdministrator& GetForAdministrator() const NN_NOEXCEPT;
};

}}} // ~namespace nn::account::detail

/* --------------------------------------------------------------------------------------------
    実装
 */

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/account/account_ServiceResource.h>
#endif

namespace nn { namespace account { namespace detail {

#if defined(NN_BUILD_CONFIG_OS_WIN32)

DefaultServiceResource& GetServiceResourceRef() NN_NOEXCEPT;

template <typename Interface>
inline Result ObjectHolder::AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT
{
    NN_UNUSED(serviceName);
    *ppOutService = GetServiceResourceRef().GetServicePointer<Interface>().Detach();
    NN_ABORT_UNLESS(*ppOutService != nullptr, "[nn::account] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}
template <>
inline Result ObjectHolder::AcquireImpl(IAccountServiceForApplication** ppOutService, const char* serviceName) NN_NOEXCEPT
{
    NN_UNUSED(serviceName);
    *ppOutService = GetServiceResourceRef().CreateServiceObjectForApplication().Detach();
    NN_ABORT_UNLESS(*ppOutService != nullptr, "[nn::account] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}
#else
template <typename Interface>
inline Result ObjectHolder::AcquireImpl(Interface** ppOutService, const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<Interface> ptr;
    NN_RESULT_DO((m_HipcDomain.InitializeByName<Interface, AllocatorType::Policy>(&ptr, serviceName)));
    NN_RESULT_DO(m_HipcDomain.SetSessionCount(ClientSessionConcurrency));

    *ppOutService = ptr.Detach();
    NN_ABORT_UNLESS(*ppOutService != nullptr, "[nn::account] Initialization failure while connecting.\n");
    NN_RESULT_SUCCESS;
}
#endif

inline ObjectHolder::ObjectHolder() NN_NOEXCEPT
    : m_pBase(nullptr)
    , m_pForApplication(nullptr)
    , m_pForSystemService(nullptr)
    , m_pForAdministrator(nullptr)
{
    AllocatorType::Initialize(lmem::CreationOption_NoOption);
}
inline ObjectHolder::~ObjectHolder() NN_NOEXCEPT
{
    Release();
}
inline Result ObjectHolder::AcquireForApplicatoion() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_pBase);
    NN_RESULT_DO(AcquireImpl(&m_pForApplication, ServiceNameForApplication));
    m_pBase = m_pForApplication;
    NN_RESULT_SUCCESS;
}
inline Result ObjectHolder::AcquireForSystemService() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_pBase);
    NN_RESULT_DO(AcquireImpl(&m_pForSystemService, ServiceNameForSystemService));
    m_pBase = m_pForSystemService;
    NN_RESULT_SUCCESS;
}
inline Result ObjectHolder::AcquireForAdministrator() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_pBase);
    NN_RESULT_DO(AcquireImpl(&m_pForAdministrator, ServiceNameForAdministrator));
    m_pBase = m_pForSystemService = m_pForAdministrator;
    NN_RESULT_SUCCESS;
}

inline Result ObjectHolder::InitializeWith(
        sf::SharedPointer<IAccountServiceForAdministrator>&& servicePtr) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_pBase);
    sf::SharedPointer<IAccountServiceForAdministrator> p(std::move(servicePtr));
    m_pBase = m_pForSystemService = m_pForAdministrator = p.Detach();
    NN_RESULT_SUCCESS;
}
inline void ObjectHolder::Release() NN_NOEXCEPT
{
    if (m_pBase != nullptr)
    {
        sf::ReleaseSharedObject(m_pBase);
        m_pBase = m_pForApplication = nullptr;
        m_pForSystemService = m_pForAdministrator = nullptr;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        m_HipcDomain.Finalize();
#endif
    }
}

inline IAccountServiceBase& ObjectHolder::GetBase() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pBase);
    return *m_pBase;
}
inline IAccountServiceForApplication& ObjectHolder::GetForApplication() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pForApplication);
    return *m_pForApplication;
}
inline IAccountServiceForSystemService& ObjectHolder::GetForSystemService() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pForSystemService);
    return *m_pForSystemService;
}
inline IAccountServiceForAdministrator& ObjectHolder::GetForAdministrator() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pForAdministrator);
    return *m_pForAdministrator;
}
}}} // ~namespace nn::account::detail
