﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/detail/account_FileSystem.h>

#include "account_PathUtil.h"
#include <nn/account/detail/account_LocalStorage.h>
#include <nn/account/account_ResultPrivate.h>

#include <algorithm>
#include <limits>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace account { namespace detail {
/** -------------------------------------------------------------------------------------------
    DefaultFileSystem の実装
 */
os::SdkRecursiveMutex DefaultFileSystem::s_WriterLock;

Result DefaultFileSystem::Setup(const char* rootPath) NN_NOEXCEPT
{
    char path[64];
    NN_RESULT_DO(CreateDirectoryImpl(rootPath));
    NN_RESULT_DO(CreateDirectoryImpl(
        PathUtil::GetProfileDirectoryPath(path, sizeof(path), rootPath)));
    NN_RESULT_DO(CreateDirectoryImpl(
        PathUtil::GetBaasDirectoryPath(path, sizeof(path), rootPath)));
    NN_RESULT_DO(CreateDirectoryImpl(
        PathUtil::GetNasDirectoryPath(path, sizeof(path), rootPath)));
    NN_RESULT_DO(RenewDirectoryImpl(
        PathUtil::GetCacheDirectoryPath(path, sizeof(path), rootPath)));
    NN_RESULT_SUCCESS;
}
void DefaultFileSystem::ClearProfile(const char* rootPath) NN_NOEXCEPT
{
    char path[64];
    NN_ABORT_UNLESS_RESULT_SUCCESS(RenewDirectoryImpl(
        PathUtil::GetProfileDirectoryPath(path, sizeof(path), rootPath)));
}
void DefaultFileSystem::ClearCache(const char* rootPath) NN_NOEXCEPT
{
    char path[64];
    NN_ABORT_UNLESS_RESULT_SUCCESS(RenewDirectoryImpl(
        PathUtil::GetCacheDirectoryPath(path, sizeof(path), rootPath)));
}
void DefaultFileSystem::Clear(const char* rootPath) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(RenewDirectoryImpl(rootPath));
}
FsLockGuard DefaultFileSystem::AcquireWriterLock() NN_NOEXCEPT
{
    return FsLockGuard(s_WriterLock);
}

Result DefaultFileSystem::Commit(const char* volumeName) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    return fs::CommitSaveData(volumeName);
}
Result DefaultFileSystem::DeleteDirectoryImpl(const char* path) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    auto r = fs::DeleteDirectoryRecursively(path);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultPathNotFound::Includes(r), r);
    NN_RESULT_SUCCESS; // 削除されてさえいれば OK
}
Result DefaultFileSystem::CreateDirectoryImpl(const char* path) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    // TODO 複数階層を考慮する (当面の間は 1 階層のみなので不要)
    auto r = fs::CreateDirectory(path);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultPathAlreadyExists::Includes(r), r);
    NN_RESULT_SUCCESS; // すでに存在する場合は消さずに OK (Windows っぽく振る舞う)
}
Result DefaultFileSystem::RenewDirectoryImpl(const char* path) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    auto r = DeleteDirectoryImpl(path);
    if (r.IsSuccess())
    {
        return CreateDirectoryImpl(path);
    }
    return r;
}
Result DefaultFileSystem::DeleteFileImpl(const char* path) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    auto r = fs::DeleteFile(path);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || fs::ResultPathNotFound::Includes(r), r);
    NN_RESULT_SUCCESS; // 削除されてさえいれば OK
}
Result DefaultFileSystem::GetFileSizeImpl(size_t* pOutSize, const char* path) NN_NOEXCEPT
{
    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(file);
    };

    int64_t size;
    NN_RESULT_DO(fs::GetFileSize(&size, file));
    NN_ABORT_UNLESS(
        size >= 0 && size <= std::numeric_limits<uint32_t>::max(),
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Cannot handle large (> 0xFFFFFFFF bytes) file\n"); // nn::account はこれでよい
    *pOutSize = static_cast<size_t>(size);
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::SetFileSizeImpl(const char* path, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(
        size >= 0 && size <= std::numeric_limits<uint32_t>::max(),
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Cannot handle large (> 0xFFFFFFFF bytes) file\n"); // nn::account はこれでよい

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(file);
    };
    NN_RESULT_DO(fs::SetFileSize(file, size));
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::ReadFileImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) NN_NOEXCEPT
{
    NN_SDK_ASSERT(offset <= static_cast<size_t>(std::numeric_limits<int64_t>::max()));

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(file);
    };

    int64_t size;
    NN_RESULT_DO(fs::GetFileSize(&size, file));
    NN_ABORT_UNLESS(
        size >= 0 && size <= std::numeric_limits<uint32_t>::max(),
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Cannot handle large (> 0xFFFFFFFF bytes) file\n"); // nn::account はこれでよい
    size_t readSize;
    NN_RESULT_DO(fs::ReadFile(&readSize, file, static_cast<int64_t>(offset), buffer, bufferSize));
    *pOutActualSize = static_cast<size_t>(size) - offset;
    NN_SDK_ASSERT(readSize == *pOutActualSize || (*pOutActualSize > bufferSize && readSize == bufferSize));
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::WriteFileImpl(const char* path, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    fs::FileHandle file;
    NN_RESULT_TRY(fs::OpenFile(&file, path, fs::OpenMode_Write))
        NN_RESULT_CATCH(fs::ResultPathNotFound)
    {
            NN_RESULT_DO(CreateFileImpl(path, bufferSize));
            NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write));
        }
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(file);
    };
    NN_RESULT_DO(fs::SetFileSize(file, bufferSize));
    fs::WriteOption op = {};
    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, bufferSize, op));
    NN_RESULT_DO(fs::FlushFile(file));
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::CreateFileImpl(const char* path, size_t sizeToReserve) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    NN_ABORT_UNLESS(
        sizeToReserve <= 0xFFFFFFFFull,
        "[nn::account] -----------------------------------------------\n"
        "  ABORT: Cannot handle large (> 0xFFFFFFFF bytes) file\n"); // nn::account はこれでよい
    NN_RESULT_DO(fs::CreateFile(path, static_cast<int64_t>(sizeToReserve)));
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::AppendFileImpl(const char* path, size_t offset, const void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    NN_SDK_ASSERT(offset <= static_cast<size_t>(std::numeric_limits<int64_t>::max()));

    fs::FileHandle file;
    NN_RESULT_DO(fs::OpenFile(&file, path, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(file);
    };

    fs::WriteOption op = {};
    NN_RESULT_DO(fs::WriteFile(file, static_cast<int64_t>(offset), buffer, bufferSize, op));
    NN_RESULT_DO(fs::FlushFile(file));
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::MoveFileImpl(const char* to, const char* from) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    NN_RESULT_TRY(fs::RenameFile(from, to))
        NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        {
            NN_RESULT_DO(DeleteFileImpl(to));
            NN_RESULT_DO(fs::RenameFile(from, to));
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}
Result DefaultFileSystem::CopyFileImpl(const char* to, const char* from, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    bool success = false;

    // 入力ファイルを開く (存在しない場合はサイズ 0 をコピーする)
    fs::FileHandle src;
    NN_RESULT_DO(fs::OpenFile(&src, from, fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(src);
    };
    int64_t sizeRaw;
    NN_RESULT_DO(fs::GetFileSize(&sizeRaw, src));

    // 出力ファイルを予約
    NN_SDK_ASSERT(s_WriterLock.IsLockedByCurrentThread());
    NN_RESULT_TRY(fs::CreateFile(to, sizeRaw))
        NN_RESULT_CATCH(fs::ResultPathAlreadyExists)
        {
            NN_RESULT_DO(fs::DeleteFile(to));
            NN_RESULT_DO(fs::CreateFile(to, sizeRaw));
        }
    NN_RESULT_END_TRY
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            fs::DeleteFile(to);
        }
    };

    // 出力ファイルを開く
    fs::FileHandle dst;
    NN_RESULT_DO(fs::OpenFile(&dst, to, fs::OpenMode_Write | fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        fs::CloseFile(dst);
    };

    // コピーの実行
    NN_SDK_REQUIRES(bufferSize > 0);
    auto sizeLeft = static_cast<size_t>(sizeRaw);
    while (sizeLeft > 0)
    {
        auto toRead = std::min(sizeLeft, bufferSize);
        size_t actualRead;
        NN_RESULT_DO(fs::ReadFile(&actualRead, src, sizeRaw - sizeLeft, buffer, toRead));
        NN_SDK_ASSERT(actualRead == toRead);
        NN_RESULT_DO(fs::WriteFile(dst, sizeRaw - sizeLeft, buffer, toRead, fs::WriteOption()));
        sizeLeft -= toRead;
    }
    NN_RESULT_DO(fs::FlushFile(dst));
    success = true;
    NN_RESULT_SUCCESS;
}

}}}
