﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/baas/account_BaasUserInfoHolder.h>

#include "../detail/account_PathUtil.h"
#include "../nas/account_NasCredentialHolder.h"
#include <nn/account/detail/account_Settings.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>

#include <cctype>

#include <nn/result/result_HandlingUtility.h>

namespace nn {
namespace account {
namespace baas {

namespace
{
Result CheckBaasUserStateImpl(const BaasUserInfo& info) NN_NOEXCEPT
{
    switch (info.userState)
    {
    case BaasUserState_Ok:
        break;
    case BaasUserState_Inconsequent:
        NN_RESULT_THROW(ResultNetworkServiceAccountCredentialBroken());
    case BaasUserState_Unmanaged:
        NN_RESULT_THROW(ResultNetworkServiceAccountUnmanaged());
    case BaasUserState_Banned:
        NN_RESULT_THROW(ResultNetworkServiceAccountBanned());
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_RESULT_SUCCESS;
}
Result CheckNintendoAccountStateImpl(const BaasUserInfo& info) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if (detail::FirmwareSettings::IsNaRequiredForNetworkService())
#else
    if (NN_STATIC_CONDITION(false))
#endif
    {
        NN_RESULT_THROW_UNLESS(
            info.nintendoAccount.id,
            ResultNintendoAccountLinkageRequired());

        switch (info.nintendoAccount.linkState)
        {
        case NintendoAccountLinkageState_Ok:
            break;
        case NintendoAccountLinkageState_Inconsequent:
            NN_RESULT_THROW(ResultNintendoAccountLinkageBroken());
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        switch (info.nintendoAccount.userState)
        {
        case NintendoAccountUserState_Ok:
            break;
        case NintendoAccountUserState_OtherButInteractionRequired:
            NN_RESULT_THROW(ResultNintendoAccountStateOtherButInteractionRequired());
        case NintendoAccountUserState_Deleted:
            NN_RESULT_THROW(ResultNintendoAccountStateDeleted());
        case NintendoAccountUserState_Banned:
            NN_RESULT_THROW(ResultNintendoAccountStateBanned());
        case NintendoAccountUserState_Suspended:
            NN_RESULT_THROW(ResultNintendoAccountStateSuspended());
        case NintendoAccountUserState_Withdrawn:
            NN_RESULT_THROW(ResultNintendoAccountStateWithdrawn());
        case NintendoAccountUserState_TermsAgreementRequired:
            NN_RESULT_THROW(ResultNintendoAccountStateTermsAgreementRequired());
        case NintendoAccountUserState_ReauthorizationRequired:
            NN_RESULT_THROW(ResultNintendoAccountStateReauthorizationRequired());
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    NN_RESULT_SUCCESS;
}
Result CheckAvailabilityImpl(const BaasUserInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckBaasUserStateImpl(info));
    NN_RESULT_DO(CheckNintendoAccountStateImpl(info));
    NN_RESULT_SUCCESS;
}
Result IsLinkedWithNintendoAccountImpl(bool* pOut, const BaasUserInfo& info) NN_NOEXCEPT
{
    NN_RESULT_DO(CheckBaasUserStateImpl(info));
    *pOut = info.nintendoAccount.id;
    NN_RESULT_SUCCESS;
}

} // ~namespace nn::account::baas::<anonymous>

const BaasUserInfoHolder::Cache::Entry BaasUserInfoHolder::Cache::InvalidEntry = {InvalidUid, {}};
BaasUserInfoHolder::Cache::Cache() NN_NOEXCEPT
{
    for (auto& e : entries)
    {
        e.Invalidate();
    }
}
const BaasUserInfoHolder::Cache::Entry& BaasUserInfoHolder::Cache::Find(const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    for (const auto& e : entries)
    {
        if (e.user == user)
        {
            return e;
        }
    }
    return InvalidEntry;
}
void BaasUserInfoHolder::Cache::Update(const Uid& user, const BaasUserData& data) NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    Entry* pFirstInvalid = nullptr;
    for (auto& e : entries)
    {
        if (e.user == user)
        {
            e.data = data;
            return;
        }
        else if (!e && pFirstInvalid == nullptr)
        {
            pFirstInvalid = &e;
        }
    }
    NN_ABORT_UNLESS(pFirstInvalid != nullptr, "[nn::account] ERROR: Insufficient cache capacity for BaaS user info\n");
    pFirstInvalid->user = user;
    pFirstInvalid->data = data;
}
void BaasUserInfoHolder::Cache::Remove(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(user);
    for (auto& e : entries)
    {
        if (e.user == user)
        {
            e.Invalidate();
            return;
        }
    }
}

BaasUserInfoHolder::BaasUserInfoHolder() NN_NOEXCEPT
    : m_pNasUserCache(nullptr)
    , m_pStorage(nullptr)
{
}
Result BaasUserInfoHolder::Initialize(
    nas::NasUserResourceCache& nasUserCache,
    const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
    m_pNasUserCache = &nasUserCache;
    m_pStorage = &storage;
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::Initialize(const detail::AbstractLocalStorage& storage) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS(!detail::FirmwareSettings::IsNaRequiredForNetworkService());
#endif

    m_pNasUserCache = nullptr;
    m_pStorage = &storage;
    NN_RESULT_SUCCESS;
}

Result BaasUserInfoHolder::StoreImplUnsafe(const Uid& user, const BaasUserInfo& userInfo, const BaasCredential& credential) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    char path[128];
    BaasUserData data = {userInfo, credential};
    m_Cache.Update(user, data);

    NN_RESULT_DO(m_pStorage->GetFileSystem().Write(
        detail::PathUtil::GetBaasUserFilePath(path, sizeof(path), user, m_pStorage->GetRootPath()),
        &data, sizeof(data)));
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::LoadImplUnsafe(BaasUserInfo* pOutUserInfo, BaasCredential* pOutCredential, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    NN_SDK_REQUIRES(pOutUserInfo != nullptr);
    NN_SDK_REQUIRES(pOutCredential != nullptr);

    auto e = m_Cache.Find(user);
    if (e)
    {
        *pOutUserInfo = e.data.info;
        *pOutCredential = e.data.credential;
        NN_RESULT_SUCCESS;
    }

    size_t sizeActual;
    char path[128];
    BaasUserData data;

    NN_RESULT_TRY(m_pStorage->GetFileSystem().Read(
        &sizeActual, &data, sizeof(data),
        detail::PathUtil::GetBaasUserFilePath(path, sizeof(path), user, m_pStorage->GetRootPath())))
    NN_RESULT_CATCH(fs::ResultPathNotFound)
    {
        NN_RESULT_THROW(ResultNetworkServiceAccountRegistrationRequired());
    }
    NN_RESULT_END_TRY

    NN_RESULT_THROW_UNLESS(
        sizeActual == sizeof(data),
        ResultNetworkServiceAccountRegistrationRequired());

    m_Cache.Update(user, data);
    *pOutUserInfo = data.info;
    *pOutCredential = data.credential;
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::LoadImplUnsafe(BaasUserInfo* pOutUserInfo, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasCredential credential;
    NN_UNUSED(credential);
    return LoadImplUnsafe(pOutUserInfo, &credential, user);
}

Result BaasUserInfoHolder::Register(const Uid& user, const NetworkServiceAccountId& id, const BaasCredential& credential) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    bool existence;
    NN_RESULT_DO(IsRegisteredUnsafe(&existence, user));
    NN_RESULT_THROW_UNLESS(!existence, ResultNetworkServiceAccountAlreadyRegistered());

    for (auto c: credential.loginPassword)
    {
        NN_RESULT_THROW_UNLESS(std::isalnum(c), ResultBaasDataBroken());
    }
    BaasUserInfo info = NN_ACCOUNT_BAAS_USER_INFO_INITIALIZER(id);
    NN_RESULT_DO(StoreImplUnsafe(user, info, credential));

    if (CheckAvailabilityImpl(info).IsSuccess())
    {
        // Available になるのなら通知する
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::Unregister(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    NN_RESULT_DO(LoadImplUnsafe(&info, user));
    bool linkage;
    auto r = IsLinkedWithNintendoAccountImpl(&linkage, info);
    linkage = (r.IsSuccess() ? linkage : false);
    // Available → Unavailable なら通知する
    bool toNotify = CheckAvailabilityImpl(info).IsSuccess();

    if (linkage)
    {
        NN_SDK_ASSERT(m_pNasUserCache);
        m_pNasUserCache->Invalidate(info.nintendoAccount.id);
        nas::NasCredentialHolder::Delete(info.nintendoAccount.id, *m_pStorage);
    }

    m_Cache.Remove(user);
    char path[128];
    NN_RESULT_DO(m_pStorage->GetFileSystem().Delete(
        detail::PathUtil::GetBaasUserFilePath(path, sizeof(path), user, m_pStorage->GetRootPath())));

    if (toNotify)
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::IsRegisteredUnsafe(bool* pOut, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    auto r = LoadImplUnsafe(&info, user);
    NN_RESULT_THROW_UNLESS(r.IsSuccess() || ResultNetworkServiceAccountRegistrationRequired::Includes(r), r);
    *pOut = r.IsSuccess();
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::IsRegistered(bool* pOut, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    return IsRegisteredUnsafe(pOut, user);
}
Result BaasUserInfoHolder::IsLinkedWithNintendoAccount(bool* pOut, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    NN_RESULT_DO(LoadImplUnsafe(&info, user));
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    *pOut = linkage;
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::GetNetworkServiceAccountId(NetworkServiceAccountId* pOut, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    NN_RESULT_DO(LoadImplUnsafe(&info, user));
    NN_RESULT_DO(CheckBaasUserStateImpl(info));
    *pOut = info.userId;
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::GetCredential(BaasCredential* pOut, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));
    NN_RESULT_DO(CheckBaasUserStateImpl(info));
    *pOut = credential;
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::GetLinkedNintendoAccountId(NintendoAccountId* pOut, const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    NN_RESULT_DO(LoadImplUnsafe(&info, user));
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    NN_RESULT_THROW_UNLESS(linkage, ResultNintendoAccountLinkageRequired());
    *pOut = info.nintendoAccount.id;
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::CheckAvailability(const Uid& user) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsReadLockHeld() || m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    NN_RESULT_DO(LoadImplUnsafe(&info, user));
    return CheckAvailabilityImpl(info);
}

Result BaasUserInfoHolder::UpdateNetworkServiceAccountId(const Uid& user, const NetworkServiceAccountId& nsaId) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));
    NN_RESULT_DO(CheckBaasUserStateImpl(info));

    info.userId = nsaId;
    return StoreImplUnsafe(user, info, credential);
}
Result BaasUserInfoHolder::DegradeBaasUserState(const Uid& user, BaasUserState state) NN_NOEXCEPT
{
    NN_SDK_ASSERT(state != BaasUserState_Ok);
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    if (linkage)
    {
        NN_SDK_ASSERT(m_pNasUserCache);
        m_pNasUserCache->Invalidate(info.nintendoAccount.id);
        nas::NasCredentialHolder::Delete(info.nintendoAccount.id, *m_pStorage);
    }
    BaasUserInfo degraded = NN_ACCOUNT_BAAS_USER_INFO_INITIALIZER(info.userId);
    degraded.userState = state;
    BaasCredential invalidCredential = NN_ACCOUNT_BAAS_CREDENTIAL_INITIALIZER;
    NN_RESULT_DO(StoreImplUnsafe(user, degraded, invalidCredential));

    if (isAvailableBefore != CheckAvailabilityImpl(degraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}

Result BaasUserInfoHolder::InitializeNintendoAccountLinkageState(
    const Uid& user, const NintendoAccountId& naId,
    const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());
    NN_ABORT_UNLESS(m_pNasUserCache);

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));

    // 事前条件
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    NN_RESULT_THROW_UNLESS(!linkage, ResultNintendoAccountAlreadyLinked());
    NN_RESULT_THROW_UNLESS(!nas::NasCredentialHolder::HasCredential(naId, *m_pStorage), ResultDuplicativeNintendoAccount());

    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    NN_RESULT_DO(nas::NasCredentialHolder::Store(naId, nasCredentialCache, codeVerifier, *m_pStorage, workBuffer, workBufferSize));
    NN_SDK_ASSERT(m_pNasUserCache);
    m_pNasUserCache->Perpetuate(naId);

    BaasUserInfo upgraded = NN_ACCOUNT_BAAS_USER_INFO_INITIALIZER(info.userId);
    upgraded.nintendoAccount.id = naId;
    upgraded.nintendoAccount.linkState = NintendoAccountLinkageState_Inconsequent;
    upgraded.nintendoAccount.userState = NintendoAccountUserState_Ok;
    NN_RESULT_DO(StoreImplUnsafe(user, upgraded, credential));

    if (isAvailableBefore != CheckAvailabilityImpl(upgraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::CompleteNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));

    // 事前条件
    NN_SDK_ASSERT(info.nintendoAccount.id);
    NN_SDK_ASSERT(info.nintendoAccount.linkState == NintendoAccountLinkageState_Inconsequent);
    NN_SDK_ASSERT(nas::NasCredentialHolder::HasCredential(info.nintendoAccount.id, *m_pStorage));

    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    auto upgraded = info;
    upgraded.nintendoAccount.linkState = NintendoAccountLinkageState_Ok;
    upgraded.nintendoAccount.userState = NintendoAccountUserState_Ok;
    NN_RESULT_DO(StoreImplUnsafe(user, upgraded, credential));

    if (isAvailableBefore != CheckAvailabilityImpl(upgraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::DegradeNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));

    // 事前条件
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    NN_RESULT_THROW_UNLESS(linkage, ResultNintendoAccountLinkageRequired());

    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    auto degraded = info;
    degraded.nintendoAccount.linkState = NintendoAccountLinkageState_Inconsequent;
    degraded.nintendoAccount.userState = NintendoAccountUserState_Ok;
    NN_RESULT_DO(StoreImplUnsafe(user, degraded, credential));

    if (isAvailableBefore != CheckAvailabilityImpl(degraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::DismissNintendoAccountLinkageState(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    NN_RESULT_THROW_UNLESS(linkage, ResultNintendoAccountLinkageRequired());
    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    // NA の情報を削除
    nas::NasCredentialHolder::Delete(info.nintendoAccount.id, *m_pStorage);
    NN_SDK_ASSERT(m_pNasUserCache);
    m_pNasUserCache->Invalidate(info.nintendoAccount.id);

    BaasUserInfo degraded = NN_ACCOUNT_BAAS_USER_INFO_INITIALIZER(info.userId);
    NN_RESULT_DO(StoreImplUnsafe(user, degraded, credential));

    if (isAvailableBefore != CheckAvailabilityImpl(degraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}

Result BaasUserInfoHolder::DegradeNintendoAccountUserState(const Uid& user, NintendoAccountUserState state) NN_NOEXCEPT
{
    NN_SDK_ASSERT(state != NintendoAccountUserState_Ok);
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    NN_RESULT_THROW_UNLESS(linkage, ResultNintendoAccountLinkageRequired());
    NN_RESULT_THROW_UNLESS(info.nintendoAccount.linkState == NintendoAccountLinkageState_Ok, ResultNotSupported());
    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    auto degraded = info;
    degraded.nintendoAccount.userState = state;
    NN_RESULT_DO(StoreImplUnsafe(user, degraded, credential));

    if (isAvailableBefore != CheckAvailabilityImpl(degraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::RecoveryNintendoAccountUserState(
    const Uid& user, const NintendoAccountId& naId,
    const nas::NasCredentialCache& nasCredentialCache, const http::CodeVerifier& codeVerifier,
    void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NintendoAccountId linkedNaId;
    NN_RESULT_DO(GetLinkedNintendoAccountId(&linkedNaId, user));
    NN_RESULT_THROW_UNLESS(linkedNaId == naId, ResultNotSupported());

    NN_RESULT_DO(RecoveryNintendoAccountUserState(user));
    NN_RESULT_DO(nas::NasCredentialHolder::Store(naId, nasCredentialCache, codeVerifier, *m_pStorage, workBuffer, workBufferSize));
    NN_RESULT_SUCCESS;
}
Result BaasUserInfoHolder::RecoveryNintendoAccountUserState(const Uid& user) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_RwLock.IsWriteLockHeldByCurrentThread());

    BaasUserInfo info;
    BaasCredential credential;
    NN_RESULT_DO(LoadImplUnsafe(&info, &credential, user));

    // 事前条件
    bool linkage;
    NN_RESULT_DO(IsLinkedWithNintendoAccountImpl(&linkage, info));
    NN_RESULT_THROW_UNLESS(linkage, ResultNintendoAccountLinkageRequired());
    NN_RESULT_THROW_UNLESS(info.nintendoAccount.linkState == NintendoAccountLinkageState_Ok, ResultNotSupported());

    // Availability 変化時に通知する
    auto isAvailableBefore = CheckAvailabilityImpl(info).IsSuccess();

    NN_SDK_ASSERT(m_pNasUserCache);
    m_pNasUserCache->Perpetuate(info.nintendoAccount.id);

    auto upgraded = info;
    upgraded.nintendoAccount.linkState = NintendoAccountLinkageState_Ok;
    upgraded.nintendoAccount.userState = NintendoAccountUserState_Ok;
    NN_RESULT_DO(StoreImplUnsafe(user, upgraded, credential));

    if (isAvailableBefore != CheckAvailabilityImpl(upgraded).IsSuccess())
    {
        Notifiable::SignalEvents(BaasUserInfoHolderTag_AvailablityChange);
    }
    NN_RESULT_SUCCESS;
}

void BaasUserInfoHolder::lock() const NN_NOEXCEPT
{
    m_RwLock.lock();
}
void BaasUserInfoHolder::unlock() const NN_NOEXCEPT
{
    m_RwLock.unlock();
}
void BaasUserInfoHolder::lock_shared() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_RwLock.IsWriteLockHeldByCurrentThread());
    m_RwLock.lock_shared();
}
void BaasUserInfoHolder::unlock_shared() const NN_NOEXCEPT
{
    m_RwLock.unlock_shared();
}

} // ~namespace nn::account::baas
}
}
