﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account/account_NintendoAccountAuthorizationRequest.h>

#include "detail/account_ShimLibraryUtility.h"
#include <nn/account/detail/account_Log.h>
#include <nn/account/nas/account_Interface.sfdl.h>
#include <nn/account/nas/account_NasTypes.h>
#include <nn/account/account_AsyncContext.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/account/account_ResultPrivate.h>
#include <nn/account/account_TypesForSelectorImpl.h>

#include <utility>

#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_IServiceObject.h>

namespace nn {
namespace account {

NintendoAccountAuthorizationRequestContext::NintendoAccountAuthorizationRequestContext() NN_NOEXCEPT
    : m_User(InvalidUid)
    , m_Ptr(nullptr)
{
}
NintendoAccountAuthorizationRequestContext::NintendoAccountAuthorizationRequestContext(nas::IAuthorizationRequest* ptr, const Uid& user) NN_NOEXCEPT
    : m_User(user)
    , m_Ptr(ptr)
{
}
NintendoAccountAuthorizationRequestContext::NintendoAccountAuthorizationRequestContext(NintendoAccountAuthorizationRequestContext&& rhs) NN_NOEXCEPT
    : m_User(rhs.m_User)
    , m_Ptr(rhs.m_Ptr)
    , m_AsyncContext(std::move(rhs.m_AsyncContext))
{
    rhs.m_User = InvalidUid;
    rhs.m_Ptr = nullptr;
}
NintendoAccountAuthorizationRequestContext::~NintendoAccountAuthorizationRequestContext() NN_NOEXCEPT
{
    if (m_Ptr)
    {
        sf::ReleaseSharedObject(m_Ptr);
        m_Ptr = nullptr;
    }
}
NintendoAccountAuthorizationRequestContext& NintendoAccountAuthorizationRequestContext::operator=(NintendoAccountAuthorizationRequestContext&& rhs) NN_NOEXCEPT
{
    NintendoAccountAuthorizationRequestContext tmp(std::move(rhs));
    tmp.Swap(*this);
    return *this;
}
NintendoAccountAuthorizationRequestContext& NintendoAccountAuthorizationRequestContext::Swap(NintendoAccountAuthorizationRequestContext& rhs) NN_NOEXCEPT
{
    std::swap(m_User, rhs.m_User);
    std::swap(m_Ptr, rhs.m_Ptr);
    AsyncContext tmp(std::move(rhs.m_AsyncContext));
    rhs.m_AsyncContext = std::move(m_AsyncContext);
    m_AsyncContext = std::move(tmp);
    return *this;
}

Result NintendoAccountAuthorizationRequestContext::Invoke() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);

    sf::SharedPointer<detail::IAsyncContext> p;
    NN_RESULT_DO(m_Ptr->InvokeWithoutInteractionAsync(&p));
    m_AsyncContext = AsyncContext(p.Detach(), m_User);
    NN_RESULT_SUCCESS;
}
Result NintendoAccountAuthorizationRequestContext::GetSystemEvent(os::SystemEvent* pOutEvent) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);
    return m_AsyncContext.GetSystemEvent(pOutEvent);
}
Result NintendoAccountAuthorizationRequestContext::Cancel() NN_NOEXCEPT
{
    return m_AsyncContext.Cancel();
}
Result NintendoAccountAuthorizationRequestContext::HasDone(bool* pOut) NN_NOEXCEPT
{
    return m_AsyncContext.HasDone(pOut);
}

Result NintendoAccountAuthorizationRequestContext::GetRawResult() NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);

    bool isAuthorized;
    NN_RESULT_DO(m_Ptr->IsAuthorized(&isAuthorized));
    if (isAuthorized)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_THROW(m_AsyncContext.GetResult());
}
Result NintendoAccountAuthorizationRequestContext::GetResult() NN_NOEXCEPT
{
    auto r = GetRawResult();
    if (r.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }

    if (false
        || ResultNetworkServiceAccountUnavailable::Includes(r)
        || ResultNintendoAccountAuthorizationInteractionRequired::Includes(r))
    {
        NN_RESULT_THROW(ResultNintendoAccountAuthorizationInteractionRequired());
    }
    NN_RESULT_THROW(r);
}
Result NintendoAccountAuthorizationRequestContext::GetResultWithInteractionIfRequired() NN_NOEXCEPT
{
    auto r = GetResult();
    if (r.IsSuccess())
    {
        NN_RESULT_SUCCESS;
    }
    else if (!ResultNintendoAccountAuthorizationInteractionRequired::Includes(r))
    {
        NN_RESULT_THROW(r);
    }

    detail::Uuid sessionId;
    NN_RESULT_DO(m_Ptr->GetSessionId(&sessionId));

    // Psel を使用した認可取得
    NN_DETAIL_ACCOUNT_INFO("Launching LibraryAppletPlayerSelect to acquire authorization\n");
    UiSettings ui = {};
    ui.mode = UiMode_ApplicationAuthorization;
    ui.applicationAuthorization.uid = m_User;
    std::memcpy(&ui.applicationAuthorization.sessionId, &sessionId, sizeof(ui.applicationAuthorization.sessionId));
    return detail::StartPselApplet(ui);
}
Result NintendoAccountAuthorizationRequestContext::GetAuthorizationCode(size_t* pOutActualSize, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutActualSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= NintendoAccountAuthorizationCodeLengthMax);

    uint32_t sizeRaw;
    NN_RESULT_DO(m_Ptr->GetAuthorizationCode(&sizeRaw, sf::OutBuffer(buffer, bufferSize)));
    *pOutActualSize = sizeRaw;
    NN_RESULT_SUCCESS;
}
Result NintendoAccountAuthorizationRequestContext::GetIdToken(size_t* pOutActualSize, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutActualSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= NintendoAccountIdTokenLengthMax);

    uint32_t sizeRaw;
    NN_RESULT_DO(m_Ptr->GetIdToken(&sizeRaw, sf::OutBuffer(buffer, bufferSize)));
    *pOutActualSize = sizeRaw;
    NN_RESULT_SUCCESS;
}
Result NintendoAccountAuthorizationRequestContext::GetState(size_t* pOutActualSize, char* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);
    NN_SDK_REQUIRES_NOT_NULL(pOutActualSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(bufferSize >= sizeof(nas::State));

    nas::State state;
    NN_RESULT_DO(m_Ptr->GetState(&state));
    std::strncpy(buffer, state.data, std::min(bufferSize, sizeof(state.data)));
    *pOutActualSize = strnlen(buffer, bufferSize);
    NN_RESULT_SUCCESS;
}
void NintendoAccountAuthorizationRequestContext::DebugGetInfo(uint64_t* pOut, size_t count) NN_NOEXCEPT
{
    NN_ACCOUNT_DETAIL_ASSERT_OBJECT_IS_VALID(this, NintendoAccountAuthorizationRequestContext);
    NN_SDK_ASSERT(count * sizeof(uint64_t) == sizeof(detail::Uuid));
    NN_UNUSED(count);

    detail::Uuid sessionId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetSessionId(&sessionId));
    std::memcpy(pOut, &sessionId, sizeof(sessionId));
}

} // ~namespace nn::account
}
