﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_BusConfig.h"
#include "tmp451_Debug.h"
#include "tmp451_InternalFunction.h"
#include "tmp451_SettingsMeasurment.h"
#include "tmp451_Specification.h"

namespace nnd {
namespace tmp451 {
namespace detail {

namespace
{
const uint8_t    MeasurementRateOffset = 4; // 計測レートの値とレジスタ設定値のオフセット
}

nn::Result GetMeasurementRange(MeasurementRange* range, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // レンジを変更した場合、温度検知用の限度値レジスタの値はそのままなので、
    // 変更後のレンジ設定で再度設定しなおす必要がある
    // TODO:温度検知機能に設定した限度値も内部で再設定してあげたい。
    // TODO:初期化時の引数にしておけば、実行途中での変更に対応しなくて良くなりそう。

    // 現在のコンフィグ値を取得
    uint8_t config;
    NN_RESULT_DO(ReadRegister(&config, bus, RegRead::Configuration));

    switch ((config >> 2) & 0x01)
    {
    case 0:
        {
            *range = MeasurementRange::Standard;
        }
        break;
    case 1:
        {
            *range = MeasurementRange::Extended;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}


nn::Result SetMeasurementRange(MeasurementRange range, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 現在のコンフィグ値を取得
    uint8_t config;
    NN_RESULT_DO(ReadRegister(&config, bus, RegRead::Configuration));

    switch (range)
    {
    case MeasurementRange::Standard:
        {
            config = config & ~(1 << 2); // RANGE -> 0
        }
        break;
    case MeasurementRange::Extended:
        {
            config = config | (1 << 2); // RANGE -> 1
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
    NN_TMP451_LOG_DETAIL("WRITE config: %2x\n",config);

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::Configuration),
        config,
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;
}


nn::Result GetMeasurementRateBy2ToThePowerN(int* n, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t rate;
    NN_RESULT_DO(ReadRegister(&rate, bus, RegRead::ConversionRate));

    rate = rate & 0xF;
//    NN_TMP451_LOG_DETAIL("rate: %02x\n",rate);

    *n = static_cast<int>(rate) - MeasurementRateOffset;
//    NN_TMP451_LOG_DETAIL("n: %d\n",*n);

    NN_RESULT_SUCCESS;
}


nn::Result SetMeasurementRateBy2ToThePowerN(int n, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    NN_SDK_ASSERT((0x0 <= (n + MeasurementRateOffset)) || ((n + MeasurementRateOffset) <= 0x9));

    uint8_t rate = static_cast<uint8_t>(n + MeasurementRateOffset);
    rate = rate & 0xF;

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::ConversionRate),
        rate,
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;
}


nn::Result GetBusTimeoutFunctionEnabled(bool* enable, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t data;
    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::ConsecutiveAlert));

//    NN_TMP451_LOG_DETAIL("data: %02x\n",data);
    switch ((data >> 7) & 0x01)
    {
    case 1:
        {
            *enable = true;
        }
        break;
    case 0:
        {
            *enable = false;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}


nn::Result SetBusTimeoutFunctionEnabled(bool enable, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 現在のレジスタ値を収録
    uint8_t data;
    NN_RESULT_DO(ReadRegister(&data, bus, RegRead::ConsecutiveAlert));

    if (enable)
    {
        data = data | (1 << 7); // SMBTO -> 1
//        NN_TMP451_LOG_DETAIL("TRUE: data | (1 << 7): %02x\n",data);
    }
    else
    {
        data = data & ~(1 << 7); // SMBTO -> 0
//        NN_TMP451_LOG_DETAIL("FALSE: data = data & ~(1 << 7): %02x\n",data);
    }

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::ConsecutiveAlert),
        data,
    };
//    NN_TMP451_LOG_DETAIL("RegWrite: %02x %02x\n",cmdSet[0], cmdSet[1]);

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), detail::I2cTransStartStop));
    NN_RESULT_SUCCESS;
}


} // detail
} // tmp451
} // nnd
