﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_BusConfig.h"
#include "tmp451_Debug.h"
#include "tmp451_InternalFunction.h"
#include "tmp451_Specification.h"

#include "tmp451_Measurement.h"


namespace nnd {
namespace tmp451 {
namespace detail {

namespace{

static nnd::tmp451::MeasurementMode     g_RecentMode = nnd::tmp451::MeasurementMode::Standby;
static nnd::tmp451::Status              g_Status = {false, false, false, false, false, false, false, false};



// 内部関数
nn::Result StartContinuousMode(BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 現状が同じモードなら成功
    if(g_RecentMode == MeasurementMode::Continuous)
    {
        NN_RESULT_SUCCESS;
    }

    // 不正な状態なのでエラー
    NN_RESULT_THROW_UNLESS(g_RecentMode == MeasurementMode::Standby, ResultInvalidState());

    // 現在のコンフィグ値を取得
    uint8_t config;
    NN_RESULT_DO(ReadRegister(&config, bus, RegRead::Configuration));

    // コマンド実行
    config = config & ~(1 << 6); // SD FIeld -> 0
    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::Configuration),
        config,
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), I2cTransStartStop));
    g_RecentMode = MeasurementMode::Continuous;
    NN_TMP451_LOG_DETAIL("END\n");
    NN_RESULT_SUCCESS;
}


nn::Result StartOneshotMode(BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 不正な状態なのでエラー
    NN_RESULT_THROW_UNLESS(g_RecentMode == MeasurementMode::Standby, ResultInvalidState());

    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::OneShot),
        1, // One-shot を実行するためには何の値でも良い
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), I2cTransStartStop));
    g_RecentMode = MeasurementMode::OneShot;
    NN_TMP451_LOG_DETAIL("END\n");
    NN_RESULT_SUCCESS;
}


nn::Result StopMeasurement(BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    // 現状が同じモードなら成功を返す
    if(g_RecentMode == MeasurementMode::Standby)
    {
        NN_RESULT_SUCCESS;
    }

    // OneShot は自動で動作停止するので、本関数による処理は無意味
    NN_RESULT_THROW_UNLESS(g_RecentMode == MeasurementMode::Continuous, ResultInvalidState());

    uint8_t config;

    NN_RESULT_DO(ReadRegister(&config, bus, RegRead::Configuration));

    // コマンド実行
    config = config | (1 << 6); // SD FIeld -> 1
    uint8_t cmdSet[2] = {
        static_cast<uint8_t>(RegWrite::Configuration),
        config,
    };

    NN_RESULT_DO(nn::i2c::Send(bus.i2cSession, &cmdSet, sizeof(cmdSet), I2cTransStart));
    g_RecentMode = MeasurementMode::Standby;

    NN_RESULT_DO(ReadRegister(&config, bus, RegRead::Configuration));
    NN_TMP451_LOG_DETAIL("Read config: %02x\n",config);
    NN_TMP451_LOG_DETAIL("END\n");
    NN_RESULT_SUCCESS;
}

void UpdateOneShotStatus(BusSessions bus) NN_NOEXCEPT
{
    // 現在のモードが OneShot の場合で、かつ、
    // status.isBusy が false 、つまり計測していない状態なら
    // OneShot が完了しているので Standby に変更
    Status status;
    ReadStatus(&status, bus, false);
    if(g_RecentMode == nnd::tmp451::MeasurementMode::OneShot && status.isBusy == false)
    {
        g_RecentMode = nnd::tmp451::MeasurementMode::Standby;
    }
}

} // namespace


MeasurementMode GetMeasurementMode(BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    UpdateOneShotStatus(bus);

    NN_TMP451_LOG_DETAIL("END\n");
    return g_RecentMode;
}


nn::Result SetMeasurementMode(MeasurementMode mode, BusSessions bus) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    UpdateOneShotStatus(bus);

    PrintCurrentState(g_RecentMode);

    switch (mode)
    {
    case MeasurementMode::Continuous:
        {
            return StartContinuousMode(bus);
        }
    case MeasurementMode::OneShot:
        {
            return StartOneshotMode(bus);
        }
    case MeasurementMode::Standby:
        {
            return StopMeasurement(bus);
        }
    default: NN_UNEXPECTED_DEFAULT;
    }
}


nn::Result ReadTemperature(Temperature* pOutTemp, Location location, BusSessions bus) NN_NOEXCEPT
{
    RegRead readInt;
    RegRead readDec;
    TemperatureByteData data;

    // 位置別のコマンド
    switch (location)
    {
    case Location::Local:
        readInt = RegRead::LocalTempInt;
        readDec = RegRead::LocalTempLsb;
        break;
    case Location::Remote:
        // リモートがオープン回路なら、無効値になるがエラーとはしない。
        readInt = RegRead::RemoteTempInt;
        readDec = RegRead::RemoteTempDec;
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // 整数部と小数部のどちらか一方をリードした場合、
    // もう一方の値はリードされるまでアップデートされない。
    // リードする場合は、必ずセットで読む。

    MeasurementRange range;
    NN_RESULT_DO(GetMeasurementRange(&range));

    NN_RESULT_DO(ReadRegister(&data.integerByte, bus, readInt));
    NN_RESULT_DO(ReadRegister(&data.decimalByte, bus, readDec));
    *pOutTemp = ConvertByteData2Temperature(data, range);

    NN_RESULT_SUCCESS;
}


nn::Result ReadStatus(Status* pOutStatus, BusSessions bus, bool readFromExternal) NN_NOEXCEPT
{
    NN_TMP451_LOG_DETAIL("START\n");

    uint8_t status;

    NN_RESULT_DO(ReadRegister(&status, bus, RegRead::Status));

    Status latest; // レジスタの持つ最新の状態
    latest.isBusy                           = ((status >> 7) & 0x1)? true : false;
    latest.isRemoteOpenCircuit              = ((status >> 6) & 0x1)? true : false;
    latest.isPin1LocalHighLimitExceeded     = ((status >> 5) & 0x1)? true : false;
    latest.isPin1RemoteHighLimitExceeded    = ((status >> 4) & 0x1)? true : false;
    latest.isPin2LocalHighLimitExceeded     = ((status >> 3) & 0x1)? true : false;
    latest.isPin2RemoteHighLimitExceeded    = ((status >> 2) & 0x1)? true : false;
    latest.isPin2LocalLowLimitExceeded      = ((status >> 1) & 0x1)? true : false;
    latest.isPin2RemoteLowLimitExceeded     = ((status >> 0) & 0x1)? true : false;

    // 常に最新の値で更新するメンバ変数
    g_Status.isBusy                           = latest.isBusy;
    g_Status.isPin1LocalHighLimitExceeded     = latest.isPin1LocalHighLimitExceeded;
    g_Status.isPin1RemoteHighLimitExceeded    = latest.isPin1RemoteHighLimitExceeded;

    // ドライバの外部から呼び出しされるまで true を保持するメンバ変数
    g_Status.isRemoteOpenCircuit =
            (g_Status.isRemoteOpenCircuit           || latest.isRemoteOpenCircuit);
    g_Status.isPin2LocalHighLimitExceeded =
            (g_Status.isPin2LocalHighLimitExceeded  || latest.isPin2LocalHighLimitExceeded);
    g_Status.isPin2RemoteHighLimitExceeded =
            (g_Status.isPin2RemoteHighLimitExceeded || latest.isPin2RemoteHighLimitExceeded);
    g_Status.isPin2LocalLowLimitExceeded =
            (g_Status.isPin2LocalLowLimitExceeded   || latest.isPin2LocalLowLimitExceeded);
    g_Status.isPin2RemoteLowLimitExceeded =
            (g_Status.isPin2RemoteLowLimitExceeded  || latest.isPin2RemoteLowLimitExceeded);

    // 戻り値に格納
    *pOutStatus = g_Status;

    // ドライバの外部から呼び出しだった場合は内部ステータスをリセットしておく
    if(readFromExternal)
    {
        g_Status = {false, false, false, false, false, false, false, false};;
    }

    NN_TMP451_LOG_DETAIL("END\n");
    NN_RESULT_SUCCESS;
}


nn::Result ReadManufacturerId(uint8_t* pOutId, BusSessions bus) NN_NOEXCEPT
{
    NN_RESULT_DO(ReadRegister(pOutId, bus, RegRead::ManufactureId));
    NN_RESULT_SUCCESS;
}

nn::Result HasDevice(BusSessions bus) NN_NOEXCEPT
{
    nn::Bit8 id;

    // ManufactureId を読む
    // 内容はチップリビジョン等で変わりそうなので内容の精査はしない
    auto result = ReadRegister(&id, bus, RegRead::ManufactureId);
    NN_TMP451_LOG_DETAIL("Manufacture ID = 0x%02x\n", id);
    return result;
}


} // detail
} // tmp451
} // nnd
