﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ftm ドライバのメイン実装です。
 */

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnd/ftm/ftm.h>

#include "ftm_Common.h"
#include "detail/ftm_Driver.h"
#include "detail/ftm_BusConfig.h"
#include "detail/ftm_Debug.h"

namespace nnd { namespace ftm {

namespace {

::nn::i2c::I2cSession         g_I2cSession;
::nn::gpio::GpioPadSession    g_GpioSessionReset;
::nn::gpio::GpioPadSession    g_GpioSessionInterrupt;

// 割り込みイベントが紐付け済みか
bool g_IsInterruptBound = false;

} // namespace

// デバイスの状態（ftm_ApiFab.cpp からも参照される）
DeviceState g_DeviceState = DeviceState::Uninitialized;

void Initialize() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // すでに初期化済みの場合は何もしない
    if (g_DeviceState != DeviceState::Uninitialized)
    {
        NN_FTM_LOG("%s(%d)\t: Already initialized.\n", __FUNCTION__, __LINE__);
        return;
    }

    NN_FTM_LOG("%s(%d)\t: Set Power Enabled\n", __FUNCTION__, __LINE__);
    detail::SetPowerEnabled(true);

    NN_FTM_LOG("%s(%d)\t: Initialize Bus\n", __FUNCTION__, __LINE__);
    detail::InitializeBus(&g_I2cSession, &g_GpioSessionInterrupt, &g_GpioSessionReset);

    NN_FTM_LOG("%s(%d)\t: Initialize Controller\n", __FUNCTION__, __LINE__);
    detail::InitializeController(g_I2cSession);

    g_DeviceState = DeviceState::Initialized;
}

void Finalize() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // 非 Uninitialized 状態であり割り込みが解除されていることを要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");
    NN_SDK_REQUIRES(
        g_IsInterruptBound == false, "Please call nnd::ftm::UnbindInterrupt()\n");

    NN_FTM_LOG("%s(%d)\t: Finalize Controller\n", __FUNCTION__, __LINE__);
    detail::FinalizeController();

    NN_FTM_LOG("%s(%d)\t: Finalize Bus\n", __FUNCTION__, __LINE__);
    detail::FinalizeBus(&g_I2cSession, &g_GpioSessionInterrupt, &g_GpioSessionReset);

    NN_FTM_LOG("%s(%d)\t: Set Power Disabled\n", __FUNCTION__, __LINE__);
    detail::SetPowerEnabled(false);

    g_DeviceState = DeviceState::Uninitialized;
}

::nn::Result ResetDevice() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    // スリープ時には一旦起床する
    if (g_DeviceState == DeviceState::Sleep)
    {
        NN_RESULT_DO(SleepOutDevice());
    }

    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(pController->ResetDevice());

    g_DeviceState = DeviceState::Ready;
    NN_RESULT_SUCCESS;
}

::nn::Result SleepInDevice() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // Ready Sensing Sleep いずれかの状態であることを要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Initialized, "Please call nnd::ftm::ResetDevice()\n");

    // すでにスリープ状態の場合は何もしない
    if (g_DeviceState == DeviceState::Sleep)
    {
        NN_RESULT_SUCCESS;
    }

    // センシングを無効にする
    if (g_DeviceState == DeviceState::Sensing)
    {
        NN_RESULT_DO(DeactivateSensing());
    }

    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(pController->SleepInDevice());

    detail::SleepInBus(&g_I2cSession, &g_GpioSessionInterrupt, &g_GpioSessionReset);

    g_DeviceState = DeviceState::Sleep;
    NN_RESULT_SUCCESS;
}

::nn::Result SleepOutDevice() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // Ready Sensing Sleep いずれかの状態であることを要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Initialized, "Please call nnd::ftm::ResetDevice()\n");

    // すでに覚醒状態の場合は何もしない
    if (g_DeviceState != DeviceState::Sleep)
    {
        NN_RESULT_SUCCESS;
    }

    detail::SleepOutBus(&g_I2cSession, &g_GpioSessionInterrupt, &g_GpioSessionReset);

    detail::IController* pController = detail::GetController();

    // スリープ状態からの復帰時は ResetDevice で Ready を待つ
    if (g_DeviceState == DeviceState::Sleep)
    {
        NN_RESULT_DO(pController->ResetDevice());
    }

    NN_RESULT_DO(pController->SleepOutDevice());

    g_DeviceState = DeviceState::Ready;
    NN_RESULT_SUCCESS;
}

::nn::Result BindInterrupt(::nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    // すでに割り込みイベントと紐付済みの場合エラーを返す
    NN_RESULT_THROW_UNLESS(!g_IsInterruptBound, ResultAlreadyBound());

    // IC の割り込み設定を有効にする
    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(pController->BindInterrupt());

    // イベントを紐付け
    NN_RESULT_DO(::nn::gpio::BindInterrupt(pEvent, &g_GpioSessionInterrupt));

    // 割り込み有効化
    SetInterruptEnable(true);

    g_IsInterruptBound = true;
    NN_RESULT_SUCCESS;
}

::nn::Result UnbindInterrupt() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    // 割り込みイベントと紐付いていない場合は何もしない
    if (g_IsInterruptBound == false)
    {
        NN_RESULT_SUCCESS;
    }

    // 割り込み無効化
    SetInterruptEnable(false);

    // 紐付けを解除
    ::nn::gpio::UnbindInterrupt(&g_GpioSessionInterrupt);

    g_IsInterruptBound = false;
    NN_RESULT_SUCCESS;
}

void SetInterruptEnable(bool enable) NN_NOEXCEPT
{
    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    // 割り込みをクリアする
    ::nn::gpio::ClearInterruptStatus(&g_GpioSessionInterrupt);

    // 割り込み設定
    ::nn::gpio::SetInterruptEnable(&g_GpioSessionInterrupt, enable);
}

::nn::Result ActivateSensing() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // Ready Sensing いずれかの状態であることを要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Initialized, "Please call nnd::ftm::ResetDevice()\n");
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Sleep, "Please call nnd::ftm::SleepOutDevice()\n");

    if (g_DeviceState != DeviceState::Sensing)
    {
        detail::IController* pController = detail::GetController();
        NN_RESULT_DO(pController->ActivateSensing());
    }

    g_DeviceState = DeviceState::Sensing;
    NN_RESULT_SUCCESS;
}

::nn::Result DeactivateSensing() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: START\n", __FUNCTION__, __LINE__);

    // Ready Sensing いずれかの状態であることを要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Initialized, "Please call nnd::ftm::ResetDevice()\n");
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Sleep, "Please call nnd::ftm::SleepOutDevice()\n");

    if (g_DeviceState != DeviceState::Ready)
    {
        detail::IController* pController = detail::GetController();
        NN_RESULT_DO(pController->DeactivateSensing());
    }

    g_DeviceState = DeviceState::Ready;
    NN_RESULT_SUCCESS;
}

::nn::Result ReadLeftEventCount(int* pOutLeftCount) NN_NOEXCEPT
{
    // Sensing 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Sensing, "Please call nnd::ftm::ActivateSensing()\n");

    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(pController->ReadLeftEventCount(reinterpret_cast<uint32_t*>(pOutLeftCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadEventReports(char* pOutReadData, int* pOutReadCount, bool* pOutIsOverflow, int readCount) NN_NOEXCEPT
{
    // Sensing 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Sensing, "Please call nnd::ftm::ActivateSensing()\n");

    // 読み込むイベント数が 0 以下の場合エラーを返す
    NN_RESULT_THROW_UNLESS(0 < readCount, ResultInvalidArgument());

    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(
        pController->ReadEventReports(
            pOutReadData, reinterpret_cast<uint32_t*>(pOutReadCount),
            pOutIsOverflow, static_cast<uint32_t>(readCount)));

    NN_RESULT_SUCCESS;
}

::nn::Result ReadEventReports(char* pOutReadData, int* pOutLeftCount, int* pOutReadCount, bool* pOutIsOverflow, int readCount) NN_NOEXCEPT
{
    // Sensing 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Sensing, "Please call nnd::ftm::ActivateSensing()\n");

    // 読み込むイベント数が 0 以下の場合エラーを返す
    NN_RESULT_THROW_UNLESS(0 < readCount, ResultInvalidArgument());

    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(
        pController->ReadEventReports(
            pOutReadData, reinterpret_cast<uint32_t*>(pOutReadCount), reinterpret_cast<uint32_t*>(pOutLeftCount),
            pOutIsOverflow, static_cast<uint32_t>(readCount)));

    NN_RESULT_SUCCESS;
}

void ParseEventReports(EventReport* pOutEventReport, const char* pRawData, int parseCount) NN_NOEXCEPT
{
    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    detail::IController* pController = detail::GetController();
    pController->ParseEventReports(pOutEventReport, pRawData, static_cast<uint32_t>(parseCount));
}

int GetMaxTouchNumber() NN_NOEXCEPT
{
    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    detail::IController* pController = detail::GetController();
    return pController->GetMaxTouchNumber();
}

int GetMaxEventReportCount() NN_NOEXCEPT
{
    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    detail::IController* pController = detail::GetController();
    return pController->GetMaxEventReportCount();
}

size_t GetEventReportByteSize() NN_NOEXCEPT
{
    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    detail::IController* pController = detail::GetController();
    return pController->GetEventReportByteSize();
}

nn::Result UpdateFirmware(FirmwareInputFunctionPointer const pFunction, void* const pParameter, size_t fileSize) NN_NOEXCEPT
{
    // 非 Uninitialized 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState != DeviceState::Uninitialized, "Please call nnd::ftm::Initialize()\n");

    detail::IController* pController = detail::GetController();
    NN_RESULT_DO(pController->UpdateFirmware(pFunction, pParameter, fileSize));

    g_DeviceState = DeviceState::Ready;
    NN_RESULT_SUCCESS;
}

nn::Result EraseFirmware() NN_NOEXCEPT
{
    // Ready 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Ready, "Please call nnd::ftm::ResetDevice()\n");

    detail::IController* pController = detail::GetController();
    return pController->EraseFirmware();
}

::nn::Result ReadFirmwareVersion(FirmwareVersion* pOutFirmwareVersion) NN_NOEXCEPT
{
    // Ready 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Ready, "Please call nnd::ftm::ResetDevice()\n");

    detail::IController* pController = detail::GetController();
    return pController->ReadFirmwareVersion(pOutFirmwareVersion);
}

::nn::Result RequestGpioState(GpioInformEventReport* pOutGpioInformEventReport) NN_NOEXCEPT
{
    // Ready 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Ready, "Please call nnd::ftm::ResetDevice()\n");

    detail::IController* pController = detail::GetController();
    return pController->RequestGpioState(pOutGpioInformEventReport);
}

::nn::Result RunAutoTune() NN_NOEXCEPT
{
    // Ready 状態を要求
    NN_SDK_REQUIRES(
        g_DeviceState == DeviceState::Ready, "Please call nnd::ftm::ResetDevice()\n");

    detail::IController* pController = detail::GetController();
    return pController->RunAutoTune();
}

void DeassertReset() NN_NOEXCEPT
{
    NN_FTM_LOG("%s(%d)\t: Deassert Reset\n", __FUNCTION__, __LINE__);
    detail::DeassertReset(&g_GpioSessionReset);
}

}} // namespace nnd::ftm
