﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>

#include <nnd/bh1730fvc/bh1730fvc.h>

#include "bh1730fvc_Driver.h"
#include "bh1730fvc_Specification.h"

#include "bh1730fvc_Debug.h"

#include <nn/nn_Log.h>

namespace nnd {
namespace bh1730fvc {
namespace detail {

namespace{

// TODO: setting から得られる補正用係数に置き換える
const float DiffusionCoefficient = 3.6f;


int GainNameToVal(Gain gainName) NN_NOEXCEPT
{
    int ret = InvalidIndex;
    for (int var = 0; var < NumberOfGain; var++)
    {
        if (GainNameValueCombinationList[var].name == gainName)
        {
            ret = GainNameValueCombinationList[var].val;
        }
    }
    NN_SDK_REQUIRES(ret != InvalidIndex, "Invalid value: gainName (%d)",gainName);
    return ret;
}

float CycleToMsec(nn::Bit8 cycle) NN_NOEXCEPT
{
    return static_cast<float>(cycle) * OneCycleTime;
}

} // namespace


float CalculateLuxImpl(int visible, int infrared, Gain gain, nn::Bit8 cycle, bool* pOutOverflown) NN_NOEXCEPT
{
    NND_BH1730FVC_DETAIL_LOG("START\n");
    NN_SDK_ASSERT_NOT_NULL(pOutOverflown);

    if (pOutOverflown)
    {
        if (visible >= SensorDataValueMax || infrared >= SensorDataValueMax)
        {
            *pOutOverflown = true;
        }
        else
        {
            *pOutOverflown = false;
        }
    }

    float tmpLx = 0;
    const float Data0 = static_cast<float>(visible);
    const float Data1 = static_cast<float>(infrared);

    // ベンダーから提供されている NX 用の算出式
    // If      (DATA1 / DATA0 < 0.500) Lx_tmp = 5.002 * DATA0 – 7.502 * DATA1
    // else If (DATA1 / DATA0 < 0.754) Lx_tmp = 2.250 * DATA0 – 2.000 * DATA1
    // else If (DATA1 / DATA0 < 1.029) Lx_tmp = 1.999 * DATA0 – 1.667 * DATA1
    // else If (DATA1 / DATA0 < 1.373) Lx_tmp = 0.884 * DATA0 – 0.583 * DATA1
    // else If (DATA1 / DATA0 < 1.879) Lx_tmp = 0.309 * DATA0 – 0.165 * DATA1
    // else Lx_tmp = 0
    // Lx = Lx_tmp / GAIN * (100ms / ITIME_ms)
    if (Data0 == 0.0f)
    {
        tmpLx = 0.0f;
    }
    else
    {
        const float Tmp = Data1 / Data0;
        if (Tmp < 0.500f)
        {
            tmpLx = 5.002f * Data0 - 7.502f * Data1;
        }
        else if (Tmp < 0.754f)
        {
            tmpLx = 2.250f * Data0 - 2.000f * Data1;
        }
        else if (Tmp < 1.029f)
        {
            tmpLx = 1.999f * Data0 - 1.667f * Data1;
        }
        else if (Tmp < 1.373f)
        {
            tmpLx = 0.884f * Data0 - 0.583f * Data1;
        }
        else if (Tmp < 1.879f)
        {
            tmpLx = 0.309f * Data0 - 0.165f * Data1;
        }
        else
        {
            tmpLx = 0.0f;
        }
    }

    return (tmpLx / GainNameToVal(gain) * (100.0f / CycleToMsec(cycle))) * DiffusionCoefficient;
}

} // detail
} // bh1730fvc
} // nnd
