﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief   tmp451 ライブラリのデータ型や定数の定義です。
 */

#include <nn/nn_Common.h>

namespace nnd {
namespace tmp451 {


//! @name 温度計測に関する型
//! @{

const int   DecimalUnit = 625;  //!< 小数部の分解能。0.0625℃。
const int   DecimalUnitDigit = 10000;  //!< 小数部の分解能の桁を表す値。

/**
 * @brief   温度値をまとめた構造体
 */
struct Temperature
{
    int integer;    //!< 温度値の整数部。分解能は 1 ℃。
    int decimal;    //!< 温度値の小数部。分解能は 0.0625℃。小数点以下4桁の値なので、625 の場合は 0.0625℃ になる。
};

/**
 * @brief   温度計測位置の名前
 */
enum class Location: uint8_t
{
    Local,      //!< TMP451 自身の計測点
    Remote,     //!< TMP451 の外部計測点
};

/**
 * @brief   計測温度のレンジ
 */
enum class MeasurementRange: uint8_t
{
    Standard,   //!< 0℃ to 127℃
    Extended,   //!< –64℃ to 191℃
};

/**
 * @brief   温度計測のモード
 */
enum class MeasurementMode: uint8_t
{
    Continuous, //!< Continuous モード
    OneShot,    //!< OneShot モード
    Standby,    //!< Standby モード
};

/**
 * @brief   デジタルフィルタの設定値
 */
enum class DigitalFilter : uint8_t
{
    FilterOff,          //!< フィルタ無効
    Filter4Results,     //!< 4 点平均
    Filter8Results,     //!< 8 点平均
};

//! @}



//! @name 温度検知機能に関する型
//! @{
/**
 * @brief   温度検知機能のモード
 */
enum class DetectionMode: uint8_t
{
    Therm,  //!< THERM
    Alert,  //!< ALERT
    Off,    //!< 無効
};

/**
 * @brief   温度検知のシグナルが発生するピンの名前
 */
enum class SignalPin: uint8_t
{
    Pin1,   //!< THERM ピン
    Pin2,   //!< ALERT/THERM2 ピン
};

/**
 * @brief   ALERT 機能のコンセクティブ値
 */
enum class AlertConsecutive: uint8_t
{
    Consecutive1,       //!< 計測に対するコンセクティブが 1 回
    Consecutive2,       //!< 計測に対するコンセクティブが 2 回
    Consecutive3,       //!< 計測に対するコンセクティブが 3 回
    Consecutive4,       //!< 計測に対するコンセクティブが 4 回
};

//! @}



/**
 * @brief   TMP451 のステータスをまとめた構造体
 */
struct Status
{
    bool isBusy;                        //!< 温度値の計測処理を行っている場合かどうか
    bool isRemoteOpenCircuit;           //!< Location::Remote が開回路かどうか

    bool isPin1LocalHighLimitExceeded;  //!< SignalPin::Pin1 に割り当てた DetectionMode にて Location::Local の上限値を上回ったかどうか
    bool isPin1RemoteHighLimitExceeded; //!< SignalPin::Pin1 に割り当てた DetectionMode にて Location::Remote の上限値を上回ったかどうか

    bool isPin2LocalHighLimitExceeded;  //!< SignalPin::Pin2 に割り当てた DetectionMode にて Location::Local の上限値を上回ったかどうか
    bool isPin2RemoteHighLimitExceeded; //!< SignalPin::Pin2 に割り当てた DetectionMode にて Location::Remote の上限値を上回ったかどうか

    bool isPin2LocalLowLimitExceeded;   //!< SignalPin::Pin2 に割り当てた DetectionMode にて Location::Local の下限値を下回ったかどうか
    bool isPin2RemoteLowLimitExceeded;  //!< SignalPin::Pin2 に割り当てた DetectionMode にて Location::Remote の下限値を下回ったかどうか
};


} // tmp451
} // nnd
