﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief       tmp451 ライブラリのリモート計測のみの動作設定の API リストです。
 */

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nnd/tmp451/tmp451.h>

#include "tmp451_Types.h"

namespace nnd {
namespace tmp451 {

//! @name リモート計測のみの動作設定
//! @{

/**
 * @brief       リモート温度のデジタルフィルタの値を取得します。
 *
 * @param[out]  pOutFilter   デジタルフィルタの設定値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 */
nn::Result GetRemoteDigitalFilterControl(DigitalFilter* pOutFilter) NN_NOEXCEPT;

/**
 * @brief       リモート温度のデジタルフィルタ値を設定します。
 *
 * @param[in]   filter   デジタルフィルタの設定値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     計測結果に対して、設定したフィルタ値で平均した値が、
 *              最終的なリモートの計測値として使用されます。
 */
nn::Result SetRemoteDigitalFilterControl(DigitalFilter filter) NN_NOEXCEPT;

/**
 * @brief       リモート温度オフセット設定値を取得します。
 *
 * @param[out]  pOutOffset   計測温度のレンジ設定
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 */
nn::Result GetRemoteTemperatureOffset(Temperature* pOutOffset) NN_NOEXCEPT;

/**
 * @brief       リモート温度オフセット値を設定します。
 *
 * @param[in]   offset   計測温度のレンジ設定
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     設定したオフセット値が加算された値が、
 *              リモートの計測値として使用されます。
 */
nn::Result SetRemoteTemperatureOffset(Temperature offset) NN_NOEXCEPT;

/**
 * @brief       η-factor 補正の値を取得します。
 *
 * @param[out]  pOutN   η-factor 算出式の N の値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 *
 * @pre         初期化済み状態である必要があります。
 *
 * @details     設定可能なηは (1.008 * 2088)/(2088 + N) {N | -128 ≤ N ≤ 127} であり
 *              本関数の返り値は式中の N で返します。
 */
nn::Result GetRemoteEtaFactorCorrectionValueByN(int8_t* pOutN) NN_NOEXCEPT;

/**
 * @brief       η-factor 補正の値を設定します。
 *
 * @param[in]   n   η-factor 算出式の N の値
 *
 * @return      処理の結果を返します。
 * @retval      ResultSuccess
 *                  成功しました。
 * @retval      ResultInvalidState
 *                  不正なステートであるため、処理を実行出来ませんでした。
 *                  設定したいモードに遷移するための前提状態を満たしていない可能性があります。
 *
 * @pre         Standby 状態である必要があります。
 *
 * @post        本関数が完了した次の計測から、変更が反映されます。
 *
 * @details     設定可能なηは (1.008 * 2088)/(2088 + N) {N | -128 ≤ N ≤ 127} であり、
 *              本関数の引数には式中の N を指定します。
 */
nn::Result SetRemoteEtaFactorCorrectionValueByN(int8_t n) NN_NOEXCEPT;

//! @}



} // tmp451
} // nnd
