﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ftm ライブラリの API リストです。
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nnd/ftm/ftm_Types.h>
#include <nnd/ftm/ftm_Result.h>

namespace nnd {
namespace ftm {

//! @name 初期化と終了処理
//! @{

/**
 * @brief       FTM ドライバライブラリを初期化します。
 *
 * @post        FTM ライブラリが初期化済み状態になります。 @n
 *              FTM のライブラリで使用するリソースが確保され、
 *              ドライバ内部で必要なライブラリが使用可能な状態になります。 @n
 *
 *              FTM のセンシングを有効にし EventReport を受け取るためには、
 *              デバイスの初期化を行った後、センシングを有効化する必要があります。
 *              本関数の実行後に、 @ref nnd::ftm::ResetDevice() でデバイスを初期化し、
 *              @ref nnd::ftm::SleepOutDevice() でアクティブ状態にした後に、
 *              @ref nnd::ftm::ActivateSensing() でセンシングを有効化してください。 @n
 *
 *              FTM からの割り込みを検知するためには、
 *              @ref nnd::ftm::BindInterrupt() によりシグナルを受け取る
 *              イベントオブジェクトの紐付けを行う必要があります。
 *
 * @details     FTM のドライバライブラリを初期化します。 @n
 *              デバイスの接続状態を ASSERT で確認します。
 *              関連するライブラリの初期化処理とリソースの確保が行われます。 @n
 *              FTM デバイス本体の初期化は行われないので、
 *              別途デバイスに対する処理を事後に行う必要があります。
 */
void Initialize() NN_NOEXCEPT;


/**
 * @brief       FTM ドライバライブラリを終了処理をします。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *              また、デバイスがアクティブ状態である場合、スリープ状態に
 *              遷移させておく必要があります。 @n
 *              スリープ状態に遷移させるためには @ref nnd::ftm::ResetDevice()
 *              もしくは @ref nnd::ftm::SleepInDevice() を使用してください。 @n
 *
 *              また @ref nnd::ftm::BindInterrupt() によりイベントオブジェクトと紐付け済みの場合は、
 *              @ref nnd::ftm::UnbindInterrupt() で解除してください。
 *
 * @post        FTM ライブラリが初期化前の状態になります。 @n
 *              FTM のライブラリで使用するリソースが解放され、
 *              ドライバ内部で必要なライブラリが初期化前の状態になります。 @n
 *
 * @details     FTM ドライバライブラリを終了します。 @n
 *              関連するライブラリの終了処理とリソースの解放が行われます。 @n
 *              FTM デバイス本体の終了処理は行われないので、
 *              別途デバイスに対する処理を事前に完了しておく必要があります。
 */
void Finalize() NN_NOEXCEPT;

//! @}



//! @name FTM の動作設定
//! @{

/**
 * @brief       FTM デバイス本体をリセットします。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutReady Ready 待ちがタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 * @post        デバイス本体がリセットされ、その後スリープ状態になります。 @n
 *              センシングを有効化するためには、
 *              @ref nnd::ftm::SleepOutDevice() によるアクティブ状態への遷移と、
 *              @ref nnd::ftm::ActivateSensing() によるセンシングの有効化が必要です。
 *              FTM 内部の FIFO に蓄積された EventReport は全て破棄されます。@n
 *
 * @detail      FTM デバイス本体をリセットします。 @n
 *              ライブラリの初期化後に実行後にセンシングを行う準備が完了するため、
 *              @ref nnd::ftm::ActivateSensing() でセンシングを有効化することが可能です。 @n
 *              また、センシング有効な状態でエラーを検出した際のエラー解決の手段として使用します。
 */
nn::Result ResetDevice() NN_NOEXCEPT;


/**
 * @brief       FTM デバイス本体をスリープ状態に遷移させます。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりアクティブ状態に
 *              遷移している必要があります。
 *
 * @post        スリープ状態になります。 @n
 *              @ref nnd::ftm::Finalize() によるライブラリの終了処理を行うことが可能になります。 @n
 *
 * @details     FTM デバイス本体をアクティブ状態から、スリープ状態に遷移させます。 @n
 */
nn::Result SleepInDevice() NN_NOEXCEPT;


/**
 * @brief       FTM デバイス本体をアクティブ状態に遷移させます。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::ResetDevice() もしくは
 *              @ref nnd::ftm::SleepInDevice() により
 *              スリープ状態に遷移している必要があります。
 *
 * @post        アクティブ状態になります。 @n
 *              @ref nnd::ftm::ActivateSensing() によるセンシングの有効化が可能になります。 @n
 *              @ref nnd::ftm::SleepInDevice() によるスリープ状態への遷移が可能になります。 @n
 *
 * @details     FTM デバイス本体をスリープ状態から、アクティブ状態に遷移させます。 @n
 */
nn::Result SleepOutDevice() NN_NOEXCEPT;


/**
 * @brief       FTM からの割り込みを検知するためのイベントオブジェクトを紐付けます。
 *
 * @param[in]   pEvent          登録するイベントオブジェクト
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess                 成功しました。
 * @retval      nnd::ftm::ResultAlreadyBound      すでにイベントが紐付けされています。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 *              pEvent は本関数内で初期化されるため、初期化せずに本関数へ渡してください。
 *              (初期化されていない pEvent が渡されているかどうかは ASSERT でチェックされます。)
 *
 * @post        デバイスからの割り込みを検知できる状態になります。
 *              (pEvent は本関数内で初期化されます。)
 *              pEvent が紐付けられた状態となり、FTM から割り込みが発生すると
 *              pEvent がシグナルされます。
 *
 * @details     FTM からの割り込みを検知するためのイベントオブジェクトを紐付けます。
 *              紐付できるイベントオブジェクトは 1 つまでです。
 */
nn::Result BindInterrupt(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief       FTM からの割り込みとイベントの紐付けを解除します。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 *              @ref nnd::ftm::BindInterrupt() によりイベントオブジェクトが
 *              紐付けされている必要があります。 @n
 *              紐付いているイベントオブジェクトを待機しているスレッドが
 *              いない状態で実行してください。
 *
 * @post        デバイスからの割り込みを検知できない状態になります。
 *              紐付けられた SystemEvent が紐付解除状態となります。 @n
 *              また、紐付いていた SystemEvent は破棄され、未初期化状態となります。
 *
 * @details     FTM からの割り込みとイベントの紐付けを解除します。
 */
nn::Result UnbindInterrupt() NN_NOEXCEPT;

/**
 * @brief       FTM からの割り込み許可状態を設定します。
 *
 * @param[in]   enable      FTM からの割り込み許可状態を設定します。
 *                          false：禁止
 *                          true：許可
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 * @details     FTMからの割り込み許可状態を設定します。
 *              割り込み検出ステータスはクリアされます。
 */
void SetInterruptEnable(bool enable) NN_NOEXCEPT;

/**
 * @brief       FTM をセンシング有効状態に遷移させます。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や
 *                                      同一バスのトランザクションが込み合っている際に
 *                                      発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @post        センシングが有効化された状態になります。 @n
 *              センシングイベント発生時にデバイス内部のスタックに EventReport が格納されます。 @n
 *              @ref nnd::ftm::ReadEventReports() により EventReport の読み込みが可能です。
 *
 * @details     FTM をセンシング無効な状態から、センシング有効な状態に遷移させます。
 */
nn::Result ActivateSensing() NN_NOEXCEPT;


/**
 * @brief       FTM をセンシング無効状態に遷移させます。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や
 *                                      同一バスのトランザクションが込み合っている際に
 *                                      発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::ActivateSensing() によりセンシングが有効な
 *              状態である必要があります。
 *
 * @post        センシングが無効化された状態になります。 @n
 *              タッチ操作によるセンシングが無効になり、タッチに関するイベントが発生しなくなります。
 *
 * @details     FTM によるセンシングを停止します。 @n
 *              再度センシングを開始するためには @ref nnd::ftm::ActivateSensing()
 *              を実行してください。
 */
nn::Result DeactivateSensing() NN_NOEXCEPT;

//! @}

//! @name FTM のファームウェア操作
//! @{

/**
 * @brief       FTM のファームウェアを更新します。
 *
 * @param[in]   pFunction               ファームウェアバイナリ入力用の関数ポインタ
 * @param[in]   pParameter              入力用関数に与える引数データのポインタ
 * @param[in]   fileSize                ファームウェアバイナリのファイルサイズ
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や
 *                                      同一バスのトランザクションが込み合っている際に
 *                                      発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutEraseFlash IC 内のフラッシュ消去がタイムアウトしました。デバイス不良の可能性があります。
 * @retval      nnd::ftm::ResultTimeoutDma        データ転送がタイムアウトしました。デバイス不良の可能性があります。
 * @retval      nnd::ftm::ResultInvalidArgument   ファームウェアのサイズが異なります。正しいファームウェアか確認してください。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが初期化済みの状態である必要があります。
 *
 * @details     FTM のファームウェアを更新します。
 */
nn::Result UpdateFirmware(FirmwareInputFunctionPointer const pFunction, void* const pParameter, size_t fileSize) NN_NOEXCEPT;

/**
 * @brief       FTM のファームウェアを消去します。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や
 *                                      同一バスのトランザクションが込み合っている際に
 *                                      発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutEraseFlash IC 内のフラッシュ消去がタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @details     FTM のファームウェアを消去します。
 */
nn::Result EraseFirmware() NN_NOEXCEPT;

/**
 * @brief       FTM のファームウェアバージョンを返します。
 *
 * @param[out]  pOutFirmwareVersion    ファームウェアバージョン
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @details     コントローラに書き込まれているファームウェアのバージョンを返します。
 */
nn::Result ReadFirmwareVersion(FirmwareVersion* pOutFirmwareVersion) NN_NOEXCEPT;

//! @}

//! @name チューニング処理
//! @{

/**
 * @brief       オートチューンを行います。
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutReady Ready 待ちがタイムアウトしました。デバイス不良の可能性があります。
 * @retval      nnd::ftm::ResultTimeoutMutualAutoTune 相互オートチューンがタイムアウトしました。デバイス不良の可能性があります。
 * @retval      nnd::ftm::ResultTimeoutSelfAutoTune 自己オートチューンがタイムアウトしました。デバイス不良の可能性があります。
 * @retval      nnd::ftm::ResultTimeoutSaveToFlash 設定値の保存がタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @details     オートチューンを行います。Low Power Timer Calibration についても同時に行われます。
 */
nn::Result RunAutoTune() NN_NOEXCEPT;

//! @}

//! @name GPIO アクセス処理
//! @{
/**
 * @brief       GPIO のステートを読み出します。
 *
 * @param[out]  pOutGpioInformEventReport   GPIO のステート
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess       成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 * @retval      nnd::ftm::ResultTimeoutGpio Gpio ステートの取得がタイムアウトしました。デバイス不良の可能性があります。
 *
 * @pre         @ref nnd::ftm::SleepOutDevice() によりデバイスを
 *              アクティブ状態に遷移させておく必要があります。
 *
 * @details     FTM IC の GPIO ステートを読み出します。
 */
nn::Result RequestGpioState(GpioInformEventReport* pOutGpioInformEventReport) NN_NOEXCEPT;

//! @}

//! @name イベントレポートの取得と解析
//! @{

/**
 * @brief       FTM 内部のスタックに積まれている EventReport の数を取得します。
 *
 * @param[out]  pOutLeftCount   EventReport 数
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess   データの読み込みに成功しました。
 * @retval      nn::i2c::ResultBusBusy  バスが BUSY 状態です。通信間隔が短い場合や同一バスの
 *                                      トランザクションが込み合っている際に発生する可能性があります。
 *
 * @pre         @ref nnd::ftm::ActivateSensing() によりセンシング有効である必要があります。
 *
 * @details     EventReport の数を取得した後に @ref nnd::ftm::ReadEventReports() の readCount を設定することで、
 *              データ通信時間を最小限にすることができます。
 */
nn::Result ReadLeftEventCount(int* pOutLeftCount) NN_NOEXCEPT;

/**
 * @brief       FTM 内部のスタックから指定した数だけ EventReport を読み込みます。
 *
 * @param[out]  pOutReadData    読み込んだ解析前のバイトデータ配列
 * @param[out]  pOutReadCount   読み込んだ EventReport の数
 * @param[out]  isOverflow      デバイス FIFO がオーバーフローしたかどうか
 * @param[in]   readCount       読み込みを希望する EventReport の数
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess   データの読み込みに成功しました。
 *
 * @pre         @ref nnd::ftm::ActivateSensing() によりセンシング有効である必要があります。
 *
 * @details     FTM 内部のスタックにある EventReport を古い順に指定した数だけ取得します。 @n
 *              pOutReadData で確保するサイズは、@ref nnd::ftm::GetEventReportByteSize() で得られるサイズの整数倍である必要があります。
 *              一回の呼び出しで取得可能な数 (readCount の有効最大値) は @ref nnd::ftm::GetMaxEventReportCount() で取得できます。@n
 *
 *              本関数で取得したデータは未解析なので @ref nnd::ftm::ParseEventReports()
 *              を使用して解析する必要があります。 @n
 *
 *              isOverflow が true の場合、デバイス内部の FIFO がオーバーフローした事を意味します。
 *              オーバーフローした場合、 FIFO の古いデータから順に削除されます。 @n
 *
 *              戻り値で何かしらのエラーを得た場合は、@ref nnd::ftm::ResetDevice() で
 *              リセットする必要があります。
 */
nn::Result ReadEventReports(char* pOutReadData, int* pOutReadCount, bool* isOverflow, int readCount) NN_NOEXCEPT;


/**
 * @brief       FTM 内部のスタックから指定した数だけ EventReport を読み込みます。
 *
 * @param[out]  pOutReadData    読み込んだ解析前のバイトデータ配列
 * @param[out]  pOutLeftCount   読み込み完了後に FTM 内部のスタックに残っている EventReport の数
 * @param[out]  pOutReadCount   読み込んだ EventReport の数
 * @param[out]  isOverflow      デバイス FIFO がオーバーフローしたかどうか
 * @param[in]   readCount       読み込みを希望する EventReport の数
 *
 * @return      処理の結果を返します。
 * @retval      nn::ResultSuccess   データの読み込みに成功しました。
 *
 * @pre         @ref nnd::ftm::ActivateSensing() によりセンシング有効である必要があります。
 *
 * @details     FTM 内部のスタックにある EventReport を古い順に指定した数だけ取得します。 @n
 *              pOutReadData で確保するサイズは、@ref nnd::ftm::GetEventReportByteSize() で得られるサイズの整数倍である必要があります。
 *              一回の呼び出しで取得可能な数 (readCount の有効最大値) は @ref nnd::ftm::GetMaxEventReportCount() で取得できます。@n
 *
 *              本関数で取得したデータは未解析なので @ref nnd::ftm::ParseEventReports()
 *              を使用して解析する必要があります。 @n
 *
 *              余計な読み出し処理を行わないためには、まず readCount を 1 として本関数を呼び出して
 *              取得した pOutLeftCount を、次の呼び出し時に readCount に指定するようにして下さい。
 *              データの取りこぼしを防ぐためには、 pOutLeftCount を確認し、0 ではない場合は
 *              直ちに本関数を再び実行して残りの EventReport を読み出して下さい。
 *              isOverflow が true の場合、デバイス内部の FIFO がオーバーフローした事を意味します。
 *              オーバーフローした場合、 FIFO の古いデータから順に削除されます。 @n
 *
 *              戻り値で何かしらのエラーを得た場合は、@ref nnd::ftm::ResetDevice() で
 *              リセットする必要があります。
 */
nn::Result ReadEventReports(char* pOutReadData, int* pOutLeftCount, int* pOutReadCount, bool* isOverflow, int readCount) NN_NOEXCEPT;


/**
 * @brief       複数の EventReport を解析します。
 *
 * @param[out]  pOutEventReport 解析結果が格納された EventReport の配列
 * @param[in]   pRawData        解析対象となるバイトデータの配列
 * @param[in]   parseCount      解析対象となる EventReport の数
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 * @details     @ref nnd::ftm::ReadEventReports() で取得した
 *              pOutReadData の内容を parseCount の数だけ解析して返します。
 */
void ParseEventReports(EventReport* pOutEventReport, const char* pRawData, int parseCount) NN_NOEXCEPT;

/**
 * @brief       同時に検出可能な最大点数を返します。
 *
 * @return      同時に検出可能な最大点数を返します。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 * @details     使用するコントローラで同時に検出可能な最大点数を返します。 @n
 *              FW で制限されるため、これより多くの点数が同時に検出されることはありません。
 */
int GetMaxTouchNumber() NN_NOEXCEPT;

/**
 * @brief       一度に読み出すことのできる EventReport の最大数を返します。
 *
 * @return      一度に読み出すことのできる EventReport の最大数を返します。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 * @details     使用するコントローラの内部スタックに存在できる EventReport の最大数を返します。@n
 *              この値が一度の @ref nnd::ftm::ReadEventReports() で読み込むことのできる最大の数となります。
 */
int GetMaxEventReportCount() NN_NOEXCEPT;

/**
 * @brief       解析前 EventReport のデータサイズを返します。
 *
 * @return      解析前 EventReport のデータサイズを返します。
 *
 * @pre         @ref nnd::ftm::Initialize() によりライブラリが
 *              初期化済みの状態である必要があります。 @n
 *
 * @details     使用するコントローラの内部スタックに存在する 1 つあたりの EventReport のデータサイズを返します。 @n
 *              これを @ref nnd::ftm::GetMaxEventReportCount() で得られた値とかけあわせたものが
 *              @ref nnd::ftm::ReadEventReports() で必要なバッファサイズの最大値となります。
 */
size_t GetEventReportByteSize() NN_NOEXCEPT;

/**
 * @brief       FTM IC のリセットをディアサートします。
 *
 * @details     FTM IC のリセットをディアサートします。IC だけを起動状態にして個別に I2C 制御を行いたい際に用います
 */
void DeassertReset() NN_NOEXCEPT;


//! @}

} // ftm
} // nnd
