﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   UART ドライバライブラリの開発用途限定のポートアクセス機能の API 宣言（C リンケージ向けインタフェース）。
 */

#pragma once

#include <nnc/nn_Common.h>

#include "uart_PortTypes.h"

//--------------------------------------------------------------------------
//  C/C++ 共通の宣言・定義
//--------------------------------------------------------------------------

#ifdef __cplusplus
extern "C"
{
#endif // ifdef __cplusplus

//! @name ポート情報の取得
//! @{

/**
 * @brief   指定したポートが、ターゲット環境上で使用可能かを取得します。
 *
 * @param[in]   portIndex   確認対象のポート番号
 *
 * @return  ポートが使用可能かを返します。
 *
 * @details
 *  この関数は、@ref nn::uart::HasPortForDev() の C リンケージ向けインターフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::HasPortForDev()
 */
bool nnuartHasPortForDev(int portIndex);

/**
 * @brief   指定したポートが、指定したボーレートをサポートしているかを取得します。
 *
 * @param[in]   portIndex   ポート番号
 * @param[in]   baudRate    ボーレート
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::IsSupportedBaudRateForDev() の C リンケージ向けインターフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::IsSupportedBaudRateForDev()
 */
bool nnuartIsSupportedBaudRateForDev(int portIndex, nnuartBaudRate baudRate);

/**
 * @brief   指定したポートが、指定したフロー制御モードをサポートしているかを取得します。
 *
 * @param[in]   portIndex       ポート番号
 * @param[in]   flowControlMode フロー制御モード
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::IsSupportedFlowControlModeForDev() の C リンケージ向けインターフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::IsSupportedFlowControlModeForDev()
 */
bool nnuartIsSupportedFlowControlModeForDev(int portIndex, nnuartFlowControlMode flowControlMode);

/**
 * @brief   指定したポートが、指定したポートイベントをサポートしているかを取得します。
 *
 * @param[in]   portIndex       ポート番号
 * @param[in]   portEvent       ポートイベント
 *
 * @return  サポート有無を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::IsSupportedPortEventForDev() の C リンケージ向けインターフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::IsSupportedPortEventForDev()
 */
bool nnuartIsSupportedPortEventForDev(int portIndex, nnuartPortEventType portEvent);

//! @}

//! @name ポートアクセス機能
//! @{

/**
 * @brief   開発目的で使用するための特定のポートを直接指定して、ポートをオープンします。
 *
 * @param[out]  pOutSession オープンされたポートセッション
 * @param[in]   portIndex   ポート番号
 * @param[in]   pPortConfig ポート設定
 *
 * @return      処理の結果を返します。
 *
 * @details
 *  この関数は、@ref nn::uart::OpenPortForDev() の C リンケージ向けインタフェースです。
 *  詳細は C++ 向け版を参照してください。
 *
 * @see
 *  nn::uart::OpenPortForDev()
 */
bool nnuartOpenPortForDev(nnuartPortSession* pOutSession, int portIndex, const nnuartPortConfigType* pPortConfig);

//! @}

#ifdef __cplusplus
}
#endif // ifdef __cplusplus
