﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nnc/ssl/ssl_Types.h>
#include <nnc/ssl/ssl_Context.h>

// TODO: Need to define this properly
#define NN_SSL_SIZEOF_SSL_CONNECTION_TYPE 0x80 // NOLINT(readability/define)
NN_SSL_DEFINE_ALIGNED_STORAGE(nnsslSslConnectionType, nn::ssl::Connection, NN_SSL_SIZEOF_SSL_CONNECTION_TYPE, void*);

#define NN_SSL_SIZEOF_SSL_CONNECTION_SERVER_CERT_DETAIL_TYPE 0xC // NOLINT(readability/define)
NN_SSL_DEFINE_ALIGNED_STORAGE(nnsslConnectionServerCertDetailType, nn::ssl::Connection::ServerCertDetail, NN_SSL_SIZEOF_SSL_CONNECTION_SERVER_CERT_DETAIL_TYPE, void*);

typedef enum nnsslConnectionPollEvent
{
    nnsslConnectionPollEvent_None   = 0x00, //<! No event
    nnsslConnectionPollEvent_Read   = 0x01, //<! Read event
    nnsslConnectionPollEvent_Write  = 0x02, //<! Write event
    nnsslConnectionPollEvent_Except = 0x04  //<! Exception event
}nnsslConnectionPollEvent;

typedef enum nnsslConnectionVerifyOption
{
    nnsslConnectionVerifyOption_None          = 0x00, //<! Do not validate nothing
    nnsslConnectionVerifyOption_PeerCa        = 0x01, //<! Validate the peer certificate
    nnsslConnectionVerifyOption_HostName      = 0x02, //<! In addition to peer certificate validation, also check if the host name matches the Common Name field or a Subject Alternate Name field in the peer certificate
    nnsslConnectionVerifyOption_DateCheck     = 0x04, //<! In addition to peer certificate validation, calidate certificates CalidFrom and ValidTo fields
    nnsslConnectionVerifyOption_EvCertPartial = 0x08, //!< In addition to peer certificate validation, validate extended validation certificate without revocation check
    nnsslConnectionVerifyOption_All = (
        nnsslConnectionVerifyOption_PeerCa   |
        nnsslConnectionVerifyOption_HostName |
        nnsslConnectionVerifyOption_DateCheck)
}nnsslConnectionVerifyOption;

typedef enum nnsslConnectionIoMode
{
    nnsslConnectionIoMode_Blocking    = 1, //<! I/O APIs (DoHandshake/Read/Write) behave as blocking API (default)
    nnsslConnectionIoMode_NonBlocking = 2  //<! I/O APIs (DoHandshake/Read/Write) behave as non-blocking API
}nnsslConnectionIoMode;

typedef enum nnsslConnectionSessionCacheMode
{
    nnsslConnectionSessionCacheMode_None      = 0, //!< No session cache will be used for this connection
    nnsslConnectionSessionCacheMode_SessionId = 1, //!< Session ID will be used when reusing available SSL session (default)
}nnsslConnectionSessionCacheMode;

typedef enum nnsslConnectionOptionType
{
    nnsslConnectionOptionType_DoNotCloseSocket   = 0, //!< The imported socket by SetSocketDescriptor() will not be closed automatically when enabled (disabled by default)
    nnsslConnectionOptionType_GetServerCertChain = 1, //!< Return the entire server certificate chain after DoHandshake. The default setting just returns the server's certificate.
    nnsslConnectionOptionType_SkipDefaultVerify  = 2, //!< Allow the connection to skip default certificate validation (disabled by default)
}nnsslConnectionOptionType;

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

nnResult nnsslConnectionCreate(
    nnsslSslConnectionType* pSslConnection,
    nnsslSslContextType* pInClientContext);
nnResult nnsslConnectionDestroy(nnsslSslConnectionType* pSslConnection);

nnResult nnsslConnectionSetSocketDescriptor(
    nnsslSslConnectionType* pSslConnection,
    int socketDescriptor);
nnResult nnsslConnectionSetHostName(
    nnsslSslConnectionType* pSslConnection,
    const char* pInHostName,
    uint32_t hostNameLength);
nnResult nnsslConnectionSetVerifyOption(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionVerifyOption optionValue);
nnResult nnsslConnectionSetServerCertBuffer(
    nnsslSslConnectionType* pSslConnection,
    char* pInBuffer,
    uint32_t bufferLength);
nnResult nnsslConnectionSetIoMode(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionIoMode mode);
nnResult nnsslConnectionSetSessionCacheMode(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionSessionCacheMode mode);

nnResult nnsslConnectionGetSocketDescriptor(
    nnsslSslConnectionType* pSslConnection,
    int* pOutSocketDescriptor);
nnResult nnsslConnectionGetHostName(
    nnsslSslConnectionType* pSslConnection,
    char* pOutHostName,
    uint32_t* pOutHostNameLength,
    uint32_t hostNameBufferLength);
nnResult nnsslConnectionGetVerifyOption(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionVerifyOption* pOutOptionValue);
nnResult nnsslConnectionGetIoMode(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionIoMode* pOutMode);

nnResult nnsslConnectionDoHandshake(
    nnsslSslConnectionType* pSslConnection);
nnResult nnsslConnectionDoHandshakeWithCertBuffer(
    nnsslSslConnectionType* pSslConnection,
    uint32_t* pOutServerCertSize,
    uint32_t* pOutServerCertCount);
nnResult nnsslConnectionDoHandshakeWithBuffer(
    nnsslSslConnectionType* pSslConnection,
    uint32_t* pOutCertSize,
    uint32_t* pOutServerCertCount,
    char* pOutBuffer,
    uint32_t bufferLen);
nnResult nnsslConnectionGetServerCertDetail(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionServerCertDetailType* pOutCertDetail,
    const char* pInBuffer,
    uint32_t index);

nnResult nnsslConnectionRead(
    nnsslSslConnectionType* pSslConnection,
    char* pOutBuffer,
    int* pOutReadSizeCourier,
    uint32_t bufferLength);
nnResult nnsslConnectionWrite(
    nnsslSslConnectionType* pSslConnection,
    const char* pInBuffer,
    int* pOutWrittenSizeCourier,
    uint32_t bufferLength);
nnResult nnsslConnectionPending(
    nnsslSslConnectionType* pSslConnection,
    int* pOutPendingSizeCourier);
nnResult nnsslConnectionPeek(
    nnsslSslConnectionType* pSslConnection,
    char* pOutBuffer,
    int* pOutReadSizeCourier,
    uint32_t bufferLength);
nnResult nnsslConnectionPoll(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionPollEvent* pOutEvent,
    nnsslConnectionPollEvent* pInEvent,
    uint32_t msecTimeout);

nnResult nnsslConnectionGetVerifyCertError(
    nnsslSslConnectionType* pSslConnection,
    nnResult* pOutVerifyError);
nnResult nnsslConnectionGetContextId(
    nnsslSslConnectionType* pSslConnection,
    nnsslSslContextId* pOutContextId);
nnResult nnsslConnectionGetConnectionId(
    nnsslSslConnectionType* pSslConnection,
    nnsslSslConnectionId* pOutConnectionId);

nnResult nnsslConnectionSetOption(
    nnsslSslConnectionType* pSslConnection,
    nnsslConnectionOptionType optionType,
    bool enable);
nnResult nnsslConnectionGetOption(
    nnsslSslConnectionType* pSslConnection,
    bool* pOutIsEnabled,
    nnsslConnectionOptionType optionType);

#ifdef __cplusplus
}
#endif // __cplusplus
