﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file
 * @brief PCIe Driver Public Data Types
 *
 * @details This file defines the public PCIe driver data types
 *          (C language).
 */

/**
 * @brief Standard zero based bus identifier with valid range
 *        0-255 inclusive
 */
typedef uint32_t    nnpcieBusNumber;

/**
 * @brief Standard zero based device identifier with valid range
 *        0-31 inclusive
 */
typedef uint8_t     nnpcieDeviceNumber;

/**
 * @brief Standard zero based function identifier with valid
 *        range 0-7 inclusive
 */
typedef uint8_t     nnpcieFunctionNumber;

/**
 * @brief Base class code.
 */
typedef uint8_t     nnpcieBaseClassCode;

/**
 * @brief Sub class code.
 */
typedef uint8_t     nnpcieSubClassCode;

/**
 * @brief Complete class code consisting of base, subclass and
 *        programming interface code.
 *          Bits  7-0  : programming interface code;
 *          Bits 15-8  : SubClassCode code;
 *          Bits 23-16 : BaseClassCode code.
 */
typedef uint32_t    nnpcieClassCode;

/**
 * @brief Handle representing the runtime relationship
 *        established between the user's class driver and the
 *        core PCI stack. This handle tells the core PCI stack
 *        which user is making the API call.
 */
typedef uintptr_t   nnpcieClassDriverHandle;

static const nnpcieClassDriverHandle nnpcieInvalidClassDriverHandle = 0;
/**
 * @brief Handle representing the specific PCI device function
 *        which the user's class driver has acquired (taken
 *        ownership). This handle tells the core PCI stack to
 *        which function the invoked API applies.
 */
typedef uint32_t    nnpcieFunctionHandle;

static const nnpcieFunctionHandle nnpcieInvalidFunctionHandle = 0;

/**
 * @brief Physical address of data stored in RAM, accessible via
 *        PCI bus master. This is not a virtual address, and
 *        thus cannot be dereferenced by host CPU.
 */
typedef uint64_t    nnpcieBusAddress;

/**
 * @brief Logical OR combination of any number of PCI region
 *        flags, as defined by the enumeration PciRegionFlag.
 *        Region flags describe behaviors of mapped PCI device
 *        function address space.
 */
typedef uint32_t    nnpcieRegionFlagMask;

/**
 * @brief Logical OR combination of any number of PCI function
 *        select flags, as defined by the enumeration
 *        FunctionSelect. Class driver software
 *        uses these flags to indicate which fields within
 *        ClassDriverConfig are valid, refining the selection of
 *        endpoint devices which will be presented for probing.
 */
typedef uint32_t    nnpcieFunctionSelectMask;

/**
 * @brief Logical OR combination of reset flags, as defined by the
 *        enumeration FunctionResetOption
 */
typedef uint32_t    nnpcieFunctionResetOptionMask;

/**
 * @brief Speed of PCI bus to which a PCI device function is
 *        connected.
 */
typedef enum
{
    nnpcieBusSpeed_Invalid,
    nnpcieBusSpeed_Auto,
    nnpcieBusSpeed_EGen1,
    nnpcieBusSpeed_EGen2,
    nnpcieBusSpeed_EGen3
}nnpcieBusSpeed;

/**
 * @brief Power management state of a PCI device function.
 */
typedef enum
{
    nnpciePmState_D0 = 0,
    nnpciePmState_D1,
    nnpciePmState_D2,
    nnpciePmState_D3Hot,
    nnpciePmState_D3Off,
    nnpciePmState_Invalid,
}nnpciePmState;

/**
 * @brief Software interrupt service request mechanism used by
 *        a PCI device function.
 */
typedef enum
{
    nnpcieIrqType_Invalid = 0,
    nnpcieIrqType_IntX,
    nnpcieIrqType_Msi,
    nnpcieIrqType_MsiX, /* not supported yet */
}nnpcieIrqType;



/**
 * @brief   Specifies which fields of ClassDriverConfig are valid,
 *          to be considered as search criterion.
 */
typedef enum
{
    nnpcieFunctionSelect_BaseClassCode = (1 << 0),
    nnpcieFunctionSelect_SubClassCode  = (1 << 1),
    nnpcieFunctionSelect_VendorId      = (1 << 2),
    nnpcieFunctionSelect_DeviceId      = (1 << 3),
}nnpcieFunctionSelect;

/**
 * @brief   Specifies a set of options which may be passed to ResetFunction().
 */
typedef enum
{
    nnpcieFunctionResetOption_None                  = 0,
    nnpcieFunctionResetOption_DoExternalReset       = (1 << 0),
    nnpcieFunctionResetOption_DoFunctionLevelReset  = (1 << 1),
    nnpcieFunctionResetOption_DoNotReEnumerate      = (1 << 2),
}nnpcieFunctionResetOption;

/**
 * @brief Configuration specified by user's class driver upon
 *        registration.
 */
typedef struct
{
    uint64_t                  classDriverContext;
    nnpcieFunctionSelectMask  selectMask;
    uint8_t                   baseClassCode;
    uint8_t                   subClassCode;
    uint16_t                  vendorId;
    uint16_t                  deviceId;
}nnpcieClassDriverConfig;

/**
 * @brief Profile of a specific device function
 */
typedef struct
{
    nnpcieFunctionHandle functionHandle;
    nnpcieBusNumber      busNum;
    nnpcieDeviceNumber   deviceNum;
    nnpcieFunctionNumber funcNum;
    uint16_t             vendorId;
    uint16_t             deviceId;
    uint8_t              headerType;
    nnpcieClassCode      classCode;
    uint8_t              revision;
    nnpcieBusSpeed       speed;
    int32_t              maxNumMsiVectors;
    nnpciePmState        pmState;
    bool                 isAcquired;
}nnpcieFunctionState;

/**
 * @brief Describe behaviors of mapped PCI device function
 *        address space.
 */
typedef enum
{
    nnpcieResourceFlag_Io         = 0x0100,
    nnpcieResourceFlag_Mem        = 0x0200,
    nnpcieResourceFlag_Mem64      = 0x0400,
    nnpcieResourceFlag_Prefetch   = 0x0800,
    nnpcieResourceFlag_RawBarMask = 0x00ff,
}nnpcieRegionFlag;

/**
 * @brief Information about a PCI device function base address
 *        register (BAR)
 */
typedef struct
{
    size_t               size;
    nnpcieRegionFlagMask flags;
}nnpcieBarProfile;

/**
 * @brief Bus access size
 */
typedef enum
{
    nnpcieBusAccessWidth_8Bit,
    nnpcieBusAccessWidth_16Bit,
    nnpcieBusAccessWidth_32Bit,
    nnpcieBusAccessWidth_64Bit,
}nnpcieBusAccessWidth;

/**
 * @brief Describes direct memory access (DMA) flow of data
 *        between PCI endpoint function and host RAM.
 */
typedef enum
{
    nnpcieDmaDirection_BiDirectional,
    nnpcieDmaDirection_FromDevice,
    nnpcieDmaDirection_ToDevice
}nnpcieDmaDirection;

/**
 * @brief Total maximum number of device functions which will be recognized
 */
static const int32_t nnpcieMaxEnumeratedDeviceFunctions = 4;

/**
 * @brief Total maximum number of device functions which may be acquired by a single class
 *        driver instantiation.
 */
static const int32_t nnpcieMaxDeviceFunctionsPerClient = 1;

/**
 * @brief Calls to nnpcieMapDma() must ensure:
 *       - Specified base 'base' is aligned to
 *         MinimumDmaAddressAlignment.
 *       - Specified 'size' is a multiple of
 *         MinimumDmaAddressAlignment.
 */
static const size_t nnpcieMinimumDmaAddressAlignment = 4096;

/**
 * @brief Describes the maximum number of simultaneous users allowed by PCIe driver
 */
static const int32_t nnpcieMaxClientCount = 3;

/**
 * @brief Maximum number of IRQs which can be acquired with a single call to nnpciecquireIrq()
 */
static const int32_t nnpcieMaxIrqVectorsPerDeviceFunction = 4;


