﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       デバイスの列挙などに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_BitTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace xcd {

/**
 * @brief       レール部に起きたイベントの種類です
 */
enum RailUpdateEventType : uint8_t
{
    RailUpdateEventType_None = 0,       //!< 何もイベントが発生していません
    RailUpdateEventType_LeftAttached,   //!< 左レールに Joy-Con (L) が接続されました
    RailUpdateEventType_LeftDetached,   //!< 左レールに Joy-Con (L) が切断されました
    RailUpdateEventType_RightAttached,  //!< 右レールに Joy-Con (R) が接続されました
    RailUpdateEventType_RightDetached,  //!< 右レールに Joy-Con (R) が切断されました
};

/**
 * @brief       レール部に起きたイベントの内容です
 */
struct RailUpdateEventInfo
{
    RailUpdateEventType eventType;      //!< 発生したイベントの種類です
    bool colorAvailable;                //!< 有効な色情報が存在するかどうか
    nn::util::Color4u8Type mainColor;   //!< メインの色です
    nn::util::Color4u8Type subColor;    //!< サブの色です
};

/**
 * @brief       レールの識別子です
 */
enum RailType : uint8_t
{
    RailType_Left = 0,       //!< 左レールです
    RailType_Right,          //!< 右レールです
};

/**
 * @brief      Joy-Con のレール部の Uart 通信を有効にする
 */
void EnableNwcp() NN_NOEXCEPT;

/**
 * @brief      Joy-Con のレール部の Uart 通信を無効にする
 */
void DisableNwcp() NN_NOEXCEPT;


/**
 * @brief       レール部の状態に変化があった場合に通知するイベントを設定します
 *
 * @param[in]   pSystemEvent                  バインドするシステムイベント
 *
 * @pre         - pSystemEvent != nullptr
 */
void SetRailUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

/**
 * @brief      レール部に起きた変化の内容を取得します
 */
RailUpdateEventInfo GetRailUpdateEventInfo() NN_NOEXCEPT;

/**
 * @brief      レールに何かデバイスが装着されているかどうかを取得します。
 */
bool IsDeviceAttachedOnRail(RailType type) NN_NOEXCEPT;

}} // namespace nn::xcd
